/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.api.multiplayer;

import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.mojang.logging.LogUtils;
import io.github.axolotlclient.api.types.PkSystem;
import io.github.axolotlclient.api.types.Status;
import io.github.axolotlclient.api.types.User;
import io.github.axolotlclient.modules.auth.Auth;
import lombok.Getter;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1011;
import net.minecraft.class_10799;
import net.minecraft.class_11908;
import net.minecraft.class_11909;
import net.minecraft.class_124;
import net.minecraft.class_140;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_2564;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_4280;
import net.minecraft.class_5244;
import net.minecraft.class_5250;
import net.minecraft.class_5481;
import net.minecraft.class_642;
import net.minecraft.class_7413;
import net.minecraft.class_7532;
import net.minecraft.class_8573;
import net.minecraft.network.chat.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

public class FriendsMultiplayerSelectionList extends class_4280<FriendsMultiplayerSelectionList.Entry> {
	static final class_2960 INCOMPATIBLE_SPRITE = class_2960.method_60656("server_list/incompatible");
	static final class_2960 UNREACHABLE_SPRITE = class_2960.method_60656("server_list/unreachable");
	static final class_2960 PING_1_SPRITE = class_2960.method_60656("server_list/ping_1");
	static final class_2960 PING_2_SPRITE = class_2960.method_60656("server_list/ping_2");
	static final class_2960 PING_3_SPRITE = class_2960.method_60656("server_list/ping_3");
	static final class_2960 PING_4_SPRITE = class_2960.method_60656("server_list/ping_4");
	static final class_2960 PING_5_SPRITE = class_2960.method_60656("server_list/ping_5");
	static final class_2960 PINGING_1_SPRITE = class_2960.method_60656("server_list/pinging_1");
	static final class_2960 PINGING_2_SPRITE = class_2960.method_60656("server_list/pinging_2");
	static final class_2960 PINGING_3_SPRITE = class_2960.method_60656("server_list/pinging_3");
	static final class_2960 PINGING_4_SPRITE = class_2960.method_60656("server_list/pinging_4");
	static final class_2960 PINGING_5_SPRITE = class_2960.method_60656("server_list/pinging_5");
	static final class_2960 JOIN_HIGHLIGHTED_SPRITE = class_2960.method_60656("server_list/join_highlighted");
	static final class_2960 JOIN_SPRITE = class_2960.method_60656("server_list/join");
	static final Logger LOGGER = LogUtils.getLogger();
	static final ThreadPoolExecutor THREAD_POOL = new ScheduledThreadPoolExecutor(
		5,
		new ThreadFactoryBuilder()
			.setNameFormat("Friends Server Pinger #%d")
			.setDaemon(true)
			.setUncaughtExceptionHandler(new class_140(LOGGER))
			.build()
	);
	static final class_2561 CANT_RESOLVE_TEXT = class_2561.method_43471("multiplayer.status.cannot_resolve").method_54663(-65536);
	static final class_2561 CANT_CONNECT_TEXT = class_2561.method_43471("multiplayer.status.cannot_connect").method_54663(-65536);
	static final class_2561 INCOMPATIBLE_STATUS = class_2561.method_43471("multiplayer.status.incompatible");
	static final class_2561 NO_CONNECTION_STATUS = class_2561.method_43471("multiplayer.status.no_connection");
	static final class_2561 PINGING_STATUS = class_2561.method_43471("multiplayer.status.pinging");
	static final class_2561 ONLINE_STATUS = class_2561.method_43471("multiplayer.status.online");
	static final class_2561 NOT_PUBLISHED_STATUS = class_2561.method_43471("api.worldhost.joinability.not_published");
	private final FriendsMultiplayerScreen screen;
	private final List<Entry> friendEntries = new ArrayList<>();
	private final LoadingHeader loadingHeader = new LoadingHeader();

	public FriendsMultiplayerSelectionList(FriendsMultiplayerScreen screen, class_310 minecraft, int width, int height, int y, int itemHeight) {
		super(minecraft, width, height, y, itemHeight);
		this.screen = screen;
		method_25321(loadingHeader);
	}

	private void refreshEntries() {
		this.method_25339();
		this.friendEntries.forEach(this::method_25321);
	}

	public void setSelected(@Nullable FriendsMultiplayerSelectionList.Entry entry) {
		super.method_25313(entry);
		this.screen.onSelectedChange();
	}

	@Override
	public boolean method_25404(class_11908 event) {
		FriendsMultiplayerSelectionList.Entry entry = this.method_25334();
		return entry != null && entry.method_25404(event) || super.method_25404(event);
	}

	public void updateList(List<User> friends) {
		this.friendEntries.clear();

		for (User friend : friends) {
			if (friend.getStatus().isOnline()) {
				this.friendEntries.add(createEntry(friend));
			}
		}
		this.refreshEntries();
	}

	private Entry createEntry(User friend) {
		if (friend.getStatus().getActivity() != null && friend.getStatus().getActivity().hasMetadata()) {
			if (friend.getStatus().getActivity().hasMetadata(Status.Activity.ExternalServerMetadata.ID)) {
				return externalServerEntry(this.screen, friend);
			} else {
				return e4mcServerFriendEntry(this.screen, friend);
			}
		}
		return new StatusFriendEntry(friend);
	}

	public void updateEntry(User user) {
		this.friendEntries.stream().filter(e1 -> {
			if (e1 instanceof StatusFriendEntry statusFriendEntry) {
				return statusFriendEntry.getUser().equals(user);
			} else if (e1 instanceof ServerEntry serverEntry) {
				return serverEntry.getUser().equals(user);
			}
			return false;
		}).findFirst().ifPresent(e -> {
			this.friendEntries.set(friendEntries.indexOf(e), createEntry(user));
			refreshEntries();
		});
	}

	@Override
	public int method_25322() {
		return 305;
	}

	@Environment(EnvType.CLIENT)
	public abstract static class Entry extends class_4280.class_4281<FriendsMultiplayerSelectionList.Entry> implements AutoCloseable {
		public void close() {
		}

		public boolean canJoin() {
			return false;
		}

		public class_642 getServerData() {
			return null;
		}
	}

	@Getter
	public class StatusFriendEntry extends Entry {

		protected final User user;

		protected StatusFriendEntry(final User friend) {
			this.user = friend;
		}

		@Override
		public class_2561 method_37006() {
			return class_2561.method_43470(user.getName());
		}

		@Override
		public void method_25343(class_332 graphics, int mouseX, int mouseY, boolean hovering, float partialTick) {
			if (user.isSystem()) {
				class_5250 fronters = class_2561.method_43470(
					user.getSystem().getFronters().stream().map(PkSystem.Member::getDisplayName)
						.collect(Collectors.joining("/")));
				class_2561 tag = class_2561.method_43470("(" + user.getSystem().getName() + "/" + user.getName() + ")")
					.method_10862(class_2583.field_24360.method_10978(true).method_10977(class_124.field_1080));
				graphics.method_27535(field_22740.field_1772, fronters.method_10852(tag), method_73380() + 3, method_73382() + 1, -1);
			} else {
				graphics.method_25303(field_22740.field_1772, user.getName(), method_73380() + 3 + 32, method_73382() + 1, -1);
			}

			if (user.getStatus().isOnline() && user.getStatus().getActivity() != null) {
				graphics.method_25303(field_22740.field_1772, user.getStatus().getTitle(), method_73380() + 3 + 32, method_73382() + 12, 0xFF808080);
				graphics.method_25303(field_22740.field_1772, user.getStatus().getDescription(), method_73380() + 3 + 40, method_73382() + 23, 0xFF808080);
			} else if (user.getStatus().getLastOnline() != null) {
				graphics.method_25303(field_22740.field_1772, user.getStatus().getLastOnline(), method_73380() + 3 + 32, method_73382() + 12, 0xFF808080);
			}

			class_2960 texture = Auth.getInstance().getSkinTexture(user);
			class_7532.method_44445(graphics, texture, method_73380(), method_73382(), 32, true, false, -1);
		}
	}

	protected class ServerEntry extends Entry {
		private static final int ICON_WIDTH = 32;
		private static final int ICON_HEIGHT = 32;
		private static final int SPACING = 5;
		private static final int STATUS_ICON_WIDTH = 10;
		private static final int STATUS_ICON_HEIGHT = 8;
		private final FriendsMultiplayerScreen screen;
		private final class_310 minecraft;
		@Getter
		protected final class_642 serverData;
		private final class_8573 icon;
		private byte @Nullable [] lastIconBytes;
		@Nullable
		private List<class_2561> onlinePlayersTooltip;
		@Nullable
		private class_2960 statusIcon;
		@Nullable
		private class_2561 statusIconTooltip;
		@Getter
		protected final User user;

		protected ServerEntry(FriendsMultiplayerScreen screen, class_642 serverData, User user) {
			this.screen = screen;
			this.minecraft = class_310.method_1551();
			this.serverData = serverData;
			this.icon = class_8573.method_52202(minecraft.method_1531(), serverData.field_3761 != null ? serverData.field_3761 : user.getUuid() + "_" + serverData.field_3752);
			this.user = user;
		}


		protected void refreshStatus() {
			this.onlinePlayersTooltip = null;
			if (!isPublished()) {
				this.serverData.method_55824(class_642.class_9083.field_47882);
			}
			switch (this.serverData.method_55825()) {
				case field_47880:
				case field_47881:
					this.statusIcon = FriendsMultiplayerSelectionList.PING_1_SPRITE;
					this.statusIconTooltip = FriendsMultiplayerSelectionList.PINGING_STATUS;
					break;
				case field_47883:
					this.statusIcon = FriendsMultiplayerSelectionList.INCOMPATIBLE_SPRITE;
					this.onlinePlayersTooltip = this.serverData.field_3762;
					this.statusIconTooltip = FriendsMultiplayerSelectionList.INCOMPATIBLE_STATUS;
					break;
				case field_47882:
					this.statusIcon = FriendsMultiplayerSelectionList.UNREACHABLE_SPRITE;
					if (!isPublished()) {
						break;
					}
					this.statusIconTooltip = FriendsMultiplayerSelectionList.NO_CONNECTION_STATUS;
					break;
				case field_47884:
					if (this.serverData.field_3758 < 150L) {
						this.statusIcon = FriendsMultiplayerSelectionList.PING_5_SPRITE;
					} else if (this.serverData.field_3758 < 300L) {
						this.statusIcon = FriendsMultiplayerSelectionList.PING_4_SPRITE;
					} else if (this.serverData.field_3758 < 600L) {
						this.statusIcon = FriendsMultiplayerSelectionList.PING_3_SPRITE;
					} else if (this.serverData.field_3758 < 1000L) {
						this.statusIcon = FriendsMultiplayerSelectionList.PING_2_SPRITE;
					} else {
						this.statusIcon = FriendsMultiplayerSelectionList.PING_1_SPRITE;
					}

					this.statusIconTooltip = class_2561.method_43469("multiplayer.status.ping", this.serverData.field_3758);
					this.onlinePlayersTooltip = this.serverData.field_3762;
			}
		}

		@Override
		public void method_25343(class_332 guiGraphics, int mouseX, int mouseY, boolean hovering, float partialTick) {
			if (this.serverData.method_55825() == class_642.class_9083.field_47880) {
				this.serverData.method_55824(class_642.class_9083.field_47881);
				this.serverData.field_3757 = class_5244.field_39003;
				this.serverData.field_3753 = class_5244.field_39003;
				FriendsMultiplayerSelectionList.THREAD_POOL
					.submit(
						() -> {
							try {
								this.screen
									.getPinger()
									.pingServer(
										this.serverData,
										() -> {
										},
										() -> {
											this.serverData
												.setState(
													this.serverData.protocol == SharedConstants.getCurrentVersion().protocolVersion() ? ServerData.State.SUCCESSFUL : ServerData.State.INCOMPATIBLE
												);
											this.minecraft.execute(this::refreshStatus);
										}
									);
							} catch (UnknownHostException var2) {
								this.serverData.method_55824(class_642.class_9083.field_47882);
								this.serverData.field_3757 = FriendsMultiplayerSelectionList.CANT_RESOLVE_TEXT;
								this.minecraft.execute(this::refreshStatus);
							} catch (Exception var3) {
								this.serverData.method_55824(class_642.class_9083.field_47882);
								this.serverData.field_3757 = FriendsMultiplayerSelectionList.CANT_CONNECT_TEXT;
								this.minecraft.execute(this::refreshStatus);
							}
						}
					);
			}

			guiGraphics.method_25303(this.minecraft.field_1772, this.serverData.field_3752, method_73380() + ICON_WIDTH + 3, method_73382() + 1, -1);
			List<class_5481> list = this.minecraft.field_1772.method_1728(this.serverData.field_3757, field_22758 - ICON_WIDTH - 2);

			for (int i = 0; i < Math.min(list.size(), 2); i++) {
				guiGraphics.method_35720(this.minecraft.field_1772, list.get(i), method_73380() + ICON_WIDTH + 3, method_73382() + 12 + 9 * i, -8355712);
			}

			guiGraphics.method_25290(class_10799.field_56883, this.icon.method_52201(), method_73380(), method_73382(), 0.0F, 0.0F, ICON_WIDTH, ICON_HEIGHT, ICON_WIDTH, ICON_HEIGHT);
			class_2960 texture = Auth.getInstance().getSkinTexture(user);
			class_7532.method_44445(guiGraphics, texture, method_73380() + ICON_WIDTH - 10, method_73382() + ICON_HEIGHT - 10, 10, true, false, -1);
			if (this.serverData.method_55825() == class_642.class_9083.field_47881) {
				int i = (int) (class_156.method_658() / 100L + FriendsMultiplayerSelectionList.this.method_25396().indexOf(this) * 2 & 7L);
				if (i > 4) {
					i = 8 - i;
				}
				this.statusIcon = switch (i) {
					case 1 -> FriendsMultiplayerSelectionList.PINGING_2_SPRITE;
					case 2 -> FriendsMultiplayerSelectionList.PINGING_3_SPRITE;
					case 3 -> FriendsMultiplayerSelectionList.PINGING_4_SPRITE;
					case 4 -> FriendsMultiplayerSelectionList.PINGING_5_SPRITE;
					default -> FriendsMultiplayerSelectionList.PINGING_1_SPRITE;
				};
			}

			int i = method_73380() + field_22758 - STATUS_ICON_WIDTH - SPACING;
			if (this.statusIcon != null) {
				guiGraphics.method_52706(class_10799.field_56883, this.statusIcon, i, method_73382(), STATUS_ICON_WIDTH, STATUS_ICON_HEIGHT);
			}

			byte[] bs = this.serverData.method_49306();
			if (!Arrays.equals(bs, this.lastIconBytes)) {
				if (this.uploadIcon(bs)) {
					this.lastIconBytes = bs;
				} else {
					this.serverData.method_49305(null);
				}
			}

			class_2561 component;
			if (!isPublished()) {
				component = NOT_PUBLISHED_STATUS;
			} else {
				if (this.serverData.method_55825() == class_642.class_9083.field_47883) {
					component = this.serverData.field_3760.method_27661().method_27692(class_124.field_1061);
				} else {
					component = this.serverData.field_3753;
				}
			}
			int j = this.minecraft.field_1772.method_27525(component);
			int k = i - j - SPACING;
			guiGraphics.method_27535(this.minecraft.field_1772, component, k, method_73382() + 1, -8355712);
			if (this.statusIconTooltip != null && mouseX >= i && mouseX <= i + STATUS_ICON_WIDTH && mouseY >= method_73382() && mouseY <= method_73382() + STATUS_ICON_HEIGHT) {
				guiGraphics.method_71276(this.statusIconTooltip, mouseX, mouseY);
			} else if (this.onlinePlayersTooltip != null && mouseX >= k && mouseX <= k + j && mouseY >= method_73382() && mouseY <= method_73382() - 1 + 9) {
				guiGraphics.method_71274(Lists.transform(this.onlinePlayersTooltip, class_2561::method_30937), mouseX, mouseY);
			}

			if (this.minecraft.field_1690.method_42446().method_41753() || hovering) {
				int l = mouseX - method_73380();
				if (this.canJoin()) {
					guiGraphics.method_25294(method_73380(), method_73382(), method_73380() + ICON_WIDTH, method_73382() + ICON_HEIGHT, -1601138544);
					if (l < ICON_WIDTH && l > ICON_WIDTH / 2) {
						guiGraphics.method_52706(class_10799.field_56883, FriendsMultiplayerSelectionList.JOIN_HIGHLIGHTED_SPRITE, method_73380(), method_73382(), ICON_WIDTH, ICON_HEIGHT);
					} else {
						guiGraphics.method_52706(class_10799.field_56883, FriendsMultiplayerSelectionList.JOIN_SPRITE, method_73380(), method_73382(), ICON_WIDTH, ICON_HEIGHT);
					}
				}
			}
		}

		protected boolean isPublished() {
			return true;
		}

		@Override
		public boolean canJoin() {
			return serverData.method_55825() == class_642.class_9083.field_47884 && isPublished();
		}

		private boolean uploadIcon(byte @Nullable [] iconBytes) {
			if (iconBytes == null) {
				this.icon.method_52198();
			} else {
				try {
					this.icon.method_52199(class_1011.method_49277(iconBytes));
				} catch (Throwable var3) {
					FriendsMultiplayerSelectionList.LOGGER.error("Invalid icon for server {} ({})", this.serverData.field_3752, this.serverData.field_3761, var3);
					return false;
				}
			}

			return true;
		}

		@Override
		public boolean method_25402(class_11909 event, boolean doubleClick) {
			double d = event.comp_4798() - FriendsMultiplayerSelectionList.this.method_25342();
			if (d <= 32.0) {
				if (d < 32.0 && d > 16.0 && this.canJoin()) {
					this.screen.setSelected(this);
					this.screen.joinSelectedServer();
					return true;
				}
			}

			this.screen.setSelected(this);
			if (doubleClick && canJoin()) {
				this.screen.joinSelectedServer();
			}

			return super.method_25402(event, doubleClick);
		}

		@Override
		public @NotNull class_2561 method_37006() {
			class_5250 mutableComponent = class_2561.method_43473();
			mutableComponent.method_10852(class_2561.method_43469("narrator.select", this.serverData.field_3752));
			mutableComponent.method_10852(class_5244.field_33850);
			switch (this.serverData.method_55825()) {
				case field_47881:
					mutableComponent.method_10852(FriendsMultiplayerSelectionList.PINGING_STATUS);
					break;
				case field_47883:
					mutableComponent.method_10852(FriendsMultiplayerSelectionList.INCOMPATIBLE_STATUS);
					mutableComponent.method_10852(class_5244.field_33850);
					mutableComponent.method_10852(class_2561.method_43469("multiplayer.status.version.narration", this.serverData.field_3760));
					mutableComponent.method_10852(class_5244.field_33850);
					mutableComponent.method_10852(class_2561.method_43469("multiplayer.status.motd.narration", this.serverData.field_3757));
					break;
				case field_47882:
					mutableComponent.method_10852(FriendsMultiplayerSelectionList.NO_CONNECTION_STATUS);
					break;
				default:
					mutableComponent.method_10852(FriendsMultiplayerSelectionList.ONLINE_STATUS);
					mutableComponent.method_10852(class_5244.field_33850);
					mutableComponent.method_10852(class_2561.method_43469("multiplayer.status.ping.narration", this.serverData.field_3758));
					mutableComponent.method_10852(class_5244.field_33850);
					mutableComponent.method_10852(class_2561.method_43469("multiplayer.status.motd.narration", this.serverData.field_3757));
					if (this.serverData.field_41861 != null) {
						mutableComponent.method_10852(class_5244.field_33850);
						mutableComponent.method_10852(
							class_2561.method_43469("multiplayer.status.player_count.narration", this.serverData.field_41861.comp_1280(), this.serverData.field_41861.comp_1279())
						);
						mutableComponent.method_10852(class_5244.field_33850);
						mutableComponent.method_10852(class_2564.method_37112(this.serverData.field_3762, class_2561.method_43470(", ")));
					}
			}

			return mutableComponent;
		}

		@Override
		public void close() {
			this.icon.close();
		}
	}

	private ExternalServerFriendEntry externalServerEntry(FriendsMultiplayerScreen screen, User friend) {
		Status.Activity.ExternalServerMetadata metadata = (Status.Activity.ExternalServerMetadata) friend.getStatus().getActivity().metadata().attributes();
		return new ExternalServerFriendEntry(screen, metadata, new class_642(metadata.serverName(), metadata.address(), class_642.class_8678.field_45611), friend);
	}

	public class ExternalServerFriendEntry extends ServerEntry {
		private final Status.Activity.ExternalServerMetadata statusDescription;

		private ExternalServerFriendEntry(FriendsMultiplayerScreen screen, Status.Activity.ExternalServerMetadata statusDescription, class_642 serverData, User friend) {
			super(screen, serverData, friend);
			this.statusDescription = statusDescription;
			refreshStatus();
		}

		@Override
		public boolean canJoin() {
			return statusDescription.address() != null;
		}

	}

	private E4mcServerFriendEntry e4mcServerFriendEntry(FriendsMultiplayerScreen screen, User friend) {
		var activity = friend.getStatus().getActivity();
		Status.Activity.E4mcMetadata metadata;
		if (activity.hasMetadata(Status.Activity.WorldHostMetadata.ID)) {
			metadata = ((Status.Activity.WorldHostMetadata) activity.metadata().attributes()).asE4mcMetadata();
		} else {
			metadata = (Status.Activity.E4mcMetadata) activity.metadata().attributes();
		}
		return new E4mcServerFriendEntry(screen, metadata, ServerInfoUtil.getServerData(friend.getName(), metadata), friend);
	}

	public class E4mcServerFriendEntry extends ServerEntry {

		private final Status.Activity.E4mcMetadata statusDescription;

		protected E4mcServerFriendEntry(FriendsMultiplayerScreen screen, Status.Activity.E4mcMetadata statusDescription, class_642 serverData, User friend) {
			super(screen, serverData, friend);
			this.statusDescription = statusDescription;
			refreshStatus();
		}

		@Override
		protected void refreshStatus() {
			super.refreshStatus();
			serverData.field_3757 = class_2561.method_30163(statusDescription.serverInfo().levelName());
		}

		@Override
		protected boolean isPublished() {
			return statusDescription.domain() != null;
		}
	}

	@Environment(EnvType.CLIENT)
	public static class LoadingHeader extends FriendsMultiplayerSelectionList.Entry {
		private final class_310 minecraft = class_310.method_1551();

		@Override
		public void method_25343(class_332 guiGraphics, int mouseX, int mouseY, boolean hovering, float partialTick) {
			int i = method_73385() - 9 / 2;
			String string = class_7413.method_43449(class_156.method_658());
			guiGraphics.method_25303(this.minecraft.field_1772, string, this.minecraft.field_1755.field_22789 / 2 - this.minecraft.field_1772.method_1727(string) / 2, i, -8355712);
		}

		@Override
		public @NotNull class_2561 method_37006() {
			return class_2561.method_43473();
		}
	}

}
