/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.config;

import java.util.ArrayList;
import java.util.List;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientCommon;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.options.OptionCategory;
import io.github.axolotlclient.AxolotlClientConfig.api.ui.ConfigUI;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.*;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.RecreatableScreen;
import io.github.axolotlclient.AxolotlClientConfigCommon;
import io.github.axolotlclient.config.screen.CreditsScreen;
import io.github.axolotlclient.config.screen.ProfilesScreen;
import io.github.axolotlclient.mixin.OverlayTextureAccessor;
import io.github.axolotlclient.util.keybinds.KeyBinds;
import io.github.axolotlclient.util.options.ForceableBooleanOption;
import io.github.axolotlclient.util.options.GenericOption;
import lombok.Getter;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_304;
import net.minecraft.class_310;
import net.minecraft.class_437;

public class AxolotlClientConfig extends AxolotlClientConfigCommon {

	public final BooleanOption showOwnNametag = new BooleanOption("showOwnNametag", false);
	public final BooleanOption useShadows = new BooleanOption("useShadows", false);
	public final BooleanOption nametagBackground = new BooleanOption("nametagBackground", true);

	public final BooleanOption showBadges = new BooleanOption("showBadges", true);
	public final BooleanOption customBadge = new BooleanOption("customBadge", false);
	public final StringOption badgeText = new StringOption("badgeText", "");

	public final ForceableBooleanOption timeChangerEnabled = new ForceableBooleanOption("enabled", false);
	public final IntegerOption customTime = new IntegerOption("time", 0, 0, 24000);
	public final BooleanOption dynamicFOV = new BooleanOption("dynamicFov", true);
	public final ForceableBooleanOption fullBright = new ForceableBooleanOption("fullBright", false);
	public final BooleanOption removeVignette = new BooleanOption("removeVignette", false);
	public final ForceableBooleanOption lowFire = new ForceableBooleanOption("lowFire", false);
	public final BooleanOption lowShield = new BooleanOption("lowShield", false);
	public final ColorOption hitColor = new ColorOption("hitColor", new Color(255, 0, 0, 77),
		value -> {
			class_1043 texture = ((OverlayTextureAccessor) class_310.method_1551().field_1773.method_22975()).axolotlclient$getTexture();
			class_1011 nativeImage = texture.method_4525();
			if (nativeImage != null) {
				int color = 255 - value.getAlpha();
				color = (color << 8) + value.getRed();
				color = (color << 8) + value.getGreen();
				color = (color << 8) + value.getBlue();

				for (int i = 0; i < 8; ++i) {
					for (int j = 0; j < 8; ++j) {
						nativeImage.method_61941(j, i, color);
					}
				}

				texture.method_4524();
			}
		});
	public final BooleanOption minimalViewBob = new BooleanOption("minimalViewBob", false);
	public final BooleanOption noHurtCam = new BooleanOption("noHurtCam", false);

	public final ColorOption loadingScreenColor = new ColorOption("loadingBgColor", new Color(239, 50, 61, 255));
	public final BooleanOption customWindowTitle = new BooleanOption("customWindowTitle", true);

	public final BooleanOption enableCustomOutlines = new BooleanOption("enabled", false);
	public final ColorOption outlineColor = new ColorOption("color", Color.parse("#DD000000"));

	public final BooleanOption noRain = new BooleanOption("noRain", false);

	public final GenericOption openCredits = new GenericOption("Credits", "Open Credits", () ->
		class_310.method_1551().method_1507(new CreditsScreen(class_310.method_1551().field_1755))
	);
	public final BooleanOption debugLogOutput = new BooleanOption("debugLogOutput", false);
	public final BooleanOption creditsBGM = new BooleanOption("creditsBGM", true);

	public final OptionCategory general = OptionCategory.create("general");
	public final OptionCategory nametagOptions = OptionCategory.create("nametagOptions");
	public final OptionCategory rendering = OptionCategory.create("rendering");
	public final OptionCategory outlines = OptionCategory.create("blockOutlines");
	public final OptionCategory timeChanger = OptionCategory.create("timeChanger");

	@Getter
	private final List<Option<?>> options = new ArrayList<>();

	public AxolotlClientConfig() {
		config.add(general);
		config.add(nametagOptions);
		config.add(rendering);

		rendering.add(outlines);

		nametagOptions.add(showOwnNametag);
		nametagOptions.add(useShadows);
		nametagOptions.add(nametagBackground);

		nametagOptions.add(showBadges);
		nametagOptions.add(customBadge);
		nametagOptions.add(badgeText);

		general.add(loadingScreenColor);
		general.add(customWindowTitle);
		general.add(openCredits);
		general.add(debugLogOutput);
		general.add(datetimeFormat);
		general.add(titleScreenOptionButtonMode);
		general.add(gameMenuScreenOptionButtonMode);

		ConfigUI.getInstance().runWhenLoaded(() -> {
			general.getOptions().removeIf(o -> "configStyle".equals(o.getName()));
			String[] themes = ConfigUI.getInstance().getStyleNames().stream().map(s -> "configStyle." + s)
				.filter(s -> AxolotlClientCommon.NVG_SUPPORTED || !s.startsWith("rounded"))
				.toArray(String[]::new);
			if (themes.length > 1) {
				StringArrayOption configStyle;
				general.add(configStyle = new StringArrayOption("configStyle", themes,
					"configStyle." + ConfigUI.getInstance().getCurrentStyle().getName(), s -> {
					ConfigUI.getInstance().setStyle(s.split("\\.")[1]);
					class_310.method_1551().method_63588(() -> {
						class_437 newScreen = RecreatableScreen.tryRecreate(class_310.method_1551().field_1755);
						class_310.method_1551().method_1507(newScreen);
					});
				}));
				AxolotlClient.getInstance().getConfigManager().load();
				ConfigUI.getInstance().setStyle(configStyle.get().split("\\.")[1]);
			} else {
				AxolotlClient.getInstance().getConfigManager().load();
			}
		});

		rendering.add(
			dynamicFOV,
			fullBright,
			removeVignette,
			lowFire,
			lowShield,
			hitColor,
			minimalViewBob,
			noHurtCam);

		timeChanger.add(timeChangerEnabled);
		timeChanger.add(customTime);
		rendering.add(timeChanger);

		outlines.add(enableCustomOutlines);
		outlines.add(outlineColor);

		rendering.add(noRain);

		hidden.add(creditsBGM, someNiceBackground);

		general.add(new GenericOption("profiles.title", "profiles.configure", () ->
			class_310.method_1551().method_1507(new ProfilesScreen(class_310.method_1551().field_1755))), false);

		var toggleFullbright = new class_304("toggle_fullbright", -1, KeyBinds.CATEGORY_AXOLOTLCLIENT);
		KeyBinds.getInstance().registerWithSimpleAction(toggleFullbright, fullBright::toggle);
	}
}
