/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.hud.vanilla;

import java.util.List;

import com.mojang.blaze3d.pipeline.BlendFunction;
import com.mojang.blaze3d.pipeline.RenderPipeline;
import com.mojang.blaze3d.platform.DestFactor;
import com.mojang.blaze3d.platform.SourceFactor;
import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.ColorOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.EnumOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.GraphicsOption;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.bridge.util.AxoIdentifier;
import io.github.axolotlclient.mixin.GameRendererAccessor;
import io.github.axolotlclient.modules.hud.gui.component.DynamicallyPositionable;
import io.github.axolotlclient.modules.hud.gui.entry.AbstractHudEntry;
import io.github.axolotlclient.modules.hud.gui.layout.AnchorPoint;
import io.github.axolotlclient.util.ClientColors;
import io.github.axolotlclient.util.Util;
import lombok.AllArgsConstructor;
import net.minecraft.class_10799;
import net.minecraft.class_1309;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_239;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3965;
import net.minecraft.class_4061;
import net.minecraft.class_4184;
import net.minecraft.class_4739;
import org.joml.Matrix4fStack;

/**
 * This implementation of Hud modules is based on KronHUD.
 * <a href="https://github.com/DarkKronicle/KronHUD">Github Link.</a>
 *
 * <p>License: GPL-3.0</p>
 */

public class CrosshairHud extends AbstractHudEntry implements DynamicallyPositionable {
	public static final class_2960 ID = class_2960.method_60655("kronhud", "crosshairhud");
	private static final class_2960 CROSSHAIR_TEXTURE = class_2960.method_60656("hud/crosshair");
	private static final class_2960 ATTACK_INDICATOR_FULL = class_2960.method_60656("hud/crosshair_attack_indicator_full");
	private static final class_2960 ATTACK_INDICATOR_BACKGROUND = class_2960.method_60656("hud/crosshair_attack_indicator_background");
	private static final class_2960 ATTACK_INDICATOR_PROGRESS = class_2960.method_60656("hud/crosshair_attack_indicator_progress");
	private final EnumOption<Crosshair> type = new EnumOption<>("crosshair_type", Crosshair.class, Crosshair.CROSS);
	private final BooleanOption showInF5 = new BooleanOption("showInF5", false);
	private final ColorOption defaultColor = new ColorOption("defaultcolor", ClientColors.WHITE);
	private final ColorOption entityColor = new ColorOption("entitycolor", ClientColors.SELECTOR_RED);
	private final ColorOption containerColor = new ColorOption("blockcolor", ClientColors.SELECTOR_BLUE);
	private final ColorOption attackIndicatorBackgroundColor = new ColorOption("attackindicatorbg",
		new Color(0xFF141414));
	private final ColorOption attackIndicatorForegroundColor = new ColorOption("attackindicatorfg", ClientColors.WHITE);
	private final BooleanOption applyBlend = new BooleanOption("applyBlend", true);
	private final BooleanOption overrideF3 = new BooleanOption("overrideF3", false);
	private final BooleanOption customAttackIndicator = new BooleanOption("crosshairhud.custom_attack_indicator", false);
	private final class_310 client = (class_310) super.client;

	private final GraphicsOption customTextureGraphics = new GraphicsOption("customTextureGraphics",
		new int[][]{
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
		});

	public CrosshairHud() {
		super(15, 15);
	}

	@Override
	public boolean movable() {
		return false;
	}

	@Override
	public AxoIdentifier getId() {
		return ID;
	}

	@Override
	public List<Option<?>> getConfigurationOptions() {
		List<Option<?>> options = super.getConfigurationOptions();
		options.add(hide);
		options.add(type);
		options.add(customTextureGraphics);
		options.add(showInF5);
		options.add(overrideF3);
		options.add(applyBlend);
		options.add(defaultColor);
		options.add(entityColor);
		options.add(containerColor);
		options.add(customAttackIndicator);
		options.add(attackIndicatorBackgroundColor);
		options.add(attackIndicatorForegroundColor);
		return options;
	}

	@Override
	public boolean overridesF3() {
		return overrideF3.get();
	}

	@Override
	public double getDefaultX() {
		return 0.5;
	}

	@Override
	public double getDefaultY() {
		return 0.5;
	}

	private static final RenderPipeline CROSSHAIR_NO_TEX = class_10799.method_67887(RenderPipeline.builder(class_10799.field_56863)
		.withLocation("pipeline/crosshair_no_tex")
		.withBlend(new BlendFunction(SourceFactor.ONE_MINUS_DST_COLOR, DestFactor.ONE_MINUS_SRC_COLOR, SourceFactor.ONE, DestFactor.ZERO))
		.build()
	);

	@Override
	public void render(AxoRenderContext graphics, float delta) {
	}

	public void renderCrosshair(class_332 graphics) {
		if (!client.field_1690.method_31044().method_31034() && !showInF5.get()) {
			return;
		}

		graphics.method_51448().pushMatrix();
		scale(graphics);

		int x = getPos().x();
		int y = getPos().y() + 1;
		Color color = getColor();
		class_4061 indicator = this.client.field_1690.method_42565().method_41753();

		// Need to not enable blend while the debug HUD is open because it does weird stuff. Why? no idea.
		boolean blend = color == defaultColor.get() && !type.get().equals(Crosshair.DIRECTION) && applyBlend.get()
			&& !client.field_1705.method_53531().method_53536();

		if (type.get().equals(Crosshair.DOT)) {
			fillRenderType(graphics, blend, x + (getWidth() / 2) - 2, y + (getHeight() / 2) - 2, 3, 3, color);
		} else if (type.get().equals(Crosshair.CROSS)) {
			fillRenderType(graphics, blend, x + (getWidth() / 2) - 6, y + (getHeight() / 2) - 1, 6, 1, color);
			fillRenderType(graphics, blend, x + (getWidth() / 2), y + (getHeight() / 2) - 1, 5, 1, color);
			fillRenderType(graphics, blend, x + (getWidth() / 2) - 1, y + (getHeight() / 2) - 6, 1, 5, color);
			fillRenderType(graphics, blend, x + (getWidth() / 2) - 1, y + (getHeight() / 2), 1, 5, color);
		} else if (type.get().equals(Crosshair.DIRECTION)) {
			class_4184 camera = this.client.field_1773.method_19418();
			Matrix4fStack matrixStack = RenderSystem.getModelViewStack();
			matrixStack.pushMatrix();
			matrixStack.translate(client.method_22683().method_4486() / 2F, client.method_22683().method_4502() / 2F,
				0);
			matrixStack.rotateX(-camera.method_19329() * 0.017453292F);
			matrixStack.rotateY(camera.method_19330() * 0.017453292F);
			matrixStack.scale(-getScale(), -getScale(), -getScale());
			client.field_1705.method_53531().method_67545(((GameRendererAccessor) client.field_1773).getCamera());
			matrixStack.popMatrix();
		} else if (type.get().equals(Crosshair.TEXTURE) || type.get().equals(Crosshair.CUSTOM)) {
			if (type.get().equals(Crosshair.TEXTURE)) {
				// Draw crosshair
				graphics.method_52707(renderType(blend), CROSSHAIR_TEXTURE,
					(int) (((graphics.method_51421() / getScale()) - 15) / 2),
					(int) (((graphics.method_51443() / getScale()) - 15) / 2), 15, 15, color.toInt());
			} else {
				// Draw crosshair
				graphics.method_25291(renderType(blend), Util.getTexture(customTextureGraphics), (int) (((graphics.method_51421() / getScale()) - 15) / 2),
					(int) (((graphics.method_51443() / getScale()) - 15) / 2), 0, 0, 15, 15, 15, 15, color.toInt());
			}

			// Draw attack indicator
			if (!customAttackIndicator.get() && indicator == class_4061.field_18152) {
				//noinspection DataFlowIssue
				float progress = this.client.field_1724.method_7261(0.0F);

				// Whether a cross should be displayed under the indicator
				boolean targetingEntity = false;
				if (this.client.field_1692 != null && this.client.field_1692 instanceof class_1309
					&& progress >= 1.0F) {
					targetingEntity = this.client.field_1724.method_7279() > 5.0F;
					targetingEntity &= this.client.field_1692.method_5805();
				}

				x = (int) ((graphics.method_51421() / getScale()) / 2 - 8);
				y = (int) ((graphics.method_51443() / getScale()) / 2 - 7 + 16);

				if (targetingEntity) {
					graphics.method_52706(renderType(blend), ATTACK_INDICATOR_FULL, x, y, 16, 16);
				} else if (progress < 1.0F) {
					int k = (int) (progress * 17.0F);
					graphics.method_52706(renderType(blend), ATTACK_INDICATOR_BACKGROUND, x, y, 16, 4);
					graphics.method_70846(renderType(blend), ATTACK_INDICATOR_PROGRESS, 16, 4, 0, 0, x, y, k, 4);
				}
			}
		}
		if (((type.get().equals(Crosshair.TEXTURE) || type.get().equals(Crosshair.CUSTOM)) ? customAttackIndicator.get() : true) && indicator == class_4061.field_18152) {
			//noinspection DataFlowIssue
			float progress = this.client.field_1724.method_7261(0.0F);
			if (progress != 1.0F) {
				fillRenderType(graphics, blend, getRawX() + (getWidth() / 2) - 6, getRawY() + (getHeight() / 2) + 9,
					11, 1, attackIndicatorBackgroundColor.get());
				fillRenderType(graphics, blend, getRawX() + (getWidth() / 2) - 6, getRawY() + (getHeight() / 2) + 9,
					(int) (progress * 11), 1, attackIndicatorForegroundColor.get());
			}
		}
		graphics.method_51448().popMatrix();
	}

	private RenderPipeline renderType(boolean blend) {
		return blend ? class_10799.field_56890 : class_10799.field_56883;
	}

	private void fillRenderType(class_332 graphics, boolean blend, int x, int y, int width, int height, Color color) {
		if (blend) {
			graphics.method_48196(CROSSHAIR_NO_TEX, x, y, width + x, height + y, color.toInt());
		} else {
			graphics.method_25294(x, y, width + x, height + y, color.toInt());
		}
	}

	public Color getColor() {
		class_239 hit = client.field_1765;
		if (hit == null || hit.method_17783() == class_239.class_240.field_1333) {
			return defaultColor.get();
		} else if (hit.method_17783() == class_239.class_240.field_1331) {
			return entityColor.get();
		} else if (hit.method_17783() == class_239.class_240.field_1332) {
			class_2338 blockPos = ((class_3965) hit).method_17777();
			class_1937 world = this.client.field_1687;
			//noinspection DataFlowIssue
			if (world.method_8320(blockPos).method_26196(world, blockPos) != null
				|| world.method_8320(blockPos).method_26204() instanceof class_4739<?>) {
				return containerColor.get();
			}
		}
		return defaultColor.get();
	}

	@Override
	public void renderPlaceholder(AxoRenderContext graphics, float delta) {
		// Shouldn't need this...
	}

	@Override
	public AnchorPoint getAnchor() {
		return AnchorPoint.MIDDLE_MIDDLE;
	}

	@AllArgsConstructor
	public enum Crosshair {
		CROSS, DOT, DIRECTION, TEXTURE, CUSTOM
	}
}
