/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.api.chat;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;

import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.ContextMenu;
import io.github.axolotlclient.api.ContextMenuScreen;
import io.github.axolotlclient.api.handlers.ChatHandler;
import io.github.axolotlclient.api.requests.ChannelRequest;
import io.github.axolotlclient.api.types.Channel;
import io.github.axolotlclient.api.types.ChatMessage;
import io.github.axolotlclient.modules.auth.Auth;
import io.github.axolotlclient.util.ClientColors;
import lombok.Getter;
import lombok.Setter;
import net.minecraft.class_11909;
import net.minecraft.class_11910;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_410;
import net.minecraft.class_4280;
import net.minecraft.class_437;
import net.minecraft.class_5481;
import net.minecraft.class_7532;

public class ChatWidget extends class_4280<ChatWidget.ChatLine> {

	private final List<ChatMessage> messages = new ArrayList<>();
	private final Channel channel;
	private final class_310 client;
	private final ContextMenuScreen screen;
	@Setter
	@Getter
	private int x, y, field_22758, field_22759;

	public ChatWidget(Channel channel, int x, int y, int width, int height, ContextMenuScreen screen) {
		super(class_310.method_1551(), width, height, y, 13);
		this.channel = channel;
		this.client = class_310.method_1551();
		method_46421(x + 5);

		this.screen = screen;
		this.x = x;
		this.y = y;
		this.field_22758 = width;
		this.field_22759 = height;
		channel.getMessages().forEach(this::addMessage);

		ChatHandler.getInstance().setMessagesConsumer(chatMessages -> chatMessages.forEach(this::addMessage));
		ChatHandler.getInstance().setMessageConsumer(this::addMessage);
		ChatHandler.getInstance().setEnableNotifications(message -> !message.channelId().equals(channel.getId()));

		method_44382(method_44390());
	}

	@Override
	protected int method_65507() {
		return x + field_22758 - 6;
	}

	@Override
	public int method_25322() {
		return field_22758 - 60;
	}

	private void addMessage(ChatMessage message) {
		List<class_5481> list = client.field_1772.method_1728(class_2561.method_43470(message.content()), method_25322());

		boolean scrollToBottom = method_44387() == method_44390();

		if (!messages.isEmpty()) {
			ChatMessage prev = messages.getLast();
			if (!(prev.sender().equals(message.sender()) &&
				prev.senderDisplayName().equals(message.senderDisplayName()))) {
				method_25321(new NameChatLine(message));
			} else {
				if (message.timestamp().getEpochSecond() - prev.timestamp().getEpochSecond() > 150) {
					method_25321(new NameChatLine(message));
				}
			}
		} else {
			method_25321(new NameChatLine(message));
		}

		list.forEach(t -> method_25321(new ChatLine(t, message)));
		messages.add(message);

		method_73372(Comparator.comparingLong(c -> c.getOrigin().timestamp().getEpochSecond()));

		if (scrollToBottom) {
			method_44382(method_44390());
		}
		messages.sort(Comparator.comparingLong(value -> value.timestamp().getEpochSecond()));
	}

	private void loadMessages() {
		long before;
		if (!messages.isEmpty()) {
			before = messages.getFirst().timestamp().getEpochSecond();
		} else {
			before = Instant.now().getEpochSecond();
		}
		ChatHandler.getInstance().getMessagesBefore(channel, before);
	}

	@Override
	public boolean method_25401(double mouseX, double mouseY, double amountX, double amountY) {
		double scrollAmount = (this.method_44387() - amountY * method_44393());
		if (scrollAmount < 0) {
			loadMessages();
		}
		method_44382(scrollAmount);
		return true;
	}

	public void remove() {
		ChatHandler.getInstance().setMessagesConsumer(ChatHandler.DEFAULT_MESSAGES_CONSUMER);
		ChatHandler.getInstance().setMessageConsumer(ChatHandler.DEFAULT_MESSAGE_CONSUMER);
		ChatHandler.getInstance().setEnableNotifications(ChatHandler.DEFAULT);
	}

	@Override
	protected void renderSelection(class_332 guiGraphics, ChatLine entry, int i) {
	}

	@Override
	protected boolean method_25351(class_11910 mouseButtonInfo) {
		return true;
	}

	public class ChatLine extends Entry<ChatLine> {
		protected final class_310 client = ChatWidget.this.client;
		@Getter
		private final class_5481 content;
		@Getter
		private final ChatMessage origin;

		public ChatLine(class_5481 content, ChatMessage origin) {
			super();
			this.content = content;
			this.origin = origin;
		}

		@Override
		public boolean mouseClicked(class_11909 event, boolean doubleClick) {
			if (event.method_74245() == 0) {
				ChatWidget.this.method_25313(this);
				return true;
			}
			if (event.method_74245() == 1) {
				ContextMenu.Builder builder =
					ContextMenu.builder().title(class_2561.method_43470(origin.sender().getName())).spacer();
				if (!origin.sender().equals(API.getInstance().getSelf())) {
					builder.entry(class_2561.method_43471("api.friends.chat"),
						buttonWidget -> ChannelRequest.getOrCreateDM(origin.sender()).whenCompleteAsync(
							(channel, throwable) -> client.execute(
								() -> client.method_1507(new ChatScreen(screen.getParent(), channel))))).spacer();
				}
				builder.entry(class_2561.method_43471("api.chat.report.message"), buttonWidget -> {
					class_437 previous = client.field_1755;
					client.method_1507(new class_410(b -> {
						if (b) {
							ChatHandler.getInstance().reportMessage(origin);
						}
						client.method_1507(previous);
					}, class_2561.method_43471("api.channels.confirm_report"), class_2561.method_43469("api.channels.confirm_report.desc", origin.content())));
				}).spacer().entry(class_2561.method_43471("action.copy"), buttonWidget -> client.field_1774.method_1455(origin.content()));
				screen.setContextMenu(builder.build());
				return true;
			}
			return false;
		}

		protected void renderExtras(class_332 graphics, int x, int y, int mouseX, int mouseY) {
		}

		@Override
		public void renderContent(class_332 graphics, int mouseX, int mouseY, boolean hovered, float tickDelta) {
			for (ChatLine l : method_25396()) {
				if (l.getOrigin().equals(origin)) {
					if (Objects.equals(method_37019(), l)) {
						hovered = true;
						break;
					}
				}
			}
			if (hovered && !screen.hasContextMenu()) {
				var x = getContentX();
				var y = getContentY();
				var entryWidth = getContentWidth();
				var entryHeight = getContentHeight();
				var index = ChatWidget.this.method_25396().indexOf(this);
				graphics.method_25294(x - 2 - 22, y - 2, x + entryWidth + 20, y + entryHeight - 1, 0x33FFFFFF);
				if (index < method_25396().size() - 1 && method_25396().get(index + 1).getOrigin().equals(origin)) {
					graphics.method_25294(x - 2 - 22, y + entryHeight - 1, x + entryWidth + 20, y + entryHeight + 2,
						0x33FFFFFF
					);
				}
				if ((index < method_25396().size() - 1 && !method_25396().get(index + 1).getOrigin().equals(origin)) ||
					index == method_25396().size() - 1) {
					graphics.method_25294(x - 2 - 22, y + entryHeight - 1, x + entryWidth + 20, y + entryHeight, 0x33FFFFFF);
				}
			}
			renderExtras(graphics, getContentX(), getContentY(), mouseX, mouseY);
			graphics.method_51430(client.field_1772, content, getContentX(), getContentY(), -1, false);
		}

		@Override
		public class_2561 getNarration() {
			return class_2561.method_43470(origin.content());
		}
	}

	public class NameChatLine extends ChatLine {

		private final String formattedTime;

		public NameChatLine(ChatMessage message) {
			super(class_2561.method_43470(message.senderDisplayName()).method_10862(class_2583.field_24360.method_10982(true))
				.method_30937(), message);

			DateTimeFormatter DATE_FORMAT = DateTimeFormatter.ofPattern("d/M/yyyy H:mm");
			formattedTime = DATE_FORMAT.format(message.timestamp().atZone(ZoneId.systemDefault()));
		}

		@Override
		protected void renderExtras(class_332 graphics, int x, int y, int mouseX, int mouseY) {
			graphics.method_51448().pushMatrix();
			class_2960 texture =
				Auth.getInstance().getSkinTexture(getOrigin().sender().getUuid());
			class_7532.method_44445(graphics, texture, x - 22, y, 18, true, false, -1);
			graphics.method_51433(client.field_1772, formattedTime, client.field_1772.method_1727(getContent()) + x + 5, y,
				ClientColors.GRAY.toInt(), false
			);
			graphics.method_51448().popMatrix();
		}
	}
}
