/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.auth;

import java.util.*;
import java.util.concurrent.CompletableFuture;

import com.mojang.authlib.minecraft.UserApiService;
import com.mojang.authlib.yggdrasil.ProfileResult;
import com.mojang.authlib.yggdrasil.YggdrasilAuthenticationService;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.util.UUIDHelper;
import io.github.axolotlclient.mixin.DownloadedPackSourceAccessor;
import io.github.axolotlclient.mixin.MinecraftClientAccessor;
import io.github.axolotlclient.mixin.ServerPackManagerAccessor;
import io.github.axolotlclient.mixin.SplashManagerAccessor;
import io.github.axolotlclient.modules.Module;
import io.github.axolotlclient.modules.auth.skin.SkinManager;
import io.github.axolotlclient.util.ThreadExecuter;
import io.github.axolotlclient.util.notifications.Notifications;
import io.github.axolotlclient.util.options.GenericOption;
import lombok.Getter;
import net.minecraft.class_1068;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_320;
import net.minecraft.class_410;
import net.minecraft.class_437;
import net.minecraft.class_5520;
import net.minecraft.class_7569;
import net.minecraft.class_7574;
import net.minecraft.class_7853;

public class Auth extends Accounts implements Module {

	@Getter
	private final static Auth Instance = new Auth();
	public final BooleanOption showButton = new BooleanOption("auth.showButton", false);
	public final BooleanOption skinManagerAnimations = new BooleanOption("skins.manage.animations", true);
	private final class_310 mc = class_310.method_1551();
	private final GenericOption viewAccounts = new GenericOption("viewAccounts", "clickToOpen", () -> mc.method_1507(new AccountsScreen(mc.field_1755)));
	private final Set<String> loadingTexture = new HashSet<>();
	private final Map<String, class_2960> textures = new WeakHashMap<>();
	@Getter
	private final SkinManager skinManager = new SkinManager();

	@Override
	public void init() {
		load();
		this.msApi = new MSApi(this, () -> mc.field_1690.field_1883);
		if (isContained(mc.method_1548().method_1675())) {
			current = getAccounts().stream().filter(account -> account.getUuid().equals(UUIDHelper.toUndashed(mc.method_1548().method_44717()))).toList().getFirst();
			if (current.needsRefresh()) {
				current.refresh(msApi).thenRun(this::save);
			}
		} else {
			current = new Account(mc.method_1548().method_1676(), UUIDHelper.toUndashed(mc.method_1548().method_44717()), mc.method_1548().method_1674());
		}

		category.add(showButton, viewAccounts, skinManagerAnimations);
		AxolotlClient.config().general.add(category);
	}

	@Override
	protected void login(Account account) {
		if (mc.field_1687 != null) {
			return;
		}

		if (account.needsRefresh() && !account.isOffline()) {
			if (account.isExpired()) {
				Notifications.getInstance().addStatus(class_2561.method_43471("auth.notif.title"), class_2561.method_43469("auth.notif.refreshing", account.getName()));
			}
			account.refresh(msApi).thenAccept(a -> {
				if (!a.isExpired()) {
					login(a);
				}
			}).thenRun(this::save);
		} else {
			try {
				API.getInstance().shutdown();
				var mcAccessor = (MinecraftClientAccessor) mc;
				mcAccessor.axolotlclient$setSession(new class_320(account.getName(), UUIDHelper.fromUndashed(account.getUuid()), account.getAuthToken(), Optional.empty(), Optional.empty()));
				UserApiService service;
				if (account.isOffline()) {
					service = UserApiService.OFFLINE;
				} else {
					service = new YggdrasilAuthenticationService(mc.method_1487()).createUserApiService(mc.method_1548().method_1674());
				}
				mcAccessor.axolotlclient$setUserApiService(service);
				var sourceAccessor = (DownloadedPackSourceAccessor) mc.method_1516();
				((ServerPackManagerAccessor) sourceAccessor.axolotlclient$getManager())
					.axolotlclient$setDownloader(sourceAccessor.axolotlclient$createDownloader(sourceAccessor.axolotlcleint$getDownloadQueue(), mc::method_63588, mc.method_1548(), mc.method_1487()));
				mcAccessor.axolotlclient$setSocialInteractionsManager(new class_5520(mc, service));
				mcAccessor.axolotlclient$setPlayerKeyPairManager(class_7853.method_46532(service, mc.method_1548(), mc.field_1697.toPath()));
				mcAccessor.axolotlclient$setChatReportingContext(class_7574.method_44599(class_7569.method_44586(), service));
				mcAccessor.axolotlclient$setProfileFuture(CompletableFuture.supplyAsync(() -> mc.method_73361().comp_837().fetchProfile(mc.method_1548().method_44717(), true), class_156.method_55473()));
				((SplashManagerAccessor) mc.method_18095()).setUser(mc.method_1548());
				save();
				current = account;
				Notifications.getInstance().addStatus(class_2561.method_43471("auth.notif.title"), class_2561.method_43469("auth.notif.login.successful", current.getName()));
				API.getInstance().startup(account);
			} catch (Exception e) {
				Notifications.getInstance().addStatus(class_2561.method_43471("auth.notif.title"), class_2561.method_43471("auth.notif.login.failed"));
			}
		}
	}

	@Override
	CompletableFuture<Account> showAccountsExpiredScreen(Account account) {
		class_437 current = mc.field_1755;
		var fut = new CompletableFuture<Account>();
		mc.execute(() -> mc.method_1507(new class_410((bl) -> {
			if (bl) {
				msApi.startDeviceAuth().thenRun(() -> fut.complete(account));
			} else {
				fut.cancel(true);
			}
			mc.method_1507(current);
		}, class_2561.method_43471("auth"), class_2561.method_43469("auth.accountExpiredNotice", account.getName()))));
		return fut;
	}

	@Override
	void displayDeviceCode(DeviceFlowData data) {
		mc.execute(() -> mc.method_1507(new DeviceCodeDisplayScreen(mc.field_1755, data)));
	}

	private void loadTexture(String uuid) {
		if (!loadingTexture.contains(uuid)) {
			loadingTexture.add(uuid);
			ThreadExecuter.scheduleTask(() -> {
				UUID uUID = UUIDHelper.fromUndashed(uuid);
				ProfileResult profileResult = mc.method_73361().comp_837().fetchProfile(uUID, false);
				if (profileResult != null) {
					mc.method_1582().method_52863(profileResult.profile()).thenAccept(playerSkin -> playerSkin.ifPresent(skin -> textures.put(uuid, skin.comp_1626().comp_3627())));
				}
				loadingTexture.remove(uuid);
			});
		}
	}

	public class_2960 getSkinTexture(Account account) {
		return getSkinTexture(account.getUuid());
	}

	public class_2960 getSkinTexture(io.github.axolotlclient.api.types.User user) {
		return getSkinTexture(user.getUuid());
	}

	public class_2960 getSkinTexture(String uuid) {
		if (!textures.containsKey(uuid)) {
			loadTexture(uuid);
			return Objects.requireNonNullElseGet(textures.get(uuid), () -> class_1068.method_4648(UUIDHelper.fromUndashed(uuid)).comp_1626().comp_3627());
		}
		return textures.get(uuid);
	}
}
