/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.hud;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.modules.hud.gui.entry.TextHudEntry;
import io.github.axolotlclient.modules.hud.util.DrawPosition;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import lombok.Getter;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_10799;
import net.minecraft.class_2561;
import net.minecraft.class_2588;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3262;
import net.minecraft.class_332;
import net.minecraft.class_7367;
import org.jetbrains.annotations.NotNull;

import static io.github.axolotlclient.modules.hud.util.DrawUtil.drawString;

public class PackDisplayHud extends TextHudEntry {

	public static final class_2960 ID = class_2960.method_60655("axolotlclient", "packdisplayhud");
	public final List<PackWidget> widgets = new ArrayList<>();
	private final BooleanOption iconsOnly = new BooleanOption("iconsonly", false);
	private final class_310 client = (class_310) super.client;
	private PackWidget placeholder;

	public PackDisplayHud() {
		super(120, 18, true);
	}

	@Override
	public void renderComponent(AxoRenderContext context, float f) {
		final var graphics = (class_332) context;

		DrawPosition pos = getContentPos();

		if (widgets.isEmpty()) init();

		if (background.get()) {
			DrawUtil.fillRect(graphics, getBounds(), backgroundColor.get());
		}

		if (outline.get()) DrawUtil.outlineRect(graphics, getBounds(), outlineColor.get());

		int y = pos.y() + 1;
		for (int i = widgets.size() - 1; i >= 0; i--) { // Badly reverse the order (I'm sure there are better ways to do this)
			widgets.get(i).render(graphics, pos.x() + 1, y);
			y += 17;
		}
		if (y - pos.y() != getContentHeight()) {
			setContentHeight(y - pos.y());
			onBoundsUpdate();
		}
	}

	@Override
	public void init() {
		var selected = client.method_1520().method_14444();
		var valid = selected.stream()
			.filter(p -> !(p.method_56933().comp_2330().method_10851() instanceof class_2588 tr && tr.method_11022().matches("pack\\.name\\.fabricMods?")))
			.toList();
		var listSize = valid.size();
		valid.forEach(profile -> {
			try (class_3262 pack = profile.method_14458()) {
				if (listSize == 1) {
					widgets.add(createWidget(profile.method_14457(), pack));
				} else if (!pack.method_14409().equalsIgnoreCase("vanilla")) {
					widgets.add(createWidget(profile.method_14457(), pack));
				}
			} catch (Exception ignored) {
			}
		});

		AtomicInteger w = new AtomicInteger(20);
		widgets.forEach(packWidget -> {
			int textW = client.field_1772.method_1727(packWidget.getName()) + 20;
			if (textW > w.get()) w.set(textW);
		});
		setContentWidth(w.get());

		setContentHeight(widgets.size() * 17 + 1);
		onBoundsUpdate();
	}

	private PackWidget createWidget(class_2561 displayName, class_3262 pack) throws IOException, AssertionError {
		class_7367<@NotNull InputStream> supplier = pack.method_14410("pack.png");
		assert supplier != null;
		InputStream stream = supplier.get();
		class_2960 id = class_2960.method_60655(ID.method_12836(), ID.method_12832() + "/" + pack.method_14409());
		client.method_1531().method_4616(id, new class_1043(id::toString, class_1011.method_4309(stream)));
		stream.close();
		return new PackWidget(displayName, id);
	}

	@Override
	public void renderPlaceholderComponent(AxoRenderContext graphics, float f) {
		boolean updateBounds = false;
		if (getContentHeight() < 18) {
			setContentHeight(18);
			updateBounds = true;
		}
		if (getContentWidth() < 56) {
			setContentWidth(56);
			updateBounds = true;
		}
		if (updateBounds) {
			onBoundsUpdate();
		}
		if (placeholder == null) {
			try (class_3262 defaultPack = client.method_45573()) {
				placeholder = createWidget(class_2561.method_43470(defaultPack.method_14409()), defaultPack);
			} catch (Exception ignored) {
			}
		} else {
			placeholder.render((class_332) graphics, getContentPos().x() + 1, getContentPos().y() + 1);
		}
	}

	@Override
	public List<Option<?>> getConfigurationOptions() {
		List<Option<?>> options = super.getConfigurationOptions();
		options.add(iconsOnly);
		return options;
	}

	@Override
	public class_2960 getId() {
		return ID;
	}

	public void update() {
		widgets.clear();
		init();
	}

	public class PackWidget {

		@Getter
		public final String name;
		private final class_2960 texture;

		public PackWidget(class_2561 name, class_2960 id) {
			this.name = name.getString();
			texture = id;
		}

		public void render(class_332 graphics, int x, int y) {
			if (!iconsOnly.get()) {
				graphics.method_25290(class_10799.field_56883, texture, x, y, 0, 0, 16, 16, 16, 16);
			}
			drawString(graphics, name, x + 18, y + 16 / 2 - 9 / 2, textColor.get().toInt(), shadow.get());
		}
	}
}
