/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.hud.vanilla;

import java.util.Comparator;
import java.util.List;

import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.ColorOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.IntegerOption;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.modules.hud.gui.entry.TextHudEntry;
import io.github.axolotlclient.modules.hud.gui.layout.AnchorPoint;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.modules.hud.util.Rectangle;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_266;
import net.minecraft.class_268;
import net.minecraft.class_269;
import net.minecraft.class_274;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_8646;
import net.minecraft.class_9011;
import net.minecraft.class_9014;
import net.minecraft.class_9015;
import net.minecraft.class_9022;
import net.minecraft.class_9025;
import net.minecraft.class_9848;
import net.minecraft.world.scores.*;

/**
 * This implementation of Hud modules is based on KronHUD.
 * <a href="https://github.com/DarkKronicle/KronHUD">Github Link.</a>
 *
 * <p>License: GPL-3.0</p>
 */

public class ScoreboardHud extends TextHudEntry {

	public static final class_2960 ID = class_2960.method_60655("kronhud", "scoreboardhud");
	public static final class_266 placeholder = class_156.method_656(() -> {
		class_269 placeScore = new class_269();
		class_266 objective =
			placeScore.method_1168("placeholder", class_274.field_1468, class_2561.method_43470("Scoreboard"),
				class_274.class_275.field_1472, false, class_9025.field_47567
			);
		class_9014 dark = placeScore.method_1180(class_9015.method_55422("DarkKronicle"), objective);
		dark.method_55410(8780);

		class_9014 moeh = placeScore.method_1180(class_9015.method_55422("moehreag"), objective);
		moeh.method_55410(743);

		class_9014 kode = placeScore.method_1180(class_9015.method_55422("TheKodeToad"), objective);
		kode.method_55410(2948);

		placeScore.method_1158(class_8646.field_45157, objective);
		return objective;
	});

	private final ColorOption backgroundColor = new ColorOption("backgroundcolor", new Color(0x4C000000));
	private final ColorOption topColor = new ColorOption("topbackgroundcolor", new Color(0x66000000));
	private final IntegerOption topPadding = new IntegerOption("toppadding", 0, 0, 4);
	private final BooleanOption scores = new BooleanOption("scores", true);
	private final ColorOption scoreColor = new ColorOption("scorecolor", new Color(0xFFFF5555));
	private final IntegerOption textAlpha = new IntegerOption("text_alpha", 255, 0, 255);

	private final class_310 client = (class_310) super.client;

	public ScoreboardHud() {
		super(200, 146, true);
	}

	@Override
	public void render(AxoRenderContext graphics, float delta) {
		graphics.br$pushMatrix();
		scale(graphics);
		renderComponent(graphics, delta);
		graphics.br$popMatrix();
	}

	@Override
	public void renderComponent(AxoRenderContext graphics, float delta) {
		//noinspection DataFlowIssue
		class_269 scoreboard = this.client.field_1687.method_8428();
		class_266 objective = null;
		//noinspection DataFlowIssue
		class_268 playerTeam = scoreboard.method_1164(client.field_1724.method_5820());
		if (playerTeam != null) {
			class_8646 displaySlot = class_8646.method_52622(playerTeam.method_1202());
			if (displaySlot != null) {
				objective = scoreboard.method_1189(displaySlot);
			}
		}

		class_266 objective2 =
			objective != null ? objective : scoreboard.method_1189(class_8646.field_45157);
		if (objective2 != null) {
			this.displayScoreboardSidebar((class_332) graphics, objective2);
		}
	}

	@Override
	public void renderPlaceholderComponent(AxoRenderContext graphics, float delta) {
		displayScoreboardSidebar((class_332) graphics, placeholder);
	}

	// Abusing this could break some stuff/could allow for unfair advantages. The goal is not to do this, so it won't
	// show any more information than it would have in vanilla.
	private void displayScoreboardSidebar(class_332 guiGraphics, class_266 objective) {
		class_327 font = client.field_1772;
		class_269 scoreboard = objective.method_1117();
		class_9022 numberFormat = objective.method_55380(class_9025.field_47567);

		@Environment(EnvType.CLIENT)
		record DisplayEntry(class_2561 name, class_2561 score, int scoreWidth) {
		}

		DisplayEntry[] entries = scoreboard.method_1184(objective)
			.stream()
			.filter(entry -> !entry.method_55385())
			.sorted(Comparator.comparing(class_9011::comp_2128)
				.reversed()
				.thenComparing(class_9011::comp_2127, String.CASE_INSENSITIVE_ORDER))
			.limit(15L)
			.map(playerScoreEntry -> {
				class_268 playerTeam = scoreboard.method_1164(playerScoreEntry.comp_2127());
				class_2561 componentx = playerScoreEntry.method_55387();
				class_2561 component2 = class_268.method_1142(playerTeam, componentx);
				class_2561 component3 = playerScoreEntry.method_55386(numberFormat);
				int ix = font.method_27525(component3);
				return new DisplayEntry(component2, component3, ix);
			})
			.toArray(DisplayEntry[]::new);
		class_2561 title = objective.method_1114();
		int titleWidth = font.method_27525(title);
		int maxWidth = titleWidth;
		int textOffset = font.method_1727(": ");

		for (DisplayEntry lv : entries) {
			maxWidth = Math.max(maxWidth, font.method_27525(lv.name) + (lv.scoreWidth > 0 && scores.get() ?
				textOffset + lv.scoreWidth : 0));
		}

		maxWidth += 3;
		int m = entries.length;
		int mainHeight = m * 9;

		int newHeight = mainHeight + 10 + topPadding.get() * 2;

		boolean updated = false;
		if (newHeight + 1 != height) {
			setHeight(newHeight + 1);
			updated = true;
		}
		if (maxWidth + 1 != width) {
			setWidth(maxWidth + 1);
			updated = true;
		}
		if (updated) {
			onBoundsUpdate();
		}

		Rectangle bounds = getBounds();

		int yEnd = bounds.y() + bounds.height();
		int textX = bounds.x() + 3;
		int xEnd = bounds.x() + bounds.width() - 1;
		int titleEnd = yEnd - mainHeight;
		if (background.get()) {
			guiGraphics.method_25294(textX - 2, titleEnd - 9 - 1 - topPadding.get() * 2, xEnd, titleEnd - 1,
				topColor.get().toInt());
			guiGraphics.method_25294(textX - 2, titleEnd - 1, xEnd, yEnd, backgroundColor.get().toInt());
		}
		guiGraphics.method_51439(font, title, textX + maxWidth / 2 - titleWidth / 2, titleEnd - 9 - topPadding.get(),
			class_9848.method_61330(textAlpha.get(), -1), shadow.get());

		for (int v = 0; v < m; v++) {
			DisplayEntry lv2 = entries[v];
			int w = yEnd - (m - v) * 9;
			guiGraphics.method_51439(font, lv2.name, textX, w, class_9848.method_61330(textAlpha.get(), -1), shadow.get());
			if (scores.get()) {
				guiGraphics.method_51439(font, lv2.score, xEnd - lv2.scoreWidth, w, scoreColor.get().toInt(),
					shadow.get());
			}
		}

		if (outline.get() && outlineColor.get().getAlpha() > 0) {
			DrawUtil.outlineRect(guiGraphics, bounds, outlineColor.get());
		}
	}

	@Override
	public List<Option<?>> getConfigurationOptions() {
		List<Option<?>> options = super.getConfigurationOptions();
		options.remove(backgroundPadding);
		options.remove(backgroundRounding);
		options.remove(roundBackground);
		options.set(options.indexOf(super.backgroundColor), backgroundColor);
		options.add(hide);
		options.add(topColor);
		options.add(scores);
		options.add(scoreColor);
		options.add(topPadding);
		options.remove(textColor);
		options.add(textAlpha);
		return options;
	}

	@Override
	public class_2960 getId() {
		return ID;
	}

	@Override
	protected AnchorPoint getDefaultAnchor() {
		return AnchorPoint.MIDDLE_RIGHT;
	}

	@Override
	public double getDefaultX() {
		return 1.0;
	}

	@Override
	public double getDefaultY() {
		return 0.5;
	}
}
