/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.api.chat;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;

import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.ContextMenu;
import io.github.axolotlclient.api.ContextMenuScreen;
import io.github.axolotlclient.api.handlers.ChatHandler;
import io.github.axolotlclient.api.requests.ChannelRequest;
import io.github.axolotlclient.api.types.Channel;
import io.github.axolotlclient.api.types.ChatMessage;
import io.github.axolotlclient.modules.auth.Auth;
import io.github.axolotlclient.util.ClientColors;
import lombok.Getter;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_410;
import net.minecraft.class_4280;
import net.minecraft.class_437;
import net.minecraft.class_5481;

public class ChatWidget extends class_4280<ChatWidget.ChatLine> {

	private final List<ChatMessage> messages = new ArrayList<>();
	private final Channel channel;
	private final class_310 field_22740;
	private final ContextMenuScreen screen;

	public ChatWidget(Channel channel, int x, int y, int width, int height, ContextMenuScreen screen) {
		super(class_310.method_1551(), width, height, y, y + height, 13);
		this.channel = channel;
		this.field_22740 = class_310.method_1551();
		method_25333(x + 5);

		method_25315(false, 0);
		method_31323(false);
		method_31322(false);
		this.screen = screen;
		channel.getMessages().forEach(this::addMessage);

		ChatHandler.getInstance().setMessagesConsumer(chatMessages -> chatMessages.forEach(this::addMessage));
		ChatHandler.getInstance().setMessageConsumer(this::addMessage);
		ChatHandler.getInstance().setEnableNotifications(message -> !message.channelId().equals(channel.getId()));

		method_25307(method_25331());
	}

	@Override
	protected int method_25329() {
		return method_25342() + field_22742 - 6;
	}

	@Override
	public int method_25322() {
		return field_22742 - 60;
	}

	private void addMessage(ChatMessage message) {
		List<class_5481> list = field_22740.field_1772.method_1728(class_2561.method_30163(message.content()), method_25322());

		boolean scrollToBottom = method_25341() == method_25331();

		if (!messages.isEmpty()) {
			ChatMessage prev = messages.get(messages.size() - 1);
			if (!(prev.sender().equals(message.sender()) && prev.senderDisplayName().equals(message.senderDisplayName()))) {
				method_25321(new NameChatLine(message));
			} else {
				if (message.timestamp().getEpochSecond() - prev.timestamp().getEpochSecond() > 150) {
					method_25321(new NameChatLine(message));
				}
			}
		} else {
			method_25321(new NameChatLine(message));
		}

		list.forEach(t -> method_25321(new ChatLine(t, message)));
		messages.add(message);

		method_25396().sort(Comparator.comparingLong(c -> c.getOrigin().timestamp().getEpochSecond()));

		if (scrollToBottom) {
			method_25307(method_25331());
		}
		messages.sort(Comparator.comparingLong(value -> value.timestamp().getEpochSecond()));
	}

	private void loadMessages() {
		long before;
		if (!messages.isEmpty()) {
			before = messages.get(0).timestamp().getEpochSecond();
		} else {
			before = Instant.now().getEpochSecond();
		}
		ChatHandler.getInstance().getMessagesBefore(channel, before);
	}

	@Override
	public boolean method_25401(double mouseX, double mouseY, double amountY) {
		double scrollAmount = (this.method_25341() - amountY * (double) this.field_22741 / 2.0);
		if (scrollAmount < 0) {
			loadMessages();
		}
		method_25307(scrollAmount);
		return true;
	}

	public void remove() {
		ChatHandler.getInstance().setMessagesConsumer(ChatHandler.DEFAULT_MESSAGES_CONSUMER);
		ChatHandler.getInstance().setMessageConsumer(ChatHandler.DEFAULT_MESSAGE_CONSUMER);
		ChatHandler.getInstance().setEnableNotifications(ChatHandler.DEFAULT);
	}

	@Override
	protected void method_44398(class_332 graphics, int y, int entryWidth, int entryHeight, int borderColor, int fillColor) {
	}

	public class ChatLine extends class_4280.class_4281<ChatLine> {
		protected final class_310 client = class_310.method_1551();
		@Getter
		private final class_5481 content;
		@Getter
		private final ChatMessage origin;

		public ChatLine(class_5481 content, ChatMessage origin) {
			super();
			this.content = content;
			this.origin = origin;
		}

		@Override
		public boolean method_25402(double mouseX, double mouseY, int button) {
			if (button == 0) {
				ChatWidget.this.method_25313(this);
				return true;
			}
			if (button == 1) {
				ContextMenu.Builder builder = ContextMenu.builder()
					.title(class_2561.method_30163(origin.sender().getName()))
					.spacer();
				if (!origin.sender().equals(API.getInstance().getSelf())) {
					builder.entry(class_2561.method_43471("api.friends.chat"), buttonWidget ->
							ChannelRequest.getOrCreateDM(origin.sender())
								.whenCompleteAsync((channel, throwable) -> client.execute(() -> client.method_1507(new ChatScreen(screen.getParent(), channel)))))
						.spacer();
				}
				builder.entry(class_2561.method_43471("api.chat.report.message"), buttonWidget -> {
						class_437 previous = client.field_1755;
						client.method_1507(new class_410(b -> {
							if (b) {
								ChatHandler.getInstance().reportMessage(origin);
							}
							client.method_1507(previous);
						}, class_2561.method_43471("api.channels.confirm_report"), class_2561.method_43469("api.channels.confirm_report.desc", origin.content())));
					})
					.spacer()
					.entry(class_2561.method_43471("action.copy"), buttonWidget -> client.field_1774.method_1455(origin.content()));
				screen.setContextMenu(builder.build());
				return true;
			}
			return false;
		}

		protected void renderExtras(class_332 graphics, int x, int y, int mouseX, int mouseY) {
		}

		@Override
		public void method_25343(class_332 graphics, int index, int y, int x, int entryWidth, int entryHeight, int mouseX, int mouseY, boolean hovered, float tickDelta) {
			for (ChatLine l : method_25396()) {
				if (l.getOrigin().equals(origin)) {
					if (Objects.equals(method_37019(), l)) {
						hovered = true;
						break;
					}
				}
			}
			if (hovered && !screen.hasContextMenu()) {
				graphics.method_25294(x - 2 - 22, y - 2, x + entryWidth + 20, y + entryHeight - 1, 0x33FFFFFF);
				if (index < method_25396().size() - 1 && method_25396().get(index + 1).getOrigin().equals(origin)) {
					graphics.method_25294(x - 2 - 22, y + entryHeight - 1, x + entryWidth + 20, y + entryHeight + 2, 0x33FFFFFF);
				}
				if ((index < method_25396().size() - 1 && !method_25396().get(index + 1).getOrigin().equals(origin)) || index == method_25396().size() - 1) {
					graphics.method_25294(x - 2 - 22, y + entryHeight - 1, x + entryWidth + 20, y + entryHeight, 0x33FFFFFF);
				}
			}
			renderExtras(graphics, x, y, mouseX, mouseY);
			graphics.method_51430(class_310.method_1551().field_1772, content, x, y, -1, false);
		}

		@Override
		public class_2561 method_37006() {
			return class_2561.method_30163(origin.content());
		}
	}

	public class NameChatLine extends ChatLine {

		private final String formattedTime;

		public NameChatLine(ChatMessage message) {
			super(class_2561.method_43470(message.senderDisplayName())
				.method_10862(class_2583.field_24360.method_10982(true)).method_30937(), message);

			DateTimeFormatter DATE_FORMAT = DateTimeFormatter.ofPattern("d/M/yyyy H:mm");
			formattedTime = DATE_FORMAT.format(message.timestamp().atZone(ZoneId.systemDefault()));
		}

		@Override
		protected void renderExtras(class_332 graphics, int x, int y, int mouseX, int mouseY) {
			RenderSystem.disableBlend();
			class_2960 texture = Auth.getInstance().getSkinTexture(getOrigin().sender().getUuid()
			);
			graphics.method_25293(texture, x - 22, y, 18, 18, 8, 8, 8, 8, 64, 64);
			graphics.method_25293(texture, x - 22, y, 18, 18, 40, 8, 8, 8, 64, 64);
			RenderSystem.enableBlend();
			graphics.method_51433(client.field_1772, formattedTime, client.field_1772.method_1727(getContent()) + x + 5, y, ClientColors.GRAY.toInt(), false);
		}
	}
}
