/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.auth;

import java.util.*;
import java.util.concurrent.CompletableFuture;

import com.mojang.authlib.GameProfile;
import com.mojang.authlib.minecraft.MinecraftProfileTexture;
import com.mojang.authlib.minecraft.UserApiService;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.types.User;
import io.github.axolotlclient.api.util.UUIDHelper;
import io.github.axolotlclient.mixin.MinecraftClientAccessor;
import io.github.axolotlclient.modules.Module;
import io.github.axolotlclient.modules.auth.skin.SkinManager;
import io.github.axolotlclient.util.ThreadExecuter;
import io.github.axolotlclient.util.notifications.Notifications;
import io.github.axolotlclient.util.options.GenericOption;
import lombok.Getter;
import net.minecraft.class_1068;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_320;
import net.minecraft.class_410;
import net.minecraft.class_437;
import net.minecraft.class_5520;
import net.minecraft.class_7569;
import net.minecraft.class_7574;
import net.minecraft.class_7853;

public class Auth extends Accounts implements Module {

	@Getter
	private final static Auth Instance = new Auth();
	public final BooleanOption showButton = new BooleanOption("auth.showButton", false);
	public final BooleanOption skinManagerAnimations = new BooleanOption("skins.manage.animations", true);
	private final class_310 client = class_310.method_1551();
	private final GenericOption viewAccounts = new GenericOption("viewAccounts", "clickToOpen", () -> client.method_1507(new AccountsScreen(client.field_1755)));
	private final Set<String> loadingTexture = new HashSet<>();
	private final Map<String, class_2960> textures = new HashMap<>();
	@Getter
	private final SkinManager skinManager = new SkinManager();

	@Override
	public void init() {
		load();
		this.msApi = new MSApi(this, () -> client.field_1690.field_1883);
		if (isContained(client.method_1548().method_1675())) {
			//noinspection DataFlowIssue
			current = getAccounts().stream().filter(account -> account.getUuid()
				.equals(UUIDHelper.toUndashed(client.method_1548().method_44717()))).toList().get(0);
			current.setAuthToken(client.method_1548().method_1674());
			current.setName(client.method_1548().method_1676());
			/*if (current.needsRefresh()) {
				current.refresh(auth).thenRun(this::save);
			}*/
		} else {
			//noinspection DataFlowIssue
			current = new Account(client.method_1548().method_1676(), UUIDHelper.toUndashed(client.method_1548().method_44717()), client.method_1548().method_1674());
		}

		category.add(showButton, viewAccounts, skinManagerAnimations);
		AxolotlClient.config().general.add(category);
	}

	@Override
	protected void login(Account account) {
		if (client.field_1687 != null) {
			return;
		}

		if (account.needsRefresh() && !account.isOffline()) {
			if (account.isExpired()) {
				Notifications.getInstance().addStatus(class_2561.method_43471("auth.notif.title"), class_2561.method_43469("auth.notif.refreshing", account.getName()));
			}
			account.refresh(msApi).thenAccept(a -> {
				if (!a.isExpired()) {
					login(a);
				}
			}).thenRun(this::save);
		} else {
			try {
				API.getInstance().shutdown();
				((MinecraftClientAccessor) client).axolotlclient$setSession(new class_320(account.getName(), UUIDHelper.fromUndashed(account.getUuid()).toString(), account.getAuthToken(),
					Optional.empty(), Optional.empty(),
					class_320.class_321.field_34962));
				UserApiService service;
				if (account.isOffline()) {
					service = UserApiService.OFFLINE;
				} else {
					service = ((MinecraftClientAccessor) class_310.method_1551()).getAuthService().createUserApiService(client.method_1548().method_1674());
				}
				((MinecraftClientAccessor) client).axolotlclient$setUserApiService(service);
				((MinecraftClientAccessor) client).axolotlclient$setSocialInteractionsManager(new class_5520(client, service));
				((MinecraftClientAccessor) client).axolotlclient$setPlayerKeyPairManager(class_7853.method_46532(service, client.method_1548(), client.field_1697.toPath()));
				((MinecraftClientAccessor) client).axolotlclient$setChatReportingContext(class_7574.method_44599(class_7569.method_44586(), service));
				save();
				current = account;
				Notifications.getInstance().addStatus(class_2561.method_43471("auth.notif.title"), class_2561.method_43469("auth.notif.login.successful", current.getName()));
				API.getInstance().startup(account);
			} catch (Exception e) {
				Notifications.getInstance().addStatus(class_2561.method_43471("auth.notif.title"), class_2561.method_43471("auth.notif.login.failed"));
			}
		}
	}

	@Override
	CompletableFuture<Account> showAccountsExpiredScreen(Account account) {
		class_437 current = client.field_1755;
		var fut = new CompletableFuture<Account>();
		client.execute(() -> client.method_1507(new class_410((bl) -> {
			if (bl) {
				msApi.startDeviceAuth().thenRun(() -> fut.complete(account));
			} else {
				fut.cancel(true);
			}
			client.method_1507(current);
		}, class_2561.method_43471("auth"), class_2561.method_43469("auth.accountExpiredNotice", account.getName()))));
		return fut;
	}

	@Override
	void displayDeviceCode(DeviceFlowData data) {
		client.execute(() -> client.method_1507(new DeviceCodeDisplayScreen(client.field_1755, data)));
	}

	private void loadTexture(String uuid) {
		if (!loadingTexture.contains(uuid)) {
			loadingTexture.add(uuid);
			ThreadExecuter.scheduleTask(() -> {

				try {
					UUID uUID = UUIDHelper.fromUndashed(uuid);
					GameProfile gameProfile = new GameProfile(uUID, null);
					gameProfile = client.method_1495().fillProfileProperties(gameProfile, false);

					client.method_1582().method_4652(gameProfile, ((type, id, tex) -> {
						if (type == MinecraftProfileTexture.Type.SKIN) {
							textures.put(uuid, id);
							loadingTexture.remove(uuid);
						}
					}), false);
				} catch (IllegalArgumentException ignored) {
				}
			});
		}
	}

	public class_2960 getSkinTexture(Account account) {
		return getSkinTexture(account.getUuid());
	}

	public class_2960 getSkinTexture(User user) {
		return getSkinTexture(user.getUuid());
	}

	public class_2960 getSkinTexture(String uuid) {
		if (!textures.containsKey(uuid)) {
			loadTexture(uuid);
			return Objects.requireNonNullElseGet(textures.get(uuid),
				() -> class_1068.method_4648(UUIDHelper.fromUndashed(uuid)));
		}
		return textures.get(uuid);
	}
}
