/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.auth.skin;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.*;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import io.github.axolotlclient.AxolotlClientCommon;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.api.SimpleTextInputScreen;
import io.github.axolotlclient.api.util.UUIDHelper;
import io.github.axolotlclient.modules.auth.Account;
import io.github.axolotlclient.modules.auth.Auth;
import io.github.axolotlclient.modules.auth.MSApi;
import io.github.axolotlclient.util.ClientColors;
import io.github.axolotlclient.util.Watcher;
import io.github.axolotlclient.util.notifications.Notifications;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_156;
import net.minecraft.class_1921;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_339;
import net.minecraft.class_364;
import net.minecraft.class_4069;
import net.minecraft.class_410;
import net.minecraft.class_4185;
import net.minecraft.class_4265;
import net.minecraft.class_437;
import net.minecraft.class_5244;
import net.minecraft.class_6379;
import net.minecraft.class_6382;
import net.minecraft.class_7413;
import net.minecraft.class_7842;
import net.minecraft.class_7919;
import net.minecraft.class_8016;
import net.minecraft.class_8023;
import net.minecraft.class_8130;
import net.minecraft.client.gui.*;
import net.minecraft.client.gui.widget.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class SkinManagementScreen extends class_437 {
	private static final Path SKINS_DIR = FabricLoader.getInstance().getGameDir().resolve("skins");
	private static final int LIST_SKIN_WIDTH = 75;
	private static final int LIST_SKIN_HEIGHT = 110;
	private static final class_2561 TEXT_EQUIPPING = class_2561.method_43471("skins.manage.equipping");
	private final class_437 parent;
	private final Account account;
	private MSApi.MCProfile cachedProfile;
	private SkinListWidget skinList;
	private SkinListWidget capesList;
	private boolean capesTab;
	private SkinWidget current;
	private final Watcher skinDirWatcher;
	private final CompletableFuture<MSApi.MCProfile> loadingFuture;

	public SkinManagementScreen(class_437 parent, Account account) {
		super(class_2561.method_43471("skins.manage"));
		this.parent = parent;
		this.account = account;
		skinDirWatcher = Watcher.createSelfTicking(SKINS_DIR, () -> {
			AxolotlClientCommon.getInstance().getLogger().info("Reloading screen as local files changed!");
			loadSkinsList();
		});
		loadingFuture = (account.needsRefresh() ? account.refresh(Auth.getInstance().getMsApi())
			: CompletableFuture.completedFuture(null))
			.thenComposeAsync(unused -> Auth.getInstance().getMsApi().getProfile(account));
	}

	@Override
	protected void method_25426() {
		int headerHeight = 33;
		int contentHeight = field_22790 - headerHeight * 2;

		var titleWidget = new class_7842(0, headerHeight / 2 - field_22793.field_2000 / 2, field_22789, field_22793.field_2000, method_25440(), field_22793);
		method_37063(titleWidget);

		var back = method_37063(class_4185.method_46430(class_5244.field_24339, btn -> method_25419())
			.method_46434(field_22789 / 2 - 75, field_22790 - headerHeight / 2 - 10, 150, 20).method_46431());

		var loadingPlaceholder = new class_339(0, headerHeight, field_22789, contentHeight, class_2561.method_43471("skins.loading")) {
			@Override
			protected void method_48579(class_332 graphics, int mouseX, int mouseY, float delta) {
				int centerX = this.method_46426() + this.method_25368() / 2;
				int centerY = this.method_46427() + this.method_25364() / 2;
				class_2561 text = this.method_25369();
				graphics.method_51439(field_22793, text, centerX - field_22793.method_27525(text) / 2, centerY - 9, -1, false);
				String string = class_7413.method_43449(class_156.method_658());
				graphics.method_51433(field_22793, string, centerX - field_22793.method_1727(string) / 2, centerY + 9, 0xFF808080, false);
			}

			@Override
			protected void method_47399(class_6382 builder) {

			}
		};
		loadingPlaceholder.field_22763 = false;
		method_37063(loadingPlaceholder);
		method_37063(back);

		skinList = new SkinListWidget(field_22787, field_22789 / 2, contentHeight - 24, headerHeight + 24, LIST_SKIN_HEIGHT + 34);
		capesList = new SkinListWidget(field_22787, field_22789 / 2, contentHeight - 24, headerHeight + 24, skinList.getEntryContentsHeight() + 24);
		skinList.method_25333(field_22789 / 2);
		capesList.method_25333(field_22789 / 2);
		var currentHeight = Math.min((field_22789 / 2f) * 120 / 85, contentHeight);
		var currentWidth = currentHeight * 85 / 120;
		current = new SkinWidget((int) currentWidth, (int) currentHeight, null, account);
		current.method_48229((int) (field_22789 / 4f - currentWidth / 2), (int) (field_22790 / 2f - currentHeight / 2));

		if (!capesTab) {
			capesList.visible = capesList.active = false;
		} else {
			skinList.visible = skinList.active = false;
		}
		List<class_339> navBar = new ArrayList<>();
		var skinsTab = class_4185.method_46430(class_2561.method_43471("skins.nav.skins"), btn -> {
			navBar.forEach(w -> {
				if (w != btn) w.field_22763 = true;
			});
			btn.field_22763 = false;
			skinList.visible = skinList.active = true;
			capesList.visible = capesList.active = false;
			capesTab = false;
		}).method_46433(Math.max(field_22789 * 3 / 4 - 102, field_22789 / 2 + 2), headerHeight).method_46432(Math.min(100, field_22789 / 4 - 2)).method_46431();
		navBar.add(skinsTab);
		var capesTab = class_4185.method_46430(class_2561.method_43471("skins.nav.capes"), btn -> {
			navBar.forEach(w -> {
				if (w != btn) w.field_22763 = true;
			});
			btn.field_22763 = false;
			skinList.visible = skinList.active = false;
			capesList.visible = capesList.active = true;
			this.capesTab = true;
		}).method_46433(field_22789 * 3 / 4 + 2, headerHeight).method_46432(Math.min(100, field_22789 / 4 - 2)).method_46431();
		navBar.add(capesTab);
		var importButton = new SpriteButton(class_2561.method_43471("skins.manage.import.local"), btn -> {
			btn.field_22763 = false;
			SkinImportUtil.openImportSkinDialog().thenAccept(this::method_29638).thenRun(() -> btn.field_22763 = true);
		}, new class_2960("axolotlclient", "textures/gui/sprites/folder.png"));
		var downloadButton = new SpriteButton(class_2561.method_43471("skins.manage.import.online"), btn -> {
			btn.field_22763 = false;
			promptForSkinDownload();
		}, new class_2960("axolotlclient", "textures/gui/sprites/download.png"));
		if (field_22789 - (capesTab.method_46426() + capesTab.method_25368()) > 28) {
			importButton.method_46421(field_22789 - importButton.method_25368() - 2);
			downloadButton.method_46421(importButton.method_46426() - downloadButton.method_25368() - 2);
			importButton.method_46419(capesTab.method_46427() + capesTab.method_25364() - 11);
			downloadButton.method_46419(importButton.method_46427());
		} else {
			importButton.method_46421(capesTab.method_46426() + capesTab.method_25368() - 11);
			importButton.method_46419(capesTab.method_46427() - 13);
			downloadButton.method_46421(importButton.method_46426() - 2 - 11);
			downloadButton.method_46419(importButton.method_46427());
		}
		skinsTab.field_22763 = this.capesTab;
		capesTab.field_22763 = !this.capesTab;
		Runnable addWidgets = () -> {
			method_37067();
			method_37063(titleWidget);
			method_37063(current);
			method_37063(skinsTab);
			method_37063(capesTab);
			method_37063(downloadButton);
			method_37063(importButton);
			method_37063(skinList);
			method_37063(capesList);
			method_37063(back);
		};
		if (cachedProfile != null) {
			initDisplay();
			addWidgets.run();
			return;
		}
		loadingFuture.thenAcceptAsync(profile -> {
				cachedProfile = profile;
				initDisplay();
				addWidgets.run();
			}).exceptionally(t -> {
				if (t.getCause() instanceof CancellationException) {
					field_22787.method_1507(parent);
					return null;
				}
				AxolotlClientCommon.getInstance().getLogger().error("Failed to load skins!", t);
				var error = class_2561.method_43471("skins.error.failed_to_load");
				var errorDesc = class_2561.method_43471("skins.error.failed_to_load_desc");
				method_37067();
				method_37063(titleWidget);
				method_37063(new class_7842(field_22789 / 2 - field_22793.method_27525(error) / 2, field_22790 / 2 - field_22793.field_2000 - 2, field_22793.method_27525(error), field_22793.field_2000, error, field_22793));
				method_37063(new class_7842(field_22789 / 2 - field_22793.method_27525(errorDesc) / 2, field_22790 / 2 + 1, field_22793.method_27525(errorDesc), field_22793.field_2000, errorDesc, field_22793));
				method_37063(back);
				return null;
			});
	}

	private void promptForSkinDownload() {
		field_22787.method_1507(new SimpleTextInputScreen(this, class_2561.method_43471("skins.manage.import.online"), class_2561.method_43471("skins.manage.import.online.input"), s ->
			UUIDHelper.ensureUuidOpt(s).thenAccept(o -> {
				if (o.isPresent()) {
					AxolotlClientCommon.getInstance().getLogger().info("Downloading skin of {} ({})", s, o.get());
					Auth.getInstance().getMsApi().getTextures(o.get())
						.exceptionally(th -> {
							AxolotlClientCommon.getInstance().getLogger().info("Failed to download skin of {} ({})", s, o.get(), th);
							return null;
						}).thenAccept(t -> {
							if (t == null) {
								Notifications.getInstance().addStatus("skins.notification.title", "skins.notification.import.online.failed_to_download", s);
								return;
							}
							try {
								var bytes = t.skin().join();
								var out = ensureNonexistent(SKINS_DIR.resolve(t.skinKey()));
								Skin.LocalSkin.writeMetadata(out, Map.of(Skin.LocalSkin.CLASSIC_METADATA_KEY, t.classicModel(), "name", t.name(), "uuid", t.id(), "download_time", Instant.now()));
								Files.write(out, bytes);
								client.execute(this::loadSkinsList);
								Notifications.getInstance().addStatus("skins.notification.title", "skins.notification.import.online.downloaded", t.name());
								AxolotlClientCommon.getInstance().getLogger().info("Downloaded skin of {} ({})", t.name(), o.get());
							} catch (IOException e) {
								AxolotlClientCommon.getInstance().getLogger().warn("Failed to write skin file", e);
								Notifications.getInstance().addStatus("skins.notification.title", "skins.notification.import.online.failed_to_save", t.name());
							}
						});
				} else {
					Notifications.getInstance().addStatus("skins.notification.title", "skins.notification.import.online.not_found", s);
				}
			})));
	}

	@Override
	public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
		method_25420(graphics);
		super.method_25394(graphics, mouseX, mouseY, delta);
	}

	private void initDisplay() {
		loadSkinsList();
		loadCapesList();
	}

	private void refreshCurrentList() {
		if (capesTab) {
			var scroll = capesList.method_25341();
			loadCapesList();
			capesList.method_25307(scroll);
		} else {
			var scroll = skinList.method_25341();
			loadSkinsList();
			skinList.method_25307(scroll);
		}
	}

	private void loadCapesList() {
		List<Row> rows = new ArrayList<>();
		var profile = cachedProfile;
		int columns = Math.max(2, (field_22789 / 2 - 25) / LIST_SKIN_WIDTH);
		var capes = profile.capes();
		var deselectCape = createWidgetForCape(current.getSkin(), null);
		var activeCape = capes.stream().filter(Cape::active).findFirst();
		current.setCape(activeCape.orElse(null));
		deselectCape.noCape(activeCape.isEmpty());
		for (int i = 0; i < capes.size() + 1; i += columns) {
			Entry widget;
			if (i == 0) {
				widget = createEntry(capesList.getEntryContentsHeight(), deselectCape, class_2561.method_43471("skins.capes.no_cape"));
			} else {
				var cape = capes.get(i - 1);
				widget = createEntryForCape(current.getSkin(), cape, capesList.getEntryContentsHeight());
			}
			List<class_339> widgets = new ArrayList<>();
			widgets.add(widget);
			for (int c = 1; c < columns; c++) {
				if (!(i < capes.size() + 1 - c)) continue;
				var cape2 = capes.get(i + c - 1);
				Entry widget2 = createEntryForCape(current.getSkin(), cape2, capesList.getEntryContentsHeight());

				widgets.add(widget2);
			}
			rows.add(new Row(widgets));
		}
		field_22787.execute(() -> capesList.method_25314(rows));
	}

	private void loadSkinsList() {
		var profile = cachedProfile;
		int columns = Math.max(2, (field_22789 / 2 - 25) / LIST_SKIN_WIDTH);
		List<Skin> skins = new ArrayList<>(profile.skins());
		var hashes = skins.stream().map(Asset::sha256).collect(Collectors.toSet());
		var defaultSkin = Skin.getDefaultSkin(account);
		var local = new ArrayList<>(loadLocalSkins());
		var localHashes = local.stream().collect(Collectors.toMap(Asset::sha256, Function.identity(), (skin, skin2) -> skin));
		local.removeIf(s -> !localHashes.containsValue(s));
		skins.replaceAll(s -> {
			if (s instanceof MSApi.MCProfile.OnlineSkin online) {
				if (localHashes.containsKey(s.sha256()) && localHashes.get(s.sha256()) instanceof Skin.LocalSkin file) {
					local.remove(localHashes.remove(s.sha256()));
					return new Skin.Shared(file, online);
				}
			}
			return s;
		});
		skins.addAll(local);
		if (!hashes.contains(defaultSkin.sha256())) {
			skins.add(defaultSkin);
		}
		populateSkinList(skins, columns);
	}

	private List<Skin> loadLocalSkins() {
		try {
			Files.createDirectories(SKINS_DIR);
			try (Stream<Path> skins = Files.list(SKINS_DIR)) {
				return skins.filter(Files::isRegularFile).sorted(Comparator.<Path>comparingLong(p -> {
					try {
						return Files.getLastModifiedTime(p).toMillis();
					} catch (IOException e) {
						return 0L;
					}
				}).reversed()).map(Auth.getInstance().getSkinManager()::read).filter(Objects::nonNull).toList();
			}
		} catch (IOException e) {
			AxolotlClientCommon.getInstance().getLogger().warn("Failed to read skins dir!", e);
		}
		return Collections.emptyList();
	}

	private void populateSkinList(List<? extends Skin> skins, int columns) {
		int entryHeight = skinList.getEntryContentsHeight();
		List<Row> rows = new ArrayList<>();
		for (int i = 0; i < skins.size(); i += columns) {
			var s = skins.get(i);
			if (s != null && s.active()) {
				current.setSkin(s);
			}
			var widget = createEntryForSkin(s, entryHeight);
			List<class_339> widgets = new ArrayList<>();
			widgets.add(widget);
			for (int c = 1; c < columns; c++) {
				if (!(i < skins.size() - c)) continue;
				var s2 = skins.get(i + c);
				if (s2 != null && s2.active()) {
					current.setSkin(s2);
				}
				var widget2 = createEntryForSkin(s2, entryHeight);
				widgets.add(widget2);
			}
			rows.add(new Row(widgets));
		}
		field_22787.execute(() -> skinList.method_25314(rows));
	}

	private Path ensureNonexistent(Path p) {
		if (Files.exists(p)) {
			int counter = 0;
			do {
				counter++;
				p = p.resolveSibling(p.getFileName().toString() + "_" + counter);
			} while (Files.exists(p));
		}
		return p;
	}

	@Override
	public void method_29638(List<Path> packs) {
		if (packs.isEmpty()) return;

		CompletableFuture<?>[] futs = new CompletableFuture[packs.size()];
		for (int i = 0; i < packs.size(); i++) {
			Path p = packs.get(i);
			futs[i] = CompletableFuture.runAsync(() -> {
				try {
					var target = ensureNonexistent(SKINS_DIR.resolve(p.getFileName()));
					var skin = Auth.getInstance().getSkinManager().read(p, false);
					if (skin != null) {
						Files.write(target, skin.image());
					} else {
						AxolotlClientCommon.getInstance().getLogger().info("Skipping dragged file {} because it does not seem to be a valid skin!", p);
						Notifications.getInstance().addStatus("skins.notification.title", "skins.notification.not_copied", p.getFileName());
					}
				} catch (IOException e) {
					AxolotlClientCommon.getInstance().getLogger().warn("Failed to copy skin file: ", e);
				}
			}, field_22787);
		}
		CompletableFuture.allOf(futs).thenRun(this::loadSkinsList);
	}

	private @NotNull Entry createEntryForSkin(Skin skin, int entryHeight) {
		return createEntry(entryHeight, new SkinWidget(LIST_SKIN_WIDTH, LIST_SKIN_HEIGHT, skin, account));
	}

	private @NotNull Entry createEntryForCape(Skin currentSkin, Cape cape, int entryHeight) {
		return createEntry(entryHeight, createWidgetForCape(currentSkin, cape), class_2561.method_43470(cape.alias()));
	}

	private SkinWidget createWidgetForCape(Skin currentSkin, Cape cape) {
		SkinWidget widget2 = new SkinWidget(LIST_SKIN_WIDTH, LIST_SKIN_HEIGHT, currentSkin, cape, account);
		widget2.setRotationY(210);
		return widget2;
	}

	@Override
	protected void method_41843() {
		Auth.getInstance().getSkinManager().releaseAll();
		super.method_41843();
	}

	@Override
	public void method_25432() {
		Auth.getInstance().getSkinManager().releaseAll();
		Watcher.close(skinDirWatcher);
	}

	@Override
	public void method_25419() {
		field_22787.method_1507(parent);
	}

	private SkinListWidget getCurrentList() {
		return capesTab ? capesList : skinList;
	}

	private class SkinListWidget extends class_4265<Row> {
		public boolean active = true, visible = true;

		public SkinListWidget(class_310 minecraft, int width, int height, int y, int entryHeight) {
			super(minecraft, width, SkinManagementScreen.this.field_22790, y, y + height, entryHeight);
			method_25315(false, 0);
		}

		@Override
		protected int method_25329() {
			return field_19087 - 8;
		}

		@Override
		public int method_25342() {
			return field_19088 + 3;
		}

		@Override
		public int method_25322() {
			if (!(method_25331() > 0)) {
				return field_22742 - 4;
			}
			return field_22742 - 14;
		}

		public int getEntryContentsHeight() {
			return field_22741 - 4;
		}

		@Override
		public @Nullable class_8016 method_48205(class_8023 event) {
			if (!active || !visible) return null;
			return super.method_48205(event);
		}

		@Override
		public void method_25314(Collection<Row> newEntries) {
			super.method_25314(newEntries);
		}

		@Override
		public void centerScrollOn(Row entry) {
			super.method_25324(entry);
		}

		@Override
		public boolean method_25401(double mouseX, double mouseY, double amountY) {
			if (!visible) return false;
			return super.method_25401(mouseX, mouseY, amountY);
		}

		@Override
		public boolean method_25405(double mouseX, double mouseY) {
			return active && visible && super.method_25405(mouseX, mouseY);
		}

		@Override
		public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
			if (!visible) return;
			super.method_25394(graphics, mouseX, mouseY, delta);
		}
	}

	private class Row extends class_4265.class_4266<Row> {
		private final List<class_339> widgets;

		public Row(List<class_339> entries) {
			this.widgets = entries;
		}

		@Override
		public @NotNull List<? extends class_6379> method_37025() {
			return widgets;
		}

		@Override
		public void method_25343(class_332 guiGraphics, int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
			int x = left;
			if (widgets.isEmpty()) return;
			int count = widgets.size();
			int padding = ((width - 5 * (count - 1)) / count);
			for (var w : widgets) {
				w.method_48229(x, top);
				w.method_25358(padding);
				w.method_25394(guiGraphics, mouseX, mouseY, partialTick);
				x += w.method_25368() + 5;
			}
		}

		@Override
		public @NotNull List<? extends class_364> method_25396() {
			return widgets;
		}

		@Override
		public void method_25395(@Nullable class_364 focused) {
			super.method_25395(focused);
			if (focused != null) {
				getCurrentList().centerScrollOn(this);
			}
		}
	}

	Entry createEntry(int height, SkinWidget widget) {
		return createEntry(height, widget, null);
	}

	Entry createEntry(int height, SkinWidget widget, class_2561 label) {
		return new Entry(height, widget, label);
	}

	private class Entry extends class_339 implements class_4069 {
		private final SkinWidget skinWidget;
		private final @Nullable class_339 label;
		private final List<class_339> actionButtons = new ArrayList<>();
		private final class_339 equipButton;
		private boolean equipping;
		private long equippingStart;
		@Nullable
		private class_364 focused;
		private boolean dragging;

		public Entry(int height, SkinWidget widget, @Nullable class_2561 label) {
			super(0, 0, widget.method_25368(), height, class_2561.method_43473());
			widget.method_25358(method_25368() - 4);
			var asset = widget.getFocusedAsset();
			if (asset != null) {
				if (asset instanceof Skin skin) {
					var wideSprite = new class_2960("axolotlclient", "textures/gui/sprites/wide.png");
					var slimSprite = new class_2960("axolotlclient", "textures/gui/sprites/slim.png");
					var slimText = class_2561.method_43471("skins.manage.variant.classic");
					var wideText = class_2561.method_43471("skins.manage.variant.slim");
					actionButtons.add(new SpriteButton(skin.classicVariant() ? wideText : slimText, btn -> {
						var self = (SpriteButton) btn;
						skin.classicVariant(!skin.classicVariant());
						self.sprite = skin.classicVariant() ? slimSprite : wideSprite;
						self.method_25355(skin.classicVariant() ? wideText : slimText);
					}, skin.classicVariant() ? slimSprite : wideSprite));
				}
				if (asset instanceof Asset.Local local) {
					this.actionButtons.add(new SpriteButton(class_2561.method_43471("skins.manage.delete"), btn -> {
						btn.field_22763 = false;
						field_22787.method_1507(new class_410(confirmed -> {
							field_22787.method_1507(new LoadingScreen(method_25440(), class_2561.method_43471("menu.working")));
							if (confirmed) {
								try {
									Files.delete(local.file());
									Skin.LocalSkin.deleteMetadata(local.file());
								} catch (IOException e) {
									AxolotlClientCommon.getInstance().getLogger().warn("Failed to delete: ", e);
								}
							}
							field_22787.method_1507(SkinManagementScreen.this);
							btn.field_22763 = true;
						}, class_2561.method_43471("skins.manage.delete.confirm"), (class_2561) (asset.active() ?
							class_2561.method_43471("skins.manage.delete.confirm.desc_active") :
							class_2561.method_43471("skins.manage.delete.confirm.desc")
						).br$color(Colors.RED.toInt())));
					}, new class_2960("axolotlclient", "textures/gui/sprites/delete.png")));
				}
				if (asset instanceof Asset.Online online && online.supportsDownload() && !(asset instanceof Asset.Local)) {
					this.actionButtons.add(new SpriteButton(class_2561.method_43471("skins.manage.download"), btn -> {
						btn.field_22763 = false;
						download(asset).thenRun(() -> {
							refreshCurrentList();
							btn.field_22763 = true;
						});
					}, new class_2960("axolotlclient", "textures/gui/sprites/download.png")));
				}
			}
			if (label != null) {
				this.label = new class_8130(0, 0, widget.method_25368(), 16, label, field_22793) {
					@Override
					protected void method_48579(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
						method_49604(guiGraphics, field_22793, 2, -1);
					}
				};
				this.label.field_22763 = false;
			} else {
				this.label = null;
			}
			this.equipButton = class_4185.method_46430(class_2561.method_43471(
					widget.isEquipped() ? "skins.manage.equipped" : "skins.manage.equip"),
				btn -> {
					equippingStart = class_156.method_658();
					equipping = true;
					btn.method_25355(TEXT_EQUIPPING);
					btn.field_22763 = false;
					Consumer<CompletableFuture<MSApi.MCProfile>> consumer = f -> f.thenAcceptAsync(p -> {
						cachedProfile = p;
						if (field_22787.field_1755 == SkinManagementScreen.this) {
							refreshCurrentList();
						} else {
							field_22787.execute(() -> field_22787.method_1507(SkinManagementScreen.this));
						}
					}).exceptionally(t -> {
						AxolotlClientCommon.getInstance().getLogger().warn("Failed to equip asset!", t);
						equipping = false;
						return null;
					});
					if (asset instanceof Skin && !(current.getSkin() instanceof Skin.Local)) {
						field_22787.method_1507(new class_410(confirmed -> {
							field_22787.method_1507(new LoadingScreen(method_25440(), TEXT_EQUIPPING));
							if (confirmed) {
								consumer.accept(download(current.getSkin()).thenCompose(a -> widget.equip()));
							} else {
								consumer.accept(widget.equip());
							}
						}, class_2561.method_43471("skins.manage.equip.confirm"), class_2561.method_43471("skins.manage.equip.download_current")));
					} else {
						consumer.accept(widget.equip());
					}
				}).method_46432(widget.method_25368()).method_46431();
			this.equipButton.field_22763 = !widget.isEquipped();
			this.skinWidget = widget;
		}

		private @NotNull CompletableFuture<?> download(Asset asset) {
			return CompletableFuture.runAsync(() -> {
				try {
					var out = SKINS_DIR.resolve(asset.sha256());
					Files.createDirectories(out.getParent());
					Files.write(out, asset.image());
					if (asset instanceof Skin skin) {
						Skin.LocalSkin.writeMetadata(out, Map.of(Skin.LocalSkin.CLASSIC_METADATA_KEY, skin.classicVariant()));
					}
				} catch (IOException e) {
					AxolotlClientCommon.getInstance().getLogger().warn("Failed to download: ", e);
				}
			});
		}

		@Override
		public final boolean method_25397() {
			return this.dragging;
		}

		@Override
		public final void method_25398(boolean dragging) {
			this.dragging = dragging;
		}

		@Nullable
		@Override
		public class_364 method_25399() {
			return this.focused;
		}

		@Override
		public void method_25395(@Nullable class_364 child) {
			if (this.focused != null) {
				this.focused.method_25365(false);
			}

			if (child != null) {
				child.method_25365(true);
			}

			this.focused = child;
		}

		@Nullable
		@Override
		public class_8016 method_48205(class_8023 event) {
			return class_4069.super.method_48205(event);
		}

		@Override
		public boolean method_25402(double mouseX, double mouseY, int button) {
			return class_4069.super.method_25402(mouseX, mouseY, button);
		}

		@Override
		public boolean method_25406(double mouseX, double mouseY, int button) {
			return class_4069.super.method_25406(mouseX, mouseY, button);
		}

		@Override
		public boolean method_25403(double mouseX, double mouseY, int button, double deltaX, double deltaY) {
			return class_4069.super.method_25403(mouseX, mouseY, button, deltaX, deltaY);
		}

		@Override
		public boolean method_25370() {
			return class_4069.super.method_25370();
		}

		@Override
		public void method_25365(boolean focused) {
			class_4069.super.method_25365(focused);
		}

		@Override
		public @NotNull List<? extends class_364> method_25396() {
			return Stream.concat(actionButtons.stream(), Stream.of(skinWidget, label, equipButton)).filter(Objects::nonNull).toList();
		}

		private float applyEasing(float x) {
			return x * x * x;
		}

		@Override
		protected void method_48579(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
			int y = method_46427() + 4;
			int x = method_46426() + 2;
			skinWidget.method_48229(x, y);
			skinWidget.method_25358(method_25368() - 4);
			if (skinWidget.isEquipped() || equipping) {
				long prog;
				if (Auth.getInstance().skinManagerAnimations.get()) {
					if (equipping) prog = (class_156.method_658() - equippingStart) / 20 % 100;
					else prog = Math.abs((class_156.method_658() / 30 % 200) - 100);
				} else prog = 100;
				var percent = (prog / 100f);
				float gradientWidth;
				if (equipping) {
					gradientWidth = percent * Math.min(method_25368() / 3f, method_25364() / 3f);
				} else {
					gradientWidth = Math.min(method_25368() / 15f, method_25364() / 6f) + applyEasing(percent) * Math.min(method_25368() * 2 / 15f, method_25364() / 6f);
				}
				GradientHoleRectangleRenderState.render(guiGraphics, method_46426() + 2, method_46427() + 2, method_46426() + method_25368() - 2,
					skinWidget.method_46427() + skinWidget.method_25364() + 2,
					gradientWidth,
					equipping ? 0xFFFF0088 : ClientColors.SELECTOR_GREEN.toInt(), 0);
			}
			skinWidget.method_25394(guiGraphics, mouseX, mouseY, partialTick);
			int actionButtonY = method_46427() + 2;
			for (var button : actionButtons) {
				button.method_48229(skinWidget.method_46426() + skinWidget.method_25368() - button.method_25368(), actionButtonY);
				if (method_49606() || button.method_25367()) {
					button.method_25394(guiGraphics, mouseX, mouseY, partialTick);
				}
				actionButtonY += button.method_25364() + 2;
			}
			if (label != null) {
				label.method_48229(x, skinWidget.method_46427() + skinWidget.method_25364() + 6);
				label.method_25394(guiGraphics, mouseX, mouseY, partialTick);
				label.method_25358(method_25368() - 4);
				equipButton.method_48229(x, label.method_46427() + label.method_25364() + 2);
			} else {
				equipButton.method_48229(x, skinWidget.method_46427() + skinWidget.method_25364() + 4);
			}
			equipButton.method_25358(method_25368() - 4);
			equipButton.method_25394(guiGraphics, mouseX, mouseY, partialTick);

			if (method_49606()) {
				guiGraphics.br$outlineRect(method_46426(), method_46427(), method_25368(), method_25364(), -1);
			}
		}

		@Override
		protected void method_47399(class_6382 narrationElementOutput) {
			skinWidget.method_37020(narrationElementOutput);
			actionButtons.forEach(w -> w.method_37020(narrationElementOutput));
			if (label != null) {
				label.method_37020(narrationElementOutput);
			}
			equipButton.method_37020(narrationElementOutput);
		}

		private static class GradientHoleRectangleRenderState {

			public static void render(class_332 graphics, int x0, int y0, int x1, int y1, float gradientWidth, int col1, int col2) {
				var vertexConsumer = graphics.method_51450().getBuffer(class_1921.method_51784());
				float z = 0;
				//top
				var pose = graphics.method_51448().method_23760().method_23761();
				vertexConsumer.method_22918(pose, x0, y0, z).method_39415(col1).method_1344();
				vertexConsumer.method_22918(pose, x0 + gradientWidth, y0 + gradientWidth, z).method_39415(col2).method_1344();
				vertexConsumer.method_22918(pose, x1 - gradientWidth, y0 + gradientWidth, z).method_39415(col2).method_1344();
				vertexConsumer.method_22918(pose, x1, y0, z).method_39415(col1).method_1344();
				//left
				vertexConsumer.method_22918(pose, x0, y1, z).method_39415(col1).method_1344();
				vertexConsumer.method_22918(pose, x0 + gradientWidth, y1 - gradientWidth, z).method_39415(col2).method_1344();
				vertexConsumer.method_22918(pose, x0 + gradientWidth, y0 + gradientWidth, z).method_39415(col2).method_1344();
				vertexConsumer.method_22918(pose, x0, y0, z).method_39415(col1).method_1344();
				//bottom
				vertexConsumer.method_22918(pose, x1, y1, z).method_39415(col1).method_1344();
				vertexConsumer.method_22918(pose, x1 - gradientWidth, y1 - gradientWidth, z).method_39415(col2).method_1344();
				vertexConsumer.method_22918(pose, x0 + gradientWidth, y1 - gradientWidth, z).method_39415(col2).method_1344();
				vertexConsumer.method_22918(pose, x0, y1, z).method_39415(col1).method_1344();
				//right
				vertexConsumer.method_22918(pose, x1, y0, z).method_39415(col1).method_1344();
				vertexConsumer.method_22918(pose, x1 - gradientWidth, y0 + gradientWidth, z).method_39415(col2).method_1344();
				vertexConsumer.method_22918(pose, x1 - gradientWidth, y1 - gradientWidth, z).method_39415(col2).method_1344();
				vertexConsumer.method_22918(pose, x1, y1, z).method_39415(col1).method_1344();
			}
		}

	}

	private static class SpriteButton extends class_4185 {
		private class_2960 sprite;

		public SpriteButton(class_2561 message, class_4241 onPress, class_2960 sprite) {
			super(0, 0, 11, 11, message, onPress, field_40754);
			this.sprite = sprite;
			method_47400(class_7919.method_47408(message, class_2561.method_43473()));
		}

		@Override
		public void method_25355(class_2561 message) {
			super.method_25355(message);
			method_47400(class_7919.method_47408(message, class_2561.method_43473()));
		}

		@Override
		protected void method_48579(class_332 graphics, int mouseX, int mouseY, float delta) {
			super.method_48579(graphics, mouseX, mouseY, delta);
			graphics.method_25290(sprite, method_46426() + 2, method_46427() + 2, 0, 0, 7, 7, 7, 7);
		}

		@Override
		public void method_48589(class_332 graphics, class_327 renderer, int color) {

		}
	}
}
