/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.hud;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.modules.hud.gui.entry.TextHudEntry;
import io.github.axolotlclient.modules.hud.util.DrawPosition;
import lombok.Getter;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3262;
import net.minecraft.class_332;
import net.minecraft.class_7367;

import static io.github.axolotlclient.modules.hud.util.DrawUtil.*;

public class PackDisplayHud extends TextHudEntry {

	public static final class_2960 ID = new class_2960("axolotlclient", "packdisplayhud");
	public final List<PackWidget> widgets = new ArrayList<>();
	private final BooleanOption iconsOnly = new BooleanOption("iconsonly", false);
	private PackWidget placeholder;

	public PackDisplayHud() {
		super(200, 50, true);
	}

	@Override
	public void renderComponent(AxoRenderContext graphics, float f) {
		DrawPosition pos = getPos();

		if (widgets.isEmpty())
			init();

		if (background.get()) {
			fillRect((class_332) graphics, getBounds(), backgroundColor.get());
		}

		if (outline.get())
			outlineRect((class_332) graphics, getBounds(), outlineColor.get());

		int y = pos.y() + 1;
		for (int i = widgets.size() - 1; i >= 0; i--) { // Badly reverse the order (I'm sure there are better ways to do this)
			widgets.get(i).render((class_332) graphics, pos.x() + 1, y);
			y += 18;
		}
		if (y - pos.y() + 1 != getHeight()) {
			setHeight(y - pos.y() - 1);
			onBoundsUpdate();
		}
	}

	@Override
	public void init() {
		int listSize = class_310.method_1551().method_1520().method_14441().size();
		class_310.method_1551().method_1520().method_14444().forEach(profile -> {
			try (class_3262 pack = profile.method_14458()) {

				if (listSize == 1) {
					widgets.add(createWidget(profile.method_14457(), pack));
				} else if (!pack.method_14409().equalsIgnoreCase("vanilla")) {
					widgets.add(createWidget(profile.method_14457(), pack));
				}

			} catch (Exception ignored) {
			}
		});

		AtomicInteger w = new AtomicInteger(20);
		widgets.forEach(packWidget -> {
			int textW = class_310.method_1551().field_1772.method_1727(packWidget.getName()) + 20;
			if (textW > w.get())
				w.set(textW);
		});
		setWidth(w.get());

		setHeight(widgets.size() * 18);
		onBoundsUpdate();
	}

	private PackWidget createWidget(class_2561 displayName, class_3262 pack) throws IOException, AssertionError {
		class_7367<InputStream> supplier = pack.method_14410("pack.png");
		assert supplier != null;
		InputStream stream = supplier.get();
		if (stream != null) {
			class_2960 id = class_310.method_1551().method_1531().method_4617(ID.method_12832(), new class_1043(class_1011.method_4309(stream)));
			stream.close();
			return new PackWidget(displayName, id);
		}
		return null;
	}

	@Override
	public void renderPlaceholderComponent(AxoRenderContext graphics, float f) {
		boolean updateBounds = false;
		if (getHeight() < 18) {
			setHeight(18);
			updateBounds = true;
		}
		if (getWidth() < 56) {
			setWidth(56);
			updateBounds = true;
		}
		if (updateBounds) {
			onBoundsUpdate();
		}
		if (placeholder == null) {
			try (class_3262 defaultPack = class_310.method_1551().method_45573()) {
				placeholder = createWidget(class_2561.method_30163(defaultPack.method_14409()), defaultPack);
			} catch (Exception ignored) {
			}
		} else {
			placeholder.render((class_332) graphics, getPos().x() + 1, getPos().y() + 1);
		}
	}

	@Override
	public List<Option<?>> getConfigurationOptions() {
		List<Option<?>> options = super.getConfigurationOptions();
		options.add(iconsOnly);
		return options;
	}

	@Override
	public class_2960 getId() {
		return ID;
	}

	public void update() {
		widgets.clear();
		init();
	}

	public class PackWidget {

		@Getter
		public final String name;
		private final class_2960 texture;

		public PackWidget(class_2561 name, class_2960 id) {
			this.name = name.getString();
			texture = id;
		}

		public void render(class_332 graphics, int x, int y) {
			if (!iconsOnly.get()) {
				RenderSystem.setShaderColor(1, 1, 1, 1F);
				graphics.method_25290(texture, x, y, 0, 0, 16, 16, 16, 16);
			}
			drawString(graphics, name, x + 18, y + 6, textColor.get().toInt(), shadow.get());
		}
	}
}
