/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.hud.vanilla;

import java.util.List;

import com.mojang.blaze3d.platform.GlStateManager;
import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.ColorOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.EnumOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.GraphicsOption;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.modules.hud.gui.component.DynamicallyPositionable;
import io.github.axolotlclient.modules.hud.gui.entry.AbstractHudEntry;
import io.github.axolotlclient.modules.hud.gui.layout.AnchorPoint;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.modules.hud.util.RenderUtil;
import io.github.axolotlclient.util.ClientColors;
import io.github.axolotlclient.util.Util;
import lombok.AllArgsConstructor;
import net.minecraft.class_1309;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_239;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3965;
import net.minecraft.class_4061;
import net.minecraft.class_4184;
import net.minecraft.class_4587;
import net.minecraft.class_4739;
import net.minecraft.class_757;
import net.minecraft.class_7833;

import static io.github.axolotlclient.modules.hud.gui.hud.vanilla.HotbarHUD.ICONS_TEXTURE;

/**
 * This implementation of Hud modules is based on KronHUD.
 * <a href="https://github.com/DarkKronicle/KronHUD">Github Link.</a>
 *
 * <p>License: GPL-3.0</p>
 */

public class CrosshairHud extends AbstractHudEntry implements DynamicallyPositionable {
	public static final class_2960 ID = new class_2960("kronhud", "crosshairhud");
	private final EnumOption<Crosshair> type = new EnumOption<>("crosshair_type", Crosshair.class, Crosshair.CROSS);
	private final BooleanOption showInF5 = new BooleanOption("showInF5", false);
	private final ColorOption defaultColor = new ColorOption("defaultcolor", ClientColors.WHITE);
	private final ColorOption entityColor = new ColorOption("entitycolor", ClientColors.SELECTOR_RED);
	private final ColorOption containerColor = new ColorOption("blockcolor", ClientColors.SELECTOR_BLUE);
	private final ColorOption attackIndicatorBackgroundColor = new ColorOption("attackindicatorbg",
		new Color(0xFF141414));
	private final ColorOption attackIndicatorForegroundColor = new ColorOption("attackindicatorfg", ClientColors.WHITE);
	private final BooleanOption applyBlend = new BooleanOption("applyBlend", true);
	private final BooleanOption overrideF3 = new BooleanOption("overrideF3", false);
	private final BooleanOption customAttackIndicator = new BooleanOption("crosshairhud.custom_attack_indicator", false);

	private final GraphicsOption customTextureGraphics = new GraphicsOption("customTextureGraphics",
		new int[][]{
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
			new int[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
		});
	private final class_310 client = (class_310) super.client;

	public CrosshairHud() {
		super(15, 15);
	}

	@Override
	public boolean movable() {
		return false;
	}

	@Override
	public class_2960 getId() {
		return ID;
	}

	@Override
	public List<Option<?>> getConfigurationOptions() {
		List<Option<?>> options = super.getConfigurationOptions();
		options.add(hide);
		options.add(type);
		options.add(customTextureGraphics);
		options.add(showInF5);
		options.add(overrideF3);
		options.add(applyBlend);
		options.add(defaultColor);
		options.add(entityColor);
		options.add(containerColor);
		options.add(customAttackIndicator);
		options.add(attackIndicatorBackgroundColor);
		options.add(attackIndicatorForegroundColor);
		return options;
	}

	@Override
	public boolean overridesF3() {
		return overrideF3.get();
	}

	@Override
	public double getDefaultX() {
		return 0.5;
	}

	@Override
	public double getDefaultY() {
		return 0.5;
	}

	@Override
	public void render(AxoRenderContext context, float delta) {
		if (!client.field_1690.method_31044().method_31034() && !showInF5.get()) {
			return;
		}

		RenderSystem.setShaderColor(1, 1, 1, 1);

		context.br$pushMatrix();
		scale(context);
		final var graphics = (class_332) context;

		int x = getPos().x();
		int y = getPos().y() + 1;
		Color color = getColor();
		class_4061 indicator = this.client.field_1690.method_42565().method_41753();

		RenderSystem.enableBlend();

		// Need to not enable blend while the debug HUD is open because it does weird stuff. Why? no idea.
		if (color == defaultColor.get() && !type.get().equals(Crosshair.DIRECTION) && applyBlend.get()
			&& !client.field_1690.field_1866) {
			RenderSystem.blendFuncSeparate(GlStateManager.class_4535.ONE_MINUS_DST_COLOR,
				GlStateManager.class_4534.ONE_MINUS_SRC_COLOR, GlStateManager.class_4535.ONE,
				GlStateManager.class_4534.ZERO);
		} else {
			RenderSystem.disableBlend();
		}

		if (type.get().equals(Crosshair.DOT)) {
			DrawUtil.fillRect(graphics, x + (getWidth() / 2) - 2, y + (getHeight() / 2) - 2, 3, 3, color.toInt());
		} else if (type.get().equals(Crosshair.CROSS)) {
			RenderUtil.fillBlend(graphics, x + (getWidth() / 2) - 6, y + (getHeight() / 2) - 1, 6, 1, color);
			RenderUtil.fillBlend(graphics, x + (getWidth() / 2), y + (getHeight() / 2) - 1, 5, 1, color);
			RenderUtil.fillBlend(graphics, x + (getWidth() / 2) - 1, y + (getHeight() / 2) - 6, 1, 5, color);
			RenderUtil.fillBlend(graphics, x + (getWidth() / 2) - 1, y + (getHeight() / 2), 1, 5, color);
		} else if (type.get().equals(Crosshair.DIRECTION)) {
			class_4184 camera = this.client.field_1773.method_19418();
			class_4587 matrixStack = RenderSystem.getModelViewStack();
			matrixStack.method_22903();
			matrixStack.method_46416(client.method_22683().method_4486() / 2F, client.method_22683().method_4502() / 2F,
				0);
			matrixStack.method_22907(class_7833.field_40713.rotationDegrees(camera.method_19329()));
			matrixStack.method_22907(class_7833.field_40716.rotationDegrees(camera.method_19330()));
			matrixStack.method_22905(-getScale(), -getScale(), -getScale());
			RenderSystem.applyModelViewMatrix();
			RenderSystem.renderCrosshair(10);
			matrixStack.method_22909();
			RenderSystem.applyModelViewMatrix();
		} else if (type.get().equals(Crosshair.TEXTURE) || type.get().equals(Crosshair.CUSTOM)) {
			RenderSystem.setShaderColor(1.0F, 1.0F, 1.0F, 1.0F);
			if (type.get().equals(Crosshair.TEXTURE)) {
				RenderSystem.setShader(class_757::method_34542);
				// Draw crosshair
				RenderSystem.setShaderColor((float) color.getRed() / 255, (float) color.getGreen() / 255,
					(float) color.getBlue() / 255, (float) color.getAlpha() / 255);
				graphics.method_25302(ICONS_TEXTURE,
					(int) (((client.method_22683().method_4486() / getScale()) - 15) / 2),
					(int) (((client.method_22683().method_4502() / getScale()) - 15) / 2), 0, 0, 15, 15);
			} else {
				// Draw crosshair
				RenderSystem.setShaderColor((float) color.getRed() / 255, (float) color.getGreen() / 255,
					(float) color.getBlue() / 255, (float) color.getAlpha() / 255);

				graphics.method_25290(Util.getTexture(customTextureGraphics), (int) (((client.method_22683().method_4486() / getScale()) - 15) / 2),
					(int) (((client.method_22683().method_4502() / getScale()) - 15) / 2), 0, 0, 15, 15, 15, 15);

				RenderSystem.setShaderTexture(0, ICONS_TEXTURE);
			}

			RenderSystem.setShaderColor(1, 1, 1, 1);

			// Draw attack indicator
			if (!customAttackIndicator.get() && indicator == class_4061.field_18152) {
				//noinspection DataFlowIssue
				float progress = this.client.field_1724.method_7261(0.0F);

				// Whether a cross should be displayed under the indicator
				boolean targetingEntity = false;
				if (this.client.field_1692 != null && this.client.field_1692 instanceof class_1309
					&& progress >= 1.0F) {
					targetingEntity = this.client.field_1724.method_7279() > 5.0F;
					targetingEntity &= this.client.field_1692.method_5805();
				}

				x = (int) ((client.method_22683().method_4486() / getScale()) / 2 - 8);
				y = (int) ((client.method_22683().method_4502() / getScale()) / 2 - 7 + 16);

				if (targetingEntity) {
					graphics.method_25302(ICONS_TEXTURE, x, y, 68, 94, 16, 16);
				} else if (progress < 1.0F) {
					int k = (int) (progress * 17.0F);
					graphics.method_25302(ICONS_TEXTURE, x, y, 36, 94, 16, 4);
					graphics.method_25290(ICONS_TEXTURE, 16, 4, 0, 0, x, y, k, 4);
				}
			}
		}
		if (((type.get().equals(Crosshair.TEXTURE) || type.get().equals(Crosshair.CUSTOM)) ? customAttackIndicator.get() : true) && indicator == class_4061.field_18152) {
			//noinspection DataFlowIssue
			float progress = this.client.field_1724.method_7261(0.0F);
			if (progress != 1.0F) {
				RenderUtil.drawRectangle(graphics, getRawX() + (getWidth() / 2) - 6, getRawY() + (getHeight() / 2) + 9,
					11, 1, attackIndicatorBackgroundColor.get());
				RenderUtil.drawRectangle(graphics, getRawX() + (getWidth() / 2) - 6, getRawY() + (getHeight() / 2) + 9,
					(int) (progress * 11), 1, attackIndicatorForegroundColor.get());
			}
		}
		RenderSystem.disableBlend();
		RenderSystem.defaultBlendFunc();
		context.br$popMatrix();
	}

	public Color getColor() {
		class_239 hit = client.field_1765;
		if (hit == null || hit.method_17783() == null) {
			return defaultColor.get();
		} else if (hit.method_17783() == class_239.class_240.field_1331) {
			return entityColor.get();
		} else if (hit.method_17783() == class_239.class_240.field_1332) {
			class_2338 blockPos = ((class_3965) hit).method_17777();
			class_1937 world = this.client.field_1687;
			//noinspection DataFlowIssue
			if (world.method_8320(blockPos).method_26196(world, blockPos) != null
				|| world.method_8320(blockPos).method_26204() instanceof class_4739<?>) {
				return containerColor.get();
			}
		}
		return defaultColor.get();
	}

	@Override
	public void renderPlaceholder(AxoRenderContext graphics, float delta) {
		// Shouldn't need this...
	}

	@Override
	public AnchorPoint getAnchor() {
		return AnchorPoint.MIDDLE_MIDDLE;
	}

	@AllArgsConstructor
	public enum Crosshair {
		CROSS, DOT, DIRECTION, TEXTURE, CUSTOM
	}
}
