/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.screenshotUtils;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.BooleanSupplier;

import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.api.options.OptionCategory;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.StringArrayOption;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.modules.AbstractModule;
import io.github.axolotlclient.util.options.GenericOption;
import lombok.AllArgsConstructor;
import lombok.Getter;
import net.minecraft.class_124;
import net.minecraft.class_156;
import net.minecraft.class_2558;
import net.minecraft.class_2561;
import net.minecraft.class_2568;
import net.minecraft.class_2583;
import net.minecraft.class_310;
import net.minecraft.class_5250;
import net.minecraft.text.*;
import org.jetbrains.annotations.Nullable;

public class ScreenshotUtils extends AbstractModule {

	@Getter
	private static final ScreenshotUtils Instance = new ScreenshotUtils();
	private final OptionCategory category = OptionCategory.create("screenshotUtils");
	private final BooleanOption enabled = new BooleanOption("enabled", false);
	private final Map<BooleanSupplier, Action> actions = class_156.method_656(() -> {
		Map<BooleanSupplier, Action> actions = new LinkedHashMap<>();
		actions.put(() -> true, new Action("copyAction", class_124.field_1075,
			"copy_image",
			ScreenshotCopying::copy));

		actions.put(() -> true, new Action("deleteAction", class_124.field_1076,
			"delete_image",
			(file) -> {
				try {
					Files.delete(file);
					io.github.axolotlclient.util.Util.addMessageToChatHud(
						Text.literal(I18n.translate("screenshot_deleted").replace("<name>", file.getFileName().toString())));
				} catch (Exception e) {
					AxolotlClient.LOGGER.warn("Couldn't delete Screenshot " + file.getFileName().toString());
				}
			}));

		actions.put(() -> true, new Action("openAction", class_124.field_1068,
			"open_image",
			(file) -> Util.getOperatingSystem().open(file.toUri())));

		actions.put(() -> true, new Action("viewInGalleryAction", class_124.field_1076, "view_in_gallery",
			file -> {
				try {
					ImageInstance instance = new ImageInstance.LocalImpl(file);
					MinecraftClient.getInstance().execute(() -> MinecraftClient.getInstance().setScreen(ImageScreen.create(null, CompletableFuture.completedFuture(instance), true)));
				} catch (Exception ignored) {
					io.github.axolotlclient.util.Util.addMessageToChatHud(Text.translatable("screenshot.gallery.view.error"));
				}
			}));

		actions.put(() -> API.getInstance().isAuthenticated(), new Action("uploadAction", class_124.field_1075,
			"upload_image",
			ImageShare.getInstance()::uploadImage));

		return actions;
	});

	private final StringArrayOption autoExec = new StringArrayOption("autoExec", class_156.method_656(() -> {
		List<String> names = new ArrayList<>();
		names.add("off");
		actions.forEach((condition, action) -> names.add(action.getName()));
		return names.toArray(new String[0]);
	}), "off");

	@Override
	public void init() {
		category.add(enabled, autoExec, new GenericOption("imageViewer", "openViewer", () -> {
			class_310.method_1551().method_1507(new GalleryScreen(class_310.method_1551().field_1755));
		}));

		AxolotlClient.config().general.add(category);
	}

	public class_5250 onScreenshotTaken(class_5250 text, File shot) {
		if (enabled.get()) {
			class_5250 t = getUtilsText(shot.toPath());
			if (t != null) {
				return text.method_27693("\n").method_10852(t);
			}
		}
		return text;
	}

	private @Nullable class_5250 getUtilsText(Path file) {
		if (!autoExec.get().equals("off")) {
			actions.forEach((condition, action) -> {
				if (condition.getAsBoolean() && autoExec.get().equals(action.getName())) {
					action.getClickEvent(file).doAction();
				}
			});
			return null;
		}

		class_5250 message = class_2561.method_43473().method_27661();
		actions.forEach((condition, action) -> {
			if (condition.getAsBoolean()) {
				message.method_10852(action.getText(file)).method_27693(" ");
			}
		});
		return message;
	}

	public interface OnActionCall {

		void doAction(Path file);
	}

	@AllArgsConstructor
	public static class Action {

		private final String translationKey;
		private final class_124 formatting;
		private final String hoverTextKey;
		private final OnActionCall clickEvent;

		public class_2561 getText(Path file) {
			return class_2561.method_43471(translationKey).method_10862(class_2583.field_24360.method_27706(formatting)
				.method_10958(getClickEvent(file)).method_10949(new class_2568(class_2568.class_5247.field_24342, class_2561.method_43471(hoverTextKey))));
		}

		public String getName() {
			return translationKey;
		}

		public CustomClickEvent getClickEvent(Path file) {
			return new CustomClickEvent(clickEvent, file);
		}
	}

	public static class CustomClickEvent extends class_2558 {

		private final OnActionCall action;
		private final Path file;

		public CustomClickEvent(OnActionCall action, Path file) {
			super(class_2559.field_11746, "");
			this.action = action;
			this.file = file;
		}

		public void doAction() {
			if (file != null) {
				action.doAction(file);
			} else {
				AxolotlClient.LOGGER.warn("How'd you manage to do this? "
					+ "Now there's a screenshot ClickEvent without a File attached to it!");
			}
		}
	}
}
