/*
 * Decompiled with CFR 0.152.
 */
package com.moud.client.runtime;

import com.moud.client.api.service.ClientAPIService;
import com.moud.client.network.ClientPacketWrapper;
import com.moud.network.MoudPackets;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.graalvm.polyglot.Context;
import org.graalvm.polyglot.HostAccess;
import org.graalvm.polyglot.PolyglotException;
import org.graalvm.polyglot.Value;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClientScriptingRuntime {
    private static final Logger LOGGER = LoggerFactory.getLogger(ClientScriptingRuntime.class);
    private static final Queue<Runnable> scriptTaskQueue = new ConcurrentLinkedQueue<Runnable>();
    private Context graalContext;
    private final ExecutorService scriptExecutor;
    private final ScheduledExecutorService timerExecutor;
    private final ClientAPIService apiService;
    private final AtomicBoolean initialized = new AtomicBoolean(false);
    private final AtomicBoolean shutdown = new AtomicBoolean(false);
    private final AtomicLong timerIdCounter = new AtomicLong(0L);
    private final ConcurrentHashMap<Long, ScheduledFuture<?>> activeTimers = new ConcurrentHashMap();
    private final List<Value> worldLoadCallbacks = new CopyOnWriteArrayList<Value>();
    private final long startTime = System.nanoTime();

    public ClientScriptingRuntime(ClientAPIService apiService) {
        this.apiService = apiService;
        this.scriptExecutor = Executors.newSingleThreadExecutor(r -> new Thread(r, "ClientScript-Executor"));
        this.timerExecutor = Executors.newScheduledThreadPool(2, r -> new Thread(r, "ClientScript-Timer"));
    }

    public static void scheduleScriptTask(Runnable task) {
        scriptTaskQueue.offer(task);
    }

    public CompletableFuture<Void> initialize() {
        if (this.initialized.get()) {
            return CompletableFuture.completedFuture(null);
        }
        return CompletableFuture.runAsync(() -> {
            try {
                LOGGER.info("Initializing client scripting runtime...");
                HostAccess hostAccess = HostAccess.newBuilder().allowAccessAnnotatedBy(HostAccess.Export.class).allowAllImplementations(true).allowAllClassImplementations(true).allowArrayAccess(true).allowListAccess(true).allowMapAccess(true).build();
                this.graalContext = Context.newBuilder((String[])new String[]{"js"}).allowHostAccess(hostAccess).allowIO(true).build();
                this.bindGlobalAPIs();
                this.bindTimerFunctionsToContext();
                this.bindAnimationFrameAPI();
                this.bindPerformanceAPI();
                this.initialized.set(true);
                LOGGER.info("Client scripting runtime initialized successfully.");
            }
            catch (Exception e) {
                LOGGER.error("Failed to initialize client scripting runtime", (Throwable)e);
                throw new RuntimeException(e);
            }
        }, this.scriptExecutor);
    }

    private void bindGlobalAPIs() {
        Value bindings = this.graalContext.getBindings("js");
        Value moudObj = this.graalContext.eval("js", (CharSequence)"({})");
        if (this.apiService.network != null) {
            moudObj.putMember("network", (Object)this.apiService.network);
        }
        if (this.apiService.rendering != null) {
            moudObj.putMember("rendering", (Object)this.apiService.rendering);
        }
        if (this.apiService.ui != null) {
            moudObj.putMember("ui", (Object)this.apiService.ui);
        }
        if (this.apiService.camera != null) {
            moudObj.putMember("camera", (Object)this.apiService.camera);
        }
        if (this.apiService.cursor != null) {
            moudObj.putMember("cursor", (Object)this.apiService.cursor);
        }
        if (this.apiService.lighting != null) {
            moudObj.putMember("lighting", (Object)this.apiService.lighting);
        }
        if (this.apiService.shared != null) {
            moudObj.putMember("shared", (Object)this.apiService.shared);
        }
        if (this.apiService.events != null) {
            moudObj.putMember("events", (Object)this.apiService.events);
        }
        bindings.putMember("Moud", (Object)moudObj);
        bindings.putMember("console", (Object)this.apiService.console);
    }

    public void triggerWorldLoad() {
        if (!this.initialized.get() || this.shutdown.get()) {
            return;
        }
        this.scriptExecutor.execute(() -> {
            if (this.graalContext == null || this.shutdown.get()) {
                return;
            }
            try {
                this.graalContext.enter();
                for (Value callback : this.worldLoadCallbacks) {
                    try {
                        if (!callback.canExecute()) continue;
                        callback.execute(new Object[0]);
                    }
                    catch (Exception e) {
                        LOGGER.error("Error executing world load callback", (Throwable)e);
                    }
                }
                this.worldLoadCallbacks.clear();
            }
            finally {
                this.graalContext.leave();
            }
        });
    }

    public void updateMoudBindings() {
        if (this.graalContext == null) {
            return;
        }
        this.scriptExecutor.execute(() -> {
            try {
                this.graalContext.enter();
                Value moudObj = this.graalContext.getBindings("js").getMember("Moud");
                if (this.apiService.input != null && !moudObj.hasMember("input")) {
                    moudObj.putMember("input", (Object)this.apiService.input);
                    LOGGER.info("Input service bound to Moud object");
                }
            }
            catch (Exception e) {
                LOGGER.error("Failed to update Moud bindings", (Throwable)e);
            }
            finally {
                this.graalContext.leave();
            }
        });
    }

    private void bindAnimationFrameAPI() {
        this.graalContext.getBindings("js").putMember("requestAnimationFrame", args -> {
            if (args.length < 1 || !args[0].canExecute()) {
                LOGGER.warn("Invalid arguments for requestAnimationFrame. Expected (function).");
                return null;
            }
            Value callback = args[0];
            return this.apiService.rendering.requestAnimationFrame(callback);
        });
        this.graalContext.getBindings("js").putMember("cancelAnimationFrame", args -> {
            if (args.length < 1 || !args[0].isString()) {
                return null;
            }
            this.apiService.rendering.cancelAnimationFrame(args[0].asString());
            return null;
        });
    }

    private void bindPerformanceAPI() {
        Value bindings = this.graalContext.getBindings("js");
        Value performanceObj = this.graalContext.eval("js", (CharSequence)"({})");
        performanceObj.putMember("now", args -> (double)(System.nanoTime() - this.startTime) / 1000000.0);
        bindings.putMember("performance", (Object)performanceObj);
    }

    private void bindTimerFunctionsToContext() {
        this.graalContext.getBindings("js").putMember("setTimeout", args -> {
            if (args.length < 2 || !args[0].canExecute() || !args[1].isNumber()) {
                LOGGER.warn("Invalid arguments for setTimeout. Expected (function, number).");
                return -1L;
            }
            Value callback = args[0];
            long delay = args[1].asLong();
            long id = this.timerIdCounter.incrementAndGet();
            ScheduledFuture<?> future = this.timerExecutor.schedule(() -> this.executeCallback(callback, id, false), delay, TimeUnit.MILLISECONDS);
            this.activeTimers.put(id, future);
            return id;
        });
        this.graalContext.getBindings("js").putMember("clearTimeout", args -> {
            if (args.length < 1 || !args[0].isNumber()) {
                return null;
            }
            this.cancelTimer(args[0].asLong());
            return null;
        });
        this.graalContext.getBindings("js").putMember("setInterval", args -> {
            if (args.length < 2 || !args[0].canExecute() || !args[1].isNumber()) {
                LOGGER.warn("Invalid arguments for setInterval. Expected (function, number).");
                return -1L;
            }
            Value callback = args[0];
            long delay = args[1].asLong();
            if (delay <= 0L) {
                LOGGER.warn("setInterval delay must be greater than 0.");
                return -1L;
            }
            long id = this.timerIdCounter.incrementAndGet();
            ScheduledFuture<?> future = this.timerExecutor.scheduleAtFixedRate(() -> this.executeCallback(callback, id, true), delay, delay, TimeUnit.MILLISECONDS);
            this.activeTimers.put(id, future);
            return id;
        });
        this.graalContext.getBindings("js").putMember("clearInterval", args -> {
            if (args.length < 1 || !args[0].isNumber()) {
                return null;
            }
            this.cancelTimer(args[0].asLong());
            return null;
        });
    }

    private void executeCallback(Value callback, long timerId, boolean isInterval) {
        if (this.graalContext == null || this.shutdown.get()) {
            return;
        }
        ClientScriptingRuntime.scheduleScriptTask(() -> {
            if (this.graalContext == null || this.shutdown.get()) {
                return;
            }
            try {
                this.graalContext.enter();
                if (callback.canExecute()) {
                    callback.execute(new Object[0]);
                }
            }
            catch (Exception e) {
                this.handleScriptException(e, "timer callback");
            }
            finally {
                if (this.graalContext != null) {
                    this.graalContext.leave();
                }
            }
            if (!isInterval) {
                this.activeTimers.remove(timerId);
            }
        });
    }

    private void cancelTimer(long id) {
        ScheduledFuture<?> future = this.activeTimers.remove(id);
        if (future != null) {
            future.cancel(false);
        }
    }

    public CompletableFuture<Void> loadScripts(Map<String, byte[]> scriptsData) {
        if (!this.initialized.get()) {
            LOGGER.error("Cannot load scripts - runtime not initialized");
            return CompletableFuture.failedFuture(new IllegalStateException("Runtime not initialized"));
        }
        return CompletableFuture.runAsync(() -> {
            try {
                this.loadScriptsInternal(scriptsData);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }, this.scriptExecutor);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void loadScriptsInternal(Map<String, byte[]> scriptsData) throws Exception {
        LOGGER.info("Loading {} client scripts", (Object)scriptsData.size());
        this.graalContext.enter();
        try {
            for (Map.Entry<String, byte[]> entry : scriptsData.entrySet()) {
                String scriptName = entry.getKey();
                String scriptContent = new String(entry.getValue());
                try {
                    LOGGER.info("Executing client script: {}", (Object)scriptName);
                    this.graalContext.eval("js", (CharSequence)scriptContent);
                    LOGGER.info("Successfully executed script: {}", (Object)scriptName);
                    LOGGER.info("Client scripts loaded, sending ready signal to server");
                    ClientPacketWrapper.sendToServer(new MoudPackets.ClientReadyPacket());
                }
                catch (PolyglotException e) {
                    this.handleScriptException(e, scriptName);
                    throw new RuntimeException(e);
                    return;
                }
            }
        }
        finally {
            this.graalContext.leave();
        }
    }

    public void processScriptQueue() {
        if (!this.initialized.get() || !this.apiService.isContextValid()) {
            return;
        }
        this.scriptExecutor.execute(() -> {
            if (this.apiService.rendering != null) {
                double timestamp = (double)System.nanoTime() / 1000000.0;
                this.apiService.rendering.processAnimationFrames(timestamp);
            }
        });
    }

    public void triggerNetworkEvent(String eventName, String eventData) {
        if (!this.initialized.get() || this.shutdown.get()) {
            return;
        }
        this.scriptExecutor.execute(() -> {
            if (this.graalContext == null || this.shutdown.get()) {
                return;
            }
            try {
                this.graalContext.enter();
                try {
                    this.apiService.network.triggerEvent(eventName, eventData);
                }
                finally {
                    this.graalContext.leave();
                }
            }
            catch (Exception e) {
                this.handleScriptException(e, "network event: " + eventName);
            }
        });
    }

    private void handleScriptException(Throwable e, String context) {
        if (e instanceof PolyglotException) {
            PolyglotException polyglotException = (PolyglotException)e;
            if (polyglotException.isGuestException()) {
                LOGGER.error("Script error in {}: {}", (Object)context, (Object)polyglotException.getMessage());
                LOGGER.debug("Stack trace: ", (Throwable)polyglotException);
            } else {
                LOGGER.error("Host error during script execution in {}: {}", (Object)context, (Object)polyglotException.getMessage());
            }
        } else {
            LOGGER.error("Unexpected error in script context '{}': {}", new Object[]{context, e.getMessage(), e});
        }
    }

    public boolean isInitialized() {
        return this.initialized.get() && !this.shutdown.get();
    }

    public Context getContext() {
        return this.graalContext;
    }

    public ExecutorService getExecutor() {
        return this.scriptExecutor;
    }

    public void shutdown() {
        if (!this.shutdown.compareAndSet(false, true)) {
            return;
        }
        LOGGER.info("Shutting down client scripting runtime...");
        this.activeTimers.forEach((id, future) -> future.cancel(false));
        this.activeTimers.clear();
        if (this.graalContext != null) {
            try {
                this.graalContext.close(true);
            }
            catch (Exception e) {
                LOGGER.error("Error closing GraalVM context", (Throwable)e);
            }
            this.graalContext = null;
        }
        this.shutdownExecutor(this.scriptExecutor, "Script Executor");
        this.shutdownExecutor(this.timerExecutor, "Timer Executor");
        this.initialized.set(false);
        LOGGER.info("Client scripting runtime shut down");
    }

    private void shutdownExecutor(ExecutorService executor, String name) {
        executor.shutdown();
        try {
            if (!executor.awaitTermination(3L, TimeUnit.SECONDS)) {
                LOGGER.warn("{} did not terminate gracefully, forcing shutdown", (Object)name);
                executor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            LOGGER.warn("Interrupted while waiting for {} to terminate", (Object)name);
            executor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    public void processAnimationFrameQueue() {
        if (!this.initialized.get() || !this.apiService.isContextValid() || this.apiService.rendering == null) {
            return;
        }
        this.scriptExecutor.execute(() -> {
            double timestamp = (double)System.nanoTime() / 1000000.0;
            this.apiService.rendering.processAnimationFrames(timestamp);
        });
    }

    public void processGeneralTaskQueue() {
        if (!this.initialized.get()) {
            return;
        }
        this.scriptExecutor.execute(() -> {
            while (!scriptTaskQueue.isEmpty()) {
                Runnable task = scriptTaskQueue.poll();
                if (task == null) continue;
                try {
                    task.run();
                }
                catch (Exception e) {
                    LOGGER.error("Error processing queued script task", (Throwable)e);
                }
            }
        });
    }
}

