package mc.recraftors.unruled_api.mixin;

import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import mc.recraftors.unruled_api.rules.OverridesManager;
import mc.recraftors.unruled_api.utils.*;
import net.minecraft.commands.CommandSourceStack;
import net.minecraft.commands.arguments.DimensionArgument;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.level.GameRules;
import org.spongepowered.asm.mixin.Final;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Shadow;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;

import java.util.Map;
import java.util.function.Function;

import static net.minecraft.commands.Commands.literal;

@Mixin(targets = "net/minecraft/server/command/GameRuleCommand$1")
public abstract class GameruleCommandVisitorMixin {

    @Shadow @Final LiteralArgumentBuilder<CommandSourceStack> field_19419;

    @Inject(method = "visit", at = @At("TAIL"))
    private <T extends GameRules.Value<T>> void setupOverridesSubArgs(
            GameRules.Key<T> key, GameRules.Type<T> type, CallbackInfo ci
    ) {
        String k = key.getCategory().name();
        Map<String, CommandAggregator> map = Utils.aggregatorsMap.get();
        if (map == null) {
            return;
        }

        Function<String, CommandAggregator> builder = s -> new CommandAggregator();

        // gamerule <category> <rule> [<value>]
        map.computeIfAbsent(k, builder).add(literal(key.getId())
                .executes(context -> GameruleCommandInvoker.callExecuteQuery(context.getSource(), key))
                .then(type.createArgument("value")
                        .executes(context -> GameruleCommandInvoker.callExecuteSet(context, key))));

        // skip command argument creation for server-bound rules
        if (((ServerBoundAccessor)type).unruled_isServerBound()) return;

        // gamerule-override get [<category>] <rule>
        LiteralArgumentBuilder<CommandSourceStack> get = literal(key.getId())
                .executes(c -> unruled_queryOverrideSimple(c, key));
        map.computeIfAbsent("get", builder).add(get);
        map.computeIfAbsent("get-"+ k, builder).add(get);
        // gamerule-override set [<category>] <rule> <value>
        LiteralArgumentBuilder<CommandSourceStack> set = literal(key.getId()).then(type.createArgument("value")
                .executes(c -> unruled_setOverrideSimple(c, key)));
        map.computeIfAbsent("set", builder).add(set);
        map.computeIfAbsent("set-"+k, builder).add(set);
        // gamerule-override unset [<category>] <rule>
        LiteralArgumentBuilder<CommandSourceStack> unset = literal(key.getId())
                .executes(c -> unruled_unsetOverrideSimple(c, key));
        map.computeIfAbsent("unset", builder).add(unset);
        map.computeIfAbsent("unset-"+k, builder).add(unset);

        // gamerule-override in <world> get [<category>] <rule>
        LiteralArgumentBuilder<CommandSourceStack> inGet = literal(key.getId())
                .executes(c -> unruled_queryOverrideInWorld(c, key));
        map.computeIfAbsent("in-get", builder).add(inGet);
        map.computeIfAbsent("in-get-"+k, builder).add(inGet);
        // gamerule-override in <world> set [<category>] <rule> <value>
        LiteralArgumentBuilder<CommandSourceStack> inSet = literal(key.getId()).then(type.createArgument("value")
                .executes(c -> unruled_setOverrideInWorld(c, key)));
        map.computeIfAbsent("in-set", builder).add(inSet);
        map.computeIfAbsent("in-set-"+k, builder).add(inSet);
        // gamerule-override in <world> unset [<category>] <rule>
        LiteralArgumentBuilder<CommandSourceStack> inUnset = literal(key.getId())
                .executes(c -> unruled_unsetOverrideInWorld(c, key));
        map.computeIfAbsent("in-unset", builder).add(inUnset);
        map.computeIfAbsent("in-unset-"+k, builder).add(inUnset);
    }

    @Unique
    private static <T extends GameRules.Value<T>> int unruled_queryOverrideSimple(
            CommandContext<CommandSourceStack> context, GameRules.Key<T> key
    ) {
        ServerLevel world = context.getSource().getLevel();
        return unruled_queryOverride(context, key, world);
    }

    @Unique
    private static <T extends GameRules.Value<T>> int unruled_queryOverrideInWorld(
            CommandContext<CommandSourceStack> context, GameRules.Key<T> key
    ) throws CommandSyntaxException {
        return unruled_queryOverride(context, key, DimensionArgument.getDimension(context, "dimension"));
    }

    @Unique
    private static <T extends GameRules.Value<T>> int unruled_setOverrideSimple(
            CommandContext<CommandSourceStack> context, GameRules.Key<T> key
    ) {
        ServerLevel world = context.getSource().getLevel();
        return unruled_setOverride(context, key, world);
    }

    @Unique
    private static <T extends GameRules.Value<T>> int unruled_setOverrideInWorld(
            CommandContext<CommandSourceStack> context, GameRules.Key<T> key
    ) throws CommandSyntaxException {
        return unruled_setOverride(context, key, DimensionArgument.getDimension(context, "dimension"));
    }

    @Unique
    private static <T extends GameRules.Value<T>> int unruled_unsetOverrideSimple(
            CommandContext<CommandSourceStack> context, GameRules.Key<T> key
    ) {
        ServerLevel world = context.getSource().getLevel();
        return unruled_unsetOverride(context, key, world);
    }

    @Unique
    private static <T extends GameRules.Value<T>> int unruled_unsetOverrideInWorld(
            CommandContext<CommandSourceStack> context, GameRules.Key<T> key
    ) throws CommandSyntaxException {
        return unruled_unsetOverride(context, key, DimensionArgument.getDimension(context, "dimension"));
    }

    @Unique
    private static <T extends GameRules.Value<T>> int unruled_queryOverride(
            CommandContext<CommandSourceStack> context, GameRules.Key<T> key, ServerLevel world
    ) {
        OverridesManager manager = ((IGameruleOverridesProvider)world).unruled_getOverridesManager();
        String keyName = key.getId();
        ResourceLocation worldId = world.dimension().location();
        if (!manager.hasOverride(key)) {
            context.getSource().sendSuccess(() -> Component.translatableWithFallback(
                    "commands.gamerule_override.none",
                    LangFallbacks.OVERRIDE_NONE.format(keyName, worldId),
                    keyName, worldId.toString()), false);
            return 0;
        }
        T rule = manager.get(key);
        context.getSource().sendSuccess(() -> Component.translatableWithFallback(
                "commands.gamerule_override.query",
                LangFallbacks.OVERRIDE_QUERY.format(keyName, worldId, rule.toString()),
                keyName, worldId.toString(), rule.toString()), false);
        return rule.getCommandResult();
    }

    @Unique
    private static <T extends GameRules.Value<T>> int unruled_setOverride(
            CommandContext<CommandSourceStack> context, GameRules.Key<T> key, ServerLevel world
    ) {
        OverridesManager manager = ((IGameruleOverridesProvider)world).unruled_getOverridesManager();
        String keyName = key.getId();
        ResourceLocation worldId = world.dimension().location();
        boolean b = manager.override(key, context);
        if (!b) {
            context.getSource().sendSuccess(() -> Component.translatableWithFallback(
                    "commands.gamerule_override.server_bound",
                    LangFallbacks.OVERRIDE_SB.format(keyName), keyName), false);
            return context.getSource().getServer().getGameRules().getRule(key).getCommandResult();
        }
        T rule = manager.get(key);
        context.getSource().sendSuccess(() -> Component.translatableWithFallback(
                "commands.gamerule_override.set",
                LangFallbacks.OVERRIDE_SET.format(keyName, worldId, rule.toString()),
                keyName, worldId.toString(), rule.toString()), true);
        return rule.getCommandResult();
    }

    @Unique
    private static <T extends GameRules.Value<T>> int unruled_unsetOverride(
            CommandContext<CommandSourceStack> context, GameRules.Key<T> key, ServerLevel world
    ) {
        OverridesManager manager = ((IGameruleOverridesProvider)world).unruled_getOverridesManager();
        String keyName = key.getId();
        ResourceLocation worldId = world.dimension().location();
        boolean b = manager.removeOverride(key);
        String tr = "command.gamerule_override." + (b ? "unset" : "none");
        LangFallbacks fb = b ? LangFallbacks.OVERRIDE_UNSET : LangFallbacks.OVERRIDE_NONE;
        T rule = manager.get(key);
        context.getSource().sendSuccess(() -> Component.translatableWithFallback(
                tr, fb.format(keyName, worldId), keyName, worldId), true);
        return rule.getCommandResult();
    }
}
