package mc.recraftors.unruled_api.rules;

import com.mojang.brigadier.arguments.FloatArgumentType;
import com.mojang.brigadier.context.CommandContext;
import mc.recraftors.unruled_api.UnruledApi;
import mc.recraftors.unruled_api.utils.GameruleAccessor;
import mc.recraftors.unruled_api.utils.IGameRulesVisitor;
import mc.recraftors.unruled_api.utils.IGameruleAdapter;
import mc.recraftors.unruled_api.utils.IGameruleValidator;
import net.minecraft.commands.CommandSourceStack;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.GameRules.*;
import net.minecraft.world.level.GameRules.GameRuleTypeVisitor;
import net.minecraft.world.level.GameRules.Key;
import net.minecraft.world.level.GameRules.Type;
import net.minecraft.world.level.GameRules.Value;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;
import java.util.Optional;

@SuppressWarnings("unused")
public class FloatRule extends Value<FloatRule> implements GameruleAccessor<Float> {
    private float value;
    private IGameruleValidator<Float> validator;
    private IGameruleAdapter<Float> adapter;

    public FloatRule(Type<FloatRule> type, float initialValue, IGameruleValidator<Float> validator, IGameruleAdapter<Float> adapter) {
        super(type);
        Objects.requireNonNull(validator);
        Objects.requireNonNull(adapter);
        this.value = initialValue;
        this.validator = validator;
        this.adapter = adapter;
    }

    public FloatRule(Type<FloatRule> type, float initialValue) {
        this(type, initialValue, IGameruleValidator::alwaysTrue, Optional::of);
    }

    public float get() {
        return this.value;
    }

    public void set(float value, MinecraftServer server) {
        this.bump(value, server);
    }

    private void bump(float value, MinecraftServer server) {
        boolean b = false;
        if (this.validator.validate(value)) b = true;
        else {
            Optional<Float> o = this.adapter.adapt(value);
            if (o.isPresent() && this.validator.validate(o.get())) {
                value = o.get();
                b = true;
            }
        }
        if (b) {
            this.value = value;
            this.onChanged(server);
        }
    }

    public boolean validate(String input) {
        try {
            float f = Float.parseFloat(input);
            if (this.validator.validate(f)) {
                this.value = f;
                return true;
            }
            return false;
        } catch (NumberFormatException e) {
            return false;
        }
    }

    private static float parseFloat(String input) {
        if (!input.isEmpty()) {
            try {
                return Float.parseFloat(input);
            } catch (NumberFormatException e) {
                UnruledApi.LOGGER.warn("Failed to parse float from '{}'", input);
            }
        }
        return 0;
    }

    private void set(float f) {
        if (this.validator.validate(f)) {
            this.value = f;
            return;
        }
        Optional<Float> o = this.adapter.adapt(f);
        if (o.isEmpty() || !this.validator.validate(o.get())) return;
        this.value = o.get();
    }

    @Override
    protected void updateFromArgument(CommandContext<CommandSourceStack> context, String name) {
        float f = FloatArgumentType.getFloat(context, name);
        this.set(f);
    }

    @Override
    protected void deserialize(String value) {
        float f = FloatRule.parseFloat(value);
        this.set(f);
    }

    @Override
    public String serialize() {
        return Float.toString(this.value);
    }

    @Override
    public int getCommandResult() {
        return (int) this.value;
    }

    @Override
    protected FloatRule getSelf() {
        return this;
    }

    @Override
    public FloatRule copy() {
        return new FloatRule(this.type, this.value, this.validator, this.adapter);
    }

    @Override
    public void setValue(FloatRule rule, @Nullable MinecraftServer server) {
        this.bump(rule.get(), server);
    }

    @Override
    public IGameruleValidator<Float> unruled_getValidator() {
        return this.validator;
    }

    @Override
    public void unruled_setValidator(IGameruleValidator<Float> validator) {
        this.validator = Objects.requireNonNull(validator);
    }

    @Override
    public IGameruleAdapter<Float> unruled_getAdapter() {
        return this.adapter;
    }

    @Override
    public void unruled_setAdapter(IGameruleAdapter<Float> adapter) {
        this.adapter = Objects.requireNonNull(adapter);
    }

    public static class Builder extends RuleBuilder<FloatRule, Float> {
        public Builder(float initialValue) {
            super(FloatArgumentType::floatArg, Builder::acceptor, FloatRule.class, initialValue);
        }

        @Override
        @NotNull
        protected FloatRule ruleBuilder(Type<FloatRule> type) {
            return new FloatRule(type, super.initialValue, super.validator, super.adapter);
        }

        static void acceptor(GameRuleTypeVisitor consumer, Key<FloatRule> key, Type<FloatRule> type) {
            ((IGameRulesVisitor)consumer).unruled_visitFloat(key, type);
        }
    }
}
