package mc.recraftors.unruled_api.rules;

import com.mojang.brigadier.context.CommandContext;
import com.mojang.serialization.*;
import java.util.HashMap;
import java.util.Map;
import net.minecraft.commands.CommandSourceStack;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.util.datafix.DataFixTypes;
import net.minecraft.world.flag.FeatureFlagSet;
import net.minecraft.world.level.GameRules;
import net.minecraft.world.level.saveddata.SavedData;
import net.minecraft.world.level.saveddata.SavedDataType;

public class OverridesManager extends SavedData {
    private final GameRulesOverrides overrides;
    private final Map<String, String> rawMap;

    public static SavedDataType<OverridesManager> getPersistentStateType() {
        return new SavedDataType<>("gamerules",
                OverridesManager::builder,
                OverridesManager::codecBuilder,
                DataFixTypes.valueOf("GAMERULES")
        );
    }

    private static OverridesManager builder(SavedData.Context ctx) {
        ServerLevel world = ctx.levelOrThrow();
        return new OverridesManager(world.getServer().getGameRules(), world.enabledFeatures());
    }

    private static Codec<OverridesManager> codecBuilder(SavedData.Context ctx) {
        ServerLevel world = ctx.levelOrThrow();
        return Codec.PASSTHROUGH.comapFlatMap(
                dynamic -> {
                    Map<String, String> map = dynamic.asMap(
                            k -> k.asString().getOrThrow(), v -> v.asString().getOrThrow());
                    var o = new OverridesManager(world.getServer().getGameRules(), map, world.enabledFeatures());
                    return DataResult.success(o);
                }, OverridesManager::serialize
        );
    }

    private static Dynamic<?> serialize(OverridesManager o) {
        o.cache();
        return new Dynamic<>(JavaOps.INSTANCE, o.rawMap);
    }

    private OverridesManager(GameRules base, Map<String, String> rawMap, FeatureFlagSet featureSet) {
        this.overrides = new GameRulesOverrides(base, rawMap, featureSet);
        this.rawMap = rawMap;
    }

    private OverridesManager(GameRules base, FeatureFlagSet featureSet) {
        this(base, new HashMap<>(), featureSet);
    }

    @Override
    public boolean isDirty() {
        this.cache();
        return super.isDirty();
    }

    public <T extends GameRules.Value<T>> boolean hasOverride(GameRules.Key<T> key) {
        return this.overrides.hasOverride(key);
    }

    public <T extends GameRules.Value<T>> boolean override(GameRules.Key<T> key, CommandContext<CommandSourceStack> context) {
        if (this.overrides.override(key, context)) {
            this.setDirty();
            return true;
        }
        return false;
    }

    public <T extends GameRules.Value<T>> boolean removeOverride(GameRules.Key<T> key) {
        if (this.overrides.removeOverride(key)) {
            this.setDirty();
            this.rawMap.remove(key.getId());
            return true;
        }
        return false;
    }

    public <T extends GameRules.Value<T>> T get(GameRules.Key<T> key) {
        return this.overrides.getRule(key);
    }

    private void cache() {
        this.overrides.getOverrides().forEach(k -> {
            String s = k.getId();
            String v = this.overrides.getRule(k).serialize();
            if (!this.rawMap.getOrDefault(s, "").equals(v)) {
                this.rawMap.put(s, v);
            }
        });
    }

    public boolean isEmpty() {
        return this.overrides == null || this.overrides.isEmpty();
    }

    public GameRulesOverrides getOverrides() {
        return this.overrides;
    }
}
