package mc.recraftors.unruled_api.rules;

import com.mojang.brigadier.arguments.ArgumentType;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.DynamicCommandExceptionType;
import mc.recraftors.unruled_api.UnruledApi;
import mc.recraftors.unruled_api.impl.FullRegistryWrapperLookup;
import mc.recraftors.unruled_api.utils.*;
import net.fabricmc.api.EnvType;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.commands.CommandSourceStack;
import net.minecraft.commands.arguments.ResourceArgument;
import net.minecraft.commands.arguments.ResourceOrTagKeyArgument;
import net.minecraft.core.Holder;
import net.minecraft.core.Registry;
import net.minecraft.core.RegistryAccess;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.GameRules.*;
import net.minecraft.world.flag.FeatureFlagSet;
import net.minecraft.world.level.GameRules.GameRuleTypeVisitor;
import net.minecraft.world.level.GameRules.Key;
import net.minecraft.world.level.GameRules.Type;
import net.minecraft.world.level.GameRules.Value;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Optional;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

public class RegistryEntryRule <T> extends Value<RegistryEntryRule<T>> implements GameruleAccessor<T> {
    private static final DynamicCommandExceptionType UNKNOWN_REGISTRY_ENTRY = new DynamicCommandExceptionType(
            id -> Component.translatableWithFallback("commands.unruled_api.unknown_registry_entry", "Unknown entry %s in registry", id)
    );

    private ProviderEntry<T> provider;
    private Registry<T> registry;
    private T value;
    private IGameruleValidator<T> validator;
    private IGameruleAdapter<T> adapter;

    public RegistryEntryRule(Type<RegistryEntryRule<T>> type, Registry<T> registry, T initialValue,
                             IGameruleValidator<T> validator, IGameruleAdapter<T> adapter) throws UnsupportedOperationException {
        super(type);
        if (registry.getKey(initialValue) == null) {
            throw new UnsupportedOperationException("initial value must be a registry member");
        }
        this.registry = requireNonNull(registry);
        this.provider = new ProviderEntry<>(registry.key(), requireNonNull(registry.getKey(initialValue)));
        this.value = requireNonNull(initialValue);
        this.validator = requireNonNull(validator);
        this.adapter = requireNonNull(adapter);
    }

    public RegistryEntryRule(Type<RegistryEntryRule<T>> type, ResourceKey<? extends Registry<T>> registryKey,
                             ResourceLocation valueKey, IGameruleValidator<T> validator, IGameruleAdapter<T> adapter) {
        super(type);
        this.registry = null;
        this.provider = new ProviderEntry<>(requireNonNull(registryKey), requireNonNull(valueKey));
        this.validator = requireNonNull(validator);
        this.adapter = requireNonNull(adapter);
    }

    @SuppressWarnings("unused")
    public RegistryEntryRule(Type<RegistryEntryRule<T>> type, Registry<T> registry, T initialValue) {
        this(type, registry, initialValue, IGameruleValidator::alwaysTrue, Optional::of);
    }

    private static <V> Supplier<ArgumentType<?>> argSupplierBuilder(
            ResourceKey<? extends Registry<V>> key, FeatureFlagSet featureSet
    ) {
        return () -> {
            try {
                return ResourceArgument.resource(
                        new FullRegistryWrapperLookup(featureSet, Utils.registryAccessThreadLocal.get()).toCommandRegAccessAccess(), key
                );
            } catch (Exception ex) {
                return ResourceOrTagKeyArgument.resourceOrTagKey(key);
            }
        };
    }

    public void provide(RegistryAccess registryManager) {
        requireNonNull(registryManager);
        if (this.registry == null) this.registry = requireNonNull(registryManager.lookupOrThrow(this.provider.regKey()));
        T t = this.registry.getValue(this.provider.entryId());
        if (t != null) this.set(t);
    }

    public T get() {
        return this.value;
    }

    public void set(T value, @Nullable MinecraftServer server) {
        this.bump(value, server);
    }

    private void bump(T value, MinecraftServer server) {
        boolean b = this.validator.validate(value);
        if (!b) {
            Optional<T> o = this.adapter.adapt(value);
            b = (o.isPresent() && this.validator.validate(value = o.get()));
        }
        if (b) {
            this.value = value;
            this.provider = requireNonNull(this.provider.withId(this.registry.getKey(value)));
            this.onChanged(server);
        }
    }

    private void set(T t) {
        if (this.validator.validate(t)) {
            this.value = t;
            return;
        }
        Optional<T> o = this.adapter.adapt(t);
        if (o.isEmpty() || !this.validator.validate(o.get())) return;
        this.value = o.get();
    }

    public boolean validate(String input) {
        if (this.registry == null && FabricLoader.getInstance().getEnvironmentType() == EnvType.CLIENT) {
            this.registry = ClientUtils.getClientWorldRegistryOrThrow(this.provider.regKey());
        }
        if (this.registry == null) return false;
        Optional<T> o = parseInput(input, this.registry);
        if (o.isPresent() && this.validator.validate(o.get())) {
            this.value = o.get();
            return true;
        }
        return false;
    }

    private static <U> Optional<U> parseInput(String input, Registry<U> registry) {
        ResourceLocation id = ResourceLocation.tryParse(input);
        if (id == null || !registry.containsKey(id)) {
            return Optional.empty();
        }
        return Optional.ofNullable(registry.getValue(id));
    }

    @Override
    protected void updateFromArgument(CommandContext<CommandSourceStack> context, String name) {
        try {
            //noinspection unchecked
            Holder.Reference<T> entry = ResourceArgument
                    .getResource(context, name, (net.minecraft.resources.ResourceKey<Registry<T>>) this.registry.key());
            T t = this.registry.getValue(entry.key().location());
            this.set(requireNonNull(t));
        } catch (IllegalArgumentException e) {
            try {
                //noinspection unchecked
                ResourceOrTagKeyArgument.Result<T> predicate = ResourceOrTagKeyArgument
                        .getResourceOrTagKey(context, name, (ResourceKey<Registry<T>>) this.registry.key(), UNKNOWN_REGISTRY_ENTRY);
                Holder.Reference<T> entry = this.registry.listElements().filter(predicate).findFirst()
                        .orElseThrow(() -> new IllegalArgumentException("No entry found"));
                this.set(entry.value());
            } catch (CommandSyntaxException ex) {
                throw new IllegalArgumentException(e);
            }
        } catch (CommandSyntaxException|NullPointerException|IllegalStateException e) {
            throw new IllegalArgumentException(e);
        }
    }

    @Override
    protected void deserialize(String value) {
        if (this.registry == null) {
            return;
        }
        parseInput(value, this.registry).ifPresentOrElse(
                this::set,
                () -> {
                    if (this.registry != null) {
                        UnruledApi.LOGGER.warn("Unable to find an entry of {} matching '{}'", this.registry.key(), value);
                    } else {
                        this.provider = this.provider.withId(ResourceLocation.tryParse(value));
                    }
                }
        );
    }

    @Override
    public String serialize() {
        if (this.registry == null) {
            return this.provider.entryId.toString();
        }
        return requireNonNull(this.registry.getKey(this.value)).toString();
    }

    @Override
    public int getCommandResult() {
        return this.registry.getId(this.value);
    }

    @Override
    protected RegistryEntryRule<T> getSelf() {
        return this;
    }

    @Override
    public RegistryEntryRule<T> copy() {
        if (this.registry == null) {
            return new RegistryEntryRule<>(this.type, this.provider.regKey, this.provider.entryId, this.validator, this.adapter);
        }
        return new RegistryEntryRule<>(this.type, this.registry, this.value, this.validator, this.adapter);
    }

    @Override
    public void setValue(RegistryEntryRule<T> rule, @Nullable MinecraftServer server) {
        this.bump(rule.value, server);
    }

    @Override
    public IGameruleValidator<T> unruled_getValidator() {
        return this.validator;
    }

    @Override
    public void unruled_setValidator(IGameruleValidator<T> validator) {
        this.validator = requireNonNull(validator);
    }

    @Override
    public IGameruleAdapter<T> unruled_getAdapter() {
        return this.adapter;
    }

    @Override
    public void unruled_setAdapter(IGameruleAdapter<T> adapter) {
        this.adapter = requireNonNull(adapter);
    }

    public static class Builder <V> extends RuleBuilder<RegistryEntryRule<V>, V> {
        private final Registry<V> registry;
        private final ProviderEntry<V> provider;
        public Builder(Registry<V> registry, V initialValue) {
            //noinspection unchecked
            super(argSupplierBuilder(registry.key(), FeatureFlagSet.of()), Builder::acceptor,
                    (Class<RegistryEntryRule<V>>) (Object) RegistryEntryRule.class, initialValue);
            this.registry = requireNonNull(registry);
            this.provider = null;
        }

        public Builder(ResourceKey<? extends Registry<V>> regKey, ResourceLocation valueId) {
            //noinspection unchecked
            super(argSupplierBuilder(regKey, FeatureFlagSet.of()), Builder::acceptor,
                    (Class<RegistryEntryRule<V>>) (Object) RegistryEntryRule.class, null);
            this.registry = null;
            this.provider = new ProviderEntry<>(regKey, valueId);
        }

        @Override
        @NotNull
        protected RegistryEntryRule<V> ruleBuilder(Type<RegistryEntryRule<V>> type) {
            if (this.registry == null) {
                return new RegistryEntryRule<>(type, this.provider.regKey, this.provider.entryId, super.validator, super.adapter);
            }
            return new RegistryEntryRule<>(type, this.registry, super.initialValue, super.validator, super.adapter);
        }

        static <U> void acceptor(GameRuleTypeVisitor consumer, Key<RegistryEntryRule<U>> key, Type<RegistryEntryRule<U>> type) {
            ((IGameRulesVisitor)consumer).unruled_visitRegistryEntry(key, type);
        }
    }

    private record ProviderEntry <T> (ResourceKey<? extends Registry<T>> regKey, ResourceLocation entryId) {
        ProviderEntry<T> withId(ResourceLocation id) {
            return new ProviderEntry<>(this.regKey, id);
        }
    }
}
