package mc.recraftors.unruled_api.rules;

import com.mojang.brigadier.arguments.ArgumentType;
import com.mojang.brigadier.arguments.IntegerArgumentType;
import mc.recraftors.unruled_api.impl.GameruleValidatorAdapter;
import mc.recraftors.unruled_api.utils.GameruleAccessor;
import mc.recraftors.unruled_api.utils.IGameruleAdapter;
import mc.recraftors.unruled_api.utils.IGameruleValidator;
import mc.recraftors.unruled_api.utils.ServerBoundAccessor;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.flag.FeatureFlagSet;
import net.minecraft.world.level.GameRules;
import net.minecraft.world.level.GameRules.Category;
import net.minecraft.world.level.GameRules.GameRuleTypeVisitor;
import net.minecraft.world.level.GameRules.IntegerValue;
import net.minecraft.world.level.GameRules.Key;
import net.minecraft.world.level.GameRules.Type;
import net.minecraft.world.level.GameRules.VisitorCaller;
import org.jetbrains.annotations.NotNull;

import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;
import static net.minecraft.world.level.GameRules.*;

/**
 * Base class for building gamerules with all the Unruled API features directly accessible.<br>
 * All base implementations can be easily accessed via the {@link mc.recraftors.unruled_api.UnruledApi}
 * static methods, to avoid multiplicative class imports.
 * @param <T> The gamerule type
 * @param <V> The gamerule's value type
 */
public abstract class RuleBuilder <T extends GameRules.Value<T>, V> {
    protected final Supplier<ArgumentType<?>> argumentSupplier;
    protected final VisitorCaller<T> screenVisitor;
    protected final Class<T> ruleClass;
    protected V initialValue;
    protected BiConsumer<MinecraftServer, T> changeCallback;
    protected IGameruleValidator<V> validator;
    protected IGameruleAdapter<V> adapter;
    protected FeatureFlagSet featureSet;
    protected boolean serverBound;

    protected RuleBuilder(Supplier<ArgumentType<?>> argumentSupplier, VisitorCaller<T> screenVisitor,
                          Class<T> ruleClass, V initialValue) {
        this.argumentSupplier = requireNonNull(argumentSupplier);
        this.screenVisitor = requireNonNull(screenVisitor);
        this.ruleClass = requireNonNull(ruleClass);
        this.initialValue = initialValue;
        this.changeCallback = (w, s) -> {};
        this.validator = IGameruleValidator::alwaysTrue;
        this.adapter = Optional::of;
        this.featureSet = FeatureFlagSet.of();
    }

    protected abstract @NotNull T ruleBuilder(Type<T> type);

    /**
     * Returns the rule builder's configured featureSet.
     * @return The rule builder's configured featureSet.
     */
    protected final FeatureFlagSet getFeatureSet() {
        return this.featureSet;
    }

    /**
     * Sets a change callback for the rule to be built.<br>
     * A change callback is triggered any time the gamerule's value gets changed.
     * @param changeCallback The change callback to be applied to a built rule.
     * @return this
     */
    public final RuleBuilder<T, V> setChangeCallback(BiConsumer<MinecraftServer, T> changeCallback) {
        this.changeCallback = requireNonNull(changeCallback);
        return this;
    }

    /**
     * Sets a value validator for the rule to be built.<br>
     * A value validator is checked any time the gamerule's value is set.<br>
     * Best used along with a value adapter. See {@link #setAdapter}.
     * @param validator The value validator to be applied to a built rule.
     * @return this
     */
    public final RuleBuilder<T, V> setValidator(IGameruleValidator<V> validator) {
        this.validator = requireNonNull(validator);
        return this;
    }

    /**
     * Sets a value adapter for the rule to be built.<br>
     * A value adapter is applied when a value gets invalidated by the rule's validator.<br>
     * As such, it requires a validator to be set to be relevant. See {@link #setValidator}.
     * @param adapter The value adapter to be applied to a built rule.
     * @return this
     */
    public final RuleBuilder<T, V> setAdapter(IGameruleAdapter<V> adapter) {
        this.adapter = requireNonNull(adapter);
        return this;
    }

    /**
     * Sets a value validatorAdapter for the rule to be built.<br>
     * A value validatorAdapter works as both validator and adapter.<br>
     * Some basic implementation can be found in {@link mc.recraftors.unruled_api.impl}.
     * @param validatorAdapter The value validatorAdapter to be applied to a built rule.
     * @return this
     */
    public final RuleBuilder<T, V> setValidatorAdapter(GameruleValidatorAdapter<V> validatorAdapter) {
        requireNonNull(validatorAdapter);
        this.validator = validatorAdapter;
        this.adapter = validatorAdapter;
        return this;
    }

    /**
     * Sets a featureSet for the rule to be built.<br>
     * A featureSet (or experimental feature set) can determine whether a rule is indeed loaded or not.<br>
     * This is a Minecraft core feature, and not inherently used by Unruled API.
     * @param featureSet The featureSet to be applied to a built rule.
     * @return this
     */
    public final RuleBuilder<T, V> setFeatureSet(FeatureFlagSet featureSet) {
        this.featureSet = requireNonNull(featureSet);
        return this;
    }

    /**
     * Sets whether a built rule should be server-bound.<br>
     * A server-bound rule cannot be overriden, and only the server value will ever be provided.
     * @param serverBound Whether a built rule should be server-bound.
     * @return this
     */
    public final RuleBuilder<T, V> setServerBound(boolean serverBound) {
        this.serverBound = serverBound;
        return this;
    }

    /**
     * Builds a gamerule Type (or core builder), which can be then registered for the rule to exist in the game.
     * @return The built rule Type.
     */
    public Type<T> build() {
        Type<T> type = new Type<>(this.argumentSupplier, this::ruleBuilder, this.changeCallback, this.screenVisitor, this.ruleClass, this.featureSet);
        ((ServerBoundAccessor)type).unruled_setServerBound(this.serverBound);
        return type;
    }

    /**
     * Registers the builder's build result with the provided name and category, and
     * returns the matching gamerule key.
     * @param name The name of the newly registered rule
     * @param category The category of the newly registered rule
     * @return The key of the newly registered rule
     */
    public Key<T> register(String name, Category category) {
        return GameRules.register(name, category, build());
    }

    public static class IntRuleBuilder extends RuleBuilder<IntegerValue, Integer> {
        public IntRuleBuilder(int initialValue) {
            super(IntegerArgumentType::integer, GameRuleTypeVisitor::visitInteger, IntegerValue.class, initialValue);
        }

        @Override
        @SuppressWarnings("unchecked")
        protected @NotNull IntegerValue ruleBuilder(Type<IntegerValue> type) {
            IntegerValue rule = new IntegerValue(type, super.initialValue);
            GameruleAccessor<Integer> accessor = ((GameruleAccessor<Integer>)rule);
            accessor.unruled_setValidator(super.validator);
            accessor.unruled_setAdapter(super.adapter);
            return rule;
        }
    }
}
