package mc.recraftors.unruled_api.rules;

import com.mojang.brigadier.arguments.LongArgumentType;
import com.mojang.brigadier.context.CommandContext;
import mc.recraftors.unruled_api.UnruledApi;
import mc.recraftors.unruled_api.utils.GameruleAccessor;
import mc.recraftors.unruled_api.utils.IGameRulesVisitor;
import mc.recraftors.unruled_api.utils.IGameruleAdapter;
import mc.recraftors.unruled_api.utils.IGameruleValidator;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.command.ServerCommandSource;
import net.minecraft.world.GameRules.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;
import java.util.Optional;

@SuppressWarnings("unused")
public class LongRule extends Rule<LongRule> implements GameruleAccessor<Long> {
    private long value;
    private IGameruleValidator<Long> validator;
    private IGameruleAdapter<Long> adapter;

    public LongRule(Type<LongRule> type, long initialValue, IGameruleValidator<Long> validator, IGameruleAdapter<Long> adapter) {
        super(type);
        Objects.requireNonNull(validator);
        Objects.requireNonNull(adapter);
        this.value = initialValue;
        this.validator = validator;
        this.adapter = adapter;
    }

    public LongRule(Type<LongRule> type, long initialValue) {
        this(type, initialValue, IGameruleValidator::alwaysTrue, Optional::of);
    }

    public long get() {
        return this.value;
    }

    public void set(long value, MinecraftServer server) {
        this.bump(value, server);
    }

    private void bump(long value, MinecraftServer server) {
        boolean b = false;
        if (this.validator.validate(value)) b = true;
        else {
            Optional<Long> o = this.adapter.adapt(value);
            if (o.isPresent() && this.validator.validate(o.get())) {
                value = o.get();
                b = true;
            }
        }
        if (b) {
            this.value = value;
            this.changed(server);
        }
    }

    private void set(long f) {
        if (this.validator.validate(f)) {
            this.value = f;
            return;
        }
        Optional<Long> o = this.adapter.adapt(f);
        if (o.isEmpty() || !this.validator.validate(o.get())) return;
        this.value = o.get();
    }

    public boolean validate(String input) {
        try {
            long l = Long.parseLong(input);
            if (this.validator.validate(l)) {
                this.value = l;
                return true;
            }
            return false;
        } catch (NumberFormatException e) {
            return false;
        }
    }

    private static long parseLong(String input) {
        if (!input.isEmpty()) {
            try {
                return Long.parseLong(input);
            } catch (NumberFormatException e) {
                UnruledApi.LOGGER.warn("Failed to parse long {}", input);
            }
        }
        return 0;
    }

    @Override
    protected void setFromArgument(CommandContext<ServerCommandSource> context, String name) {
        long l = LongArgumentType.getLong(context, name);
        this.set(l);
    }

    @Override
    protected void deserialize(String value) {
        long l = LongRule.parseLong(value);
        this.set(l);
    }

    @Override
    public String serialize() {
        return Long.toString(this.value);
    }

    @Override
    public int getCommandResult() {
        return (int) this.value;
    }

    @Override
    protected LongRule getThis() {
        return this;
    }

    @Override
    public LongRule copy() {
        return new LongRule(this.type, this.value, this.validator, this.adapter);
    }

    @Override
    public void setValue(LongRule rule, @Nullable MinecraftServer server) {
        this.bump(rule.get(), server);
    }

    @Override
    public IGameruleValidator<Long> unruled_getValidator() {
        return this.validator;
    }

    @Override
    public void unruled_setValidator(IGameruleValidator<Long> validator) {
        this.validator = Objects.requireNonNull(validator);
    }

    @Override
    public IGameruleAdapter<Long> unruled_getAdapter() {
        return this.adapter;
    }

    @Override
    public void unruled_setAdapter(IGameruleAdapter<Long> adapter) {
        this.adapter = Objects.requireNonNull(adapter);
    }

    public static class Builder extends RuleBuilder<LongRule, Long> {
        public Builder(long initialValue) {
            super(LongArgumentType::longArg, Builder::acceptor, initialValue);
        }

        @Override
        @NotNull
        protected LongRule ruleBuilder(Type<LongRule> type) {
            return new LongRule(type, super.initialValue, super.validator, super.adapter);
        }

        static void acceptor(Visitor consumer, Key<LongRule> key, Type<LongRule> type) {
            ((IGameRulesVisitor)consumer).unruled_visitLong(key, type);
        }
    }
}
