package mc.recraftors.unruled_api.rules;

import com.mojang.brigadier.arguments.ArgumentType;
import com.mojang.brigadier.arguments.IntegerArgumentType;
import mc.recraftors.unruled_api.impl.GameruleValidatorAdapter;
import mc.recraftors.unruled_api.utils.GameruleAccessor;
import mc.recraftors.unruled_api.utils.IGameruleAdapter;
import mc.recraftors.unruled_api.utils.IGameruleValidator;
import net.minecraft.resource.featuretoggle.FeatureSet;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.GameRules;
import org.jetbrains.annotations.NotNull;

import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;
import static net.minecraft.world.GameRules.*;

public abstract class RuleBuilder <T extends GameRules.Rule<T>, V> {
    protected final Supplier<ArgumentType<?>> argumentSupplier;
    protected final Acceptor<T> screenVisitor;
    protected V initialValue;
    protected BiConsumer<MinecraftServer, T> changeCallback;
    protected IGameruleValidator<V> validator;
    protected IGameruleAdapter<V> adapter;
    protected FeatureSet featureSet;

    protected RuleBuilder(Supplier<ArgumentType<?>> argumentSupplier, Acceptor<T> screenVisitor, V initialValue) {
        this.argumentSupplier = requireNonNull(argumentSupplier);
        this.screenVisitor = requireNonNull(screenVisitor);
        this.initialValue = initialValue;
        this.changeCallback = (w, s) -> {};
        this.validator = IGameruleValidator::alwaysTrue;
        this.adapter = Optional::of;
        this.featureSet = FeatureSet.empty();
    }

    protected abstract @NotNull T ruleBuilder(Type<T> type);

    public final RuleBuilder<T, V> setChangeCallback(BiConsumer<MinecraftServer, T> changeCallback) {
        this.changeCallback = requireNonNull(changeCallback);
        return this;
    }

    public final RuleBuilder<T, V> setValidator(IGameruleValidator<V> validator) {
        this.validator = requireNonNull(validator);
        return this;
    }

    public final RuleBuilder<T, V> setAdapter(IGameruleAdapter<V> adapter) {
        this.adapter = requireNonNull(adapter);
        return this;
    }

    public final RuleBuilder<T, V> setValidatorAdapter(GameruleValidatorAdapter<V> validatorAdapter) {
        requireNonNull(validatorAdapter);
        this.validator = validatorAdapter;
        this.adapter = validatorAdapter;
        return this;
    }

    public final RuleBuilder<T, V> setFeatureSet(FeatureSet featureSet) {
        this.featureSet = requireNonNull(featureSet);
        return this;
    }

    public Type<T> build() {
        return new Type<>(this.argumentSupplier, this::ruleBuilder, this.changeCallback, this.screenVisitor);
    }

    public Key<T> register(String name, Category category) {
        return GameRules.register(name, category, build());
    }

    public static class IntRuleBuilder extends RuleBuilder<IntRule, Integer> {
        public IntRuleBuilder(int initialValue) {
            super(IntegerArgumentType::integer, Visitor::visitInt, initialValue);
        }

        @Override
        @SuppressWarnings("unchecked")
        protected @NotNull IntRule ruleBuilder(Type<IntRule> type) {
            IntRule rule = new IntRule(type, super.initialValue);
            GameruleAccessor<Integer> accessor = ((GameruleAccessor<Integer>)rule);
            accessor.unruled_setValidator(super.validator);
            accessor.unruled_setAdapter(super.adapter);
            return rule;
        }
    }
}
