package mc.recraftors.unruled_api.rules;

import com.mojang.brigadier.arguments.FloatArgumentType;
import com.mojang.brigadier.context.CommandContext;
import mc.recraftors.unruled_api.UnruledApi;
import mc.recraftors.unruled_api.utils.GameruleAccessor;
import mc.recraftors.unruled_api.utils.IGameRulesVisitor;
import mc.recraftors.unruled_api.utils.IGameruleAdapter;
import mc.recraftors.unruled_api.utils.IGameruleValidator;
import net.minecraft.class_1928.class_4311;
import net.minecraft.class_1928.class_4313;
import net.minecraft.class_1928.class_4314;
import net.minecraft.class_1928.class_4315;
import net.minecraft.class_2168;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.GameRules.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;
import java.util.Optional;

@SuppressWarnings("unused")
public class FloatRule extends class_4315<FloatRule> implements GameruleAccessor<Float> {
    private float value;
    private IGameruleValidator<Float> validator;
    private IGameruleAdapter<Float> adapter;

    public FloatRule(class_4314<FloatRule> type, float initialValue, IGameruleValidator<Float> validator, IGameruleAdapter<Float> adapter) {
        super(type);
        Objects.requireNonNull(validator);
        Objects.requireNonNull(adapter);
        this.value = initialValue;
        this.validator = validator;
        this.adapter = adapter;
    }

    public FloatRule(class_4314<FloatRule> type, float initialValue) {
        this(type, initialValue, IGameruleValidator::alwaysTrue, Optional::of);
    }

    public float get() {
        return this.value;
    }

    public void set(float value, MinecraftServer server) {
        this.bump(value, server);
    }

    private void bump(float value, MinecraftServer server) {
        boolean b = false;
        if (this.validator.validate(value)) b = true;
        else {
            Optional<Float> o = this.adapter.adapt(value);
            if (o.isPresent() && this.validator.validate(o.get())) {
                value = o.get();
                b = true;
            }
        }
        if (b) {
            this.value = value;
            this.method_20778(server);
        }
    }

    public boolean validate(String input) {
        try {
            float f = Float.parseFloat(input);
            if (this.validator.validate(f)) {
                this.value = f;
                return true;
            }
            return false;
        } catch (NumberFormatException e) {
            return false;
        }
    }

    private static float parseFloat(String input) {
        if (!input.isEmpty()) {
            try {
                return Float.parseFloat(input);
            } catch (NumberFormatException e) {
                UnruledApi.LOGGER.warn("Failed to parse float from '{}'", input);
            }
        }
        return 0;
    }

    private void set(float f) {
        if (this.validator.validate(f)) {
            this.value = f;
            return;
        }
        Optional<Float> o = this.adapter.adapt(f);
        if (o.isEmpty() || !this.validator.validate(o.get())) return;
        this.value = o.get();
    }

    @Override
    protected void method_20776(CommandContext<class_2168> context, String name) {
        float f = FloatArgumentType.getFloat(context, name);
        this.set(f);
    }

    @Override
    protected void method_20777(String value) {
        float f = FloatRule.parseFloat(value);
        this.set(f);
    }

    @Override
    public String method_20779() {
        return Float.toString(this.value);
    }

    @Override
    public int method_20781() {
        return (int) this.value;
    }

    @Override
    protected FloatRule method_20782() {
        return this;
    }

    @Override
    public FloatRule method_27338() {
        return new FloatRule(this.field_19417, this.value, this.validator, this.adapter);
    }

    @Override
    public void setValue(FloatRule rule, @Nullable MinecraftServer server) {
        this.bump(rule.get(), server);
    }

    @Override
    public IGameruleValidator<Float> unruled_getValidator() {
        return this.validator;
    }

    @Override
    public void unruled_setValidator(IGameruleValidator<Float> validator) {
        this.validator = Objects.requireNonNull(validator);
    }

    @Override
    public IGameruleAdapter<Float> unruled_getAdapter() {
        return this.adapter;
    }

    @Override
    public void unruled_setAdapter(IGameruleAdapter<Float> adapter) {
        this.adapter = Objects.requireNonNull(adapter);
    }

    public static class Builder extends RuleBuilder<FloatRule, Float> {
        public Builder(float initialValue) {
            super(FloatArgumentType::floatArg, Builder::acceptor, initialValue);
        }

        @Override
        @NotNull
        protected FloatRule ruleBuilder(class_4314<FloatRule> type) {
            return new FloatRule(type, super.initialValue, super.validator, super.adapter);
        }

        static void acceptor(class_4311 consumer, class_4313<FloatRule> key, class_4314<FloatRule> type) {
            ((IGameRulesVisitor)consumer).unruled_visitFloat(key, type);
        }
    }
}
