package mc.recraftors.unruled_api.rules;

import com.mojang.brigadier.arguments.ArgumentType;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.DynamicCommandExceptionType;
import mc.recraftors.unruled_api.UnruledApi;
import mc.recraftors.unruled_api.impl.FullRegistryWrapperLookup;
import mc.recraftors.unruled_api.utils.*;
import net.fabricmc.api.EnvType;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1928.class_4311;
import net.minecraft.class_1928.class_4313;
import net.minecraft.class_1928.class_4314;
import net.minecraft.class_1928.class_4315;
import net.minecraft.class_2168;
import net.minecraft.class_2378;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_5455;
import net.minecraft.class_6880;
import net.minecraft.class_7066;
import net.minecraft.class_7699;
import net.minecraft.class_7733;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.GameRules.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Optional;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

public class RegistryEntryRule <T> extends class_4315<RegistryEntryRule<T>> implements GameruleAccessor<T> {
    private static final DynamicCommandExceptionType UNKNOWN_REGISTRY_ENTRY = new DynamicCommandExceptionType(
            id -> class_2561.method_48322("commands.unruled_api.unknown_registry_entry", "Unknown entry %s in registry", id)
    );

    private ProviderEntry<T> provider;
    private class_2378<T> registry;
    private T value;
    private IGameruleValidator<T> validator;
    private IGameruleAdapter<T> adapter;

    public RegistryEntryRule(class_4314<RegistryEntryRule<T>> type, class_2378<T> registry, T initialValue,
                             IGameruleValidator<T> validator, IGameruleAdapter<T> adapter) throws UnsupportedOperationException {
        super(type);
        if (registry.method_10221(initialValue) == null) {
            throw new UnsupportedOperationException("initial value must be a registry member");
        }
        this.registry = requireNonNull(registry);
        this.provider = new ProviderEntry<>(registry.method_46765(), requireNonNull(registry.method_10221(initialValue)));
        this.value = requireNonNull(initialValue);
        this.validator = requireNonNull(validator);
        this.adapter = requireNonNull(adapter);
    }

    public RegistryEntryRule(class_4314<RegistryEntryRule<T>> type, class_5321<? extends class_2378<T>> registryKey,
                             class_2960 valueKey, IGameruleValidator<T> validator, IGameruleAdapter<T> adapter) {
        super(type);
        this.registry = null;
        this.provider = new ProviderEntry<>(requireNonNull(registryKey), requireNonNull(valueKey));
        this.validator = requireNonNull(validator);
        this.adapter = requireNonNull(adapter);
    }

    @SuppressWarnings("unused")
    public RegistryEntryRule(class_4314<RegistryEntryRule<T>> type, class_2378<T> registry, T initialValue) {
        this(type, registry, initialValue, IGameruleValidator::alwaysTrue, Optional::of);
    }

    private static <V> Supplier<ArgumentType<?>> argSupplierBuilder(
            class_5321<? extends class_2378<V>> key, class_7699 featureSet
    ) {
        return () -> {
            try {
                return class_7733.method_45603(
                        new FullRegistryWrapperLookup(featureSet, Utils.registryAccessThreadLocal.get()).toCommandRegAccessAccess(), key
                );
            } catch (Exception ex) {
                return class_7066.method_41170(key);
            }
        };
    }

    public void provide(class_5455 registryManager) {
        requireNonNull(registryManager);
        if (this.registry == null) this.registry = requireNonNull(registryManager.method_30530(this.provider.regKey()));
        T t = this.registry.method_63535(this.provider.entryId());
        if (t != null) this.set(t);
    }

    public T get() {
        return this.value;
    }

    public void set(T value, @Nullable MinecraftServer server) {
        this.bump(value, server);
    }

    private void bump(T value, MinecraftServer server) {
        boolean b = this.validator.validate(value);
        if (!b) {
            Optional<T> o = this.adapter.adapt(value);
            b = (o.isPresent() && this.validator.validate(value = o.get()));
        }
        if (b) {
            this.value = value;
            this.provider = requireNonNull(this.provider.withId(this.registry.method_10221(value)));
            this.method_20778(server);
        }
    }

    private void set(T t) {
        if (this.validator.validate(t)) {
            this.value = t;
            return;
        }
        Optional<T> o = this.adapter.adapt(t);
        if (o.isEmpty() || !this.validator.validate(o.get())) return;
        this.value = o.get();
    }

    public boolean validate(String input) {
        if (this.registry == null && FabricLoader.getInstance().getEnvironmentType() == EnvType.CLIENT) {
            this.registry = ClientUtils.getClientWorldRegistryOrThrow(this.provider.regKey());
        }
        if (this.registry == null) return false;
        Optional<T> o = parseInput(input, this.registry);
        if (o.isPresent() && this.validator.validate(o.get())) {
            this.value = o.get();
            return true;
        }
        return false;
    }

    private static <U> Optional<U> parseInput(String input, class_2378<U> registry) {
        class_2960 id = class_2960.method_12829(input);
        if (id == null || !registry.method_10250(id)) {
            return Optional.empty();
        }
        return Optional.ofNullable(registry.method_63535(id));
    }

    @Override
    protected void method_20776(CommandContext<class_2168> context, String name) {
        try {
            //noinspection unchecked
            class_6880.class_6883<T> entry = class_7733
                    .method_45602(context, name, (net.minecraft.class_5321<class_2378<T>>) this.registry.method_46765());
            T t = this.registry.method_63535(entry.method_40237().method_29177());
            this.set(requireNonNull(t));
        } catch (IllegalArgumentException e) {
            try {
                //noinspection unchecked
                class_7066.class_7068<T> predicate = class_7066
                        .method_41166(context, name, (class_5321<class_2378<T>>) this.registry.method_46765(), UNKNOWN_REGISTRY_ENTRY);
                class_6880.class_6883<T> entry = this.registry.method_42017().filter(predicate).findFirst()
                        .orElseThrow(() -> new IllegalArgumentException("No entry found"));
                this.set(entry.comp_349());
            } catch (CommandSyntaxException ex) {
                throw new IllegalArgumentException(e);
            }
        } catch (CommandSyntaxException|NullPointerException|IllegalStateException e) {
            throw new IllegalArgumentException(e);
        }
    }

    @Override
    protected void method_20777(String value) {
        if (this.registry == null) {
            return;
        }
        parseInput(value, this.registry).ifPresentOrElse(
                this::set,
                () -> {
                    if (this.registry != null) {
                        UnruledApi.LOGGER.warn("Unable to find an entry of {} matching '{}'", this.registry.method_46765(), value);
                    } else {
                        this.provider = this.provider.withId(class_2960.method_12829(value));
                    }
                }
        );
    }

    @Override
    public String method_20779() {
        if (this.registry == null) {
            return this.provider.entryId.toString();
        }
        return requireNonNull(this.registry.method_10221(this.value)).toString();
    }

    @Override
    public int method_20781() {
        return this.registry.method_10206(this.value);
    }

    @Override
    protected RegistryEntryRule<T> method_20782() {
        return this;
    }

    @Override
    public RegistryEntryRule<T> method_27338() {
        if (this.registry == null) {
            return new RegistryEntryRule<>(this.field_19417, this.provider.regKey, this.provider.entryId, this.validator, this.adapter);
        }
        return new RegistryEntryRule<>(this.field_19417, this.registry, this.value, this.validator, this.adapter);
    }

    @Override
    public void setValue(RegistryEntryRule<T> rule, @Nullable MinecraftServer server) {
        this.bump(rule.value, server);
    }

    @Override
    public IGameruleValidator<T> unruled_getValidator() {
        return this.validator;
    }

    @Override
    public void unruled_setValidator(IGameruleValidator<T> validator) {
        this.validator = requireNonNull(validator);
    }

    @Override
    public IGameruleAdapter<T> unruled_getAdapter() {
        return this.adapter;
    }

    @Override
    public void unruled_setAdapter(IGameruleAdapter<T> adapter) {
        this.adapter = requireNonNull(adapter);
    }

    public static class Builder <V> extends RuleBuilder<RegistryEntryRule<V>, V> {
        private final class_2378<V> registry;
        private final ProviderEntry<V> provider;
        public Builder(class_2378<V> registry, V initialValue) {
            super(argSupplierBuilder(registry.method_46765(), class_7699.method_45397()), Builder::acceptor, initialValue);
            this.registry = requireNonNull(registry);
            this.provider = null;
        }

        public Builder(class_5321<? extends class_2378<V>> regKey, class_2960 valueId) {
            super(argSupplierBuilder(regKey, class_7699.method_45397()), Builder::acceptor, null);
            this.registry = null;
            this.provider = new ProviderEntry<>(regKey, valueId);
        }

        @Override
        @NotNull
        protected RegistryEntryRule<V> ruleBuilder(class_4314<RegistryEntryRule<V>> type) {
            if (this.registry == null) {
                return new RegistryEntryRule<>(type, this.provider.regKey, this.provider.entryId, super.validator, super.adapter);
            }
            return new RegistryEntryRule<>(type, this.registry, super.initialValue, super.validator, super.adapter);
        }

        static <U> void acceptor(class_4311 consumer, class_4313<RegistryEntryRule<U>> key, class_4314<RegistryEntryRule<U>> type) {
            ((IGameRulesVisitor)consumer).unruled_visitRegistryEntry(key, type);
        }
    }

    private record ProviderEntry <T> (class_5321<? extends class_2378<T>> regKey, class_2960 entryId) {
        ProviderEntry<T> withId(class_2960 id) {
            return new ProviderEntry<>(this.regKey, id);
        }
    }
}
