package mc.recraftors.unruled_api;

import com.mojang.brigadier.LiteralMessage;
import com.mojang.brigadier.exceptions.DynamicCommandExceptionType;
import com.mojang.datafixers.DSL;
import mc.recraftors.unruled_api.rules.*;
import mc.recraftors.unruled_api.utils.EncapsulatedException;
import mc.recraftors.unruled_api.utils.IGameRulesProvider;
import mc.recraftors.unruled_api.utils.ServerBoundAccessor;
import net.minecraft.class_1208;
import net.minecraft.class_1928;
import net.minecraft.class_1928.class_4310;
import net.minecraft.class_1928.class_4312;
import net.minecraft.class_1928.class_4313;
import net.minecraft.class_1928.class_4314;
import net.minecraft.class_1928.class_4315;
import net.minecraft.class_1928.class_5198;
import net.minecraft.class_2300;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.server.MinecraftServer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;

import java.util.function.BiConsumer;

import static net.minecraft.class_1928.*;

/**
 * Main API class for creating gamerules beyond the vanilla scope.
 * <p>
 * Use the {@code create<Type>} static methods in order to create your gamerules.
 * <p>
 * Warning, these methods will <i>create</i> the gamerules, but not register them.
 */
@SuppressWarnings({"unused", "UnusedReturnValue"})
public class UnruledApi {
	public static final String MOD_ID = "unruled_api";
	public static final Logger LOGGER = LogManager.getLogger(MOD_ID);

	public static final DynamicCommandExceptionType INVALID_INT = new DynamicCommandExceptionType(v -> new LiteralMessage("Invalid integer '"+v+"'"));
	public static final DSL.TypeReference GAMERULES_TYPE = class_1208.method_59518("gamerules");

	private static int strLen(String s, int i) {
		return Math.max(i, (int) Math.ceil(s.length() / 8d) * 8);
	}

	/**
	 * Registers the provided gamerule in the specified category and with the specified name,
	 * and returns the registered rule's key.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param type The type of gamerule to register.
	 * @return The new rule's registration key.
	 * @param <T> The gamerule's type.
	 */
	public static <T extends class_4315<T>> class_4313<T> register(String name, class_5198 category, class_4314<T> type) {
		return class_1928.method_8359(name, category, type);
	}

	/**
	 * Creates a vanilla Boolean gamerule with the provided default value and change callback.
	 * @param initialValue The new gamerule's default value.
	 * @param changeCallback The new gamerule's change callback.
	 * @return The newly created gamerule setting.
	 */
	@Contract(value = "_, _ -> new", pure = true)
	@NotNull public static class_4314<class_4310> createBoolean(
			boolean initialValue, BiConsumer<MinecraftServer, class_4310> changeCallback
	) {
		return class_4310.method_20760(initialValue, changeCallback);
	}

	/**
	 * Creates a possibly server-bound vanilla Boolean gamerule with the provided default value and change callback.
	 * @param initialValue The new gamerule's default value.
	 * @param changeCallback The new gamerule's change callback.
	 * @param serverBound Whether the new gamerule should be server-bound.
	 * @return The newly created gamerule setting.
	 */
	@Contract(value = "_, _, _ -> new", pure = true)
	@NotNull public static class_4314<class_4310> createBoolean(
			boolean initialValue, BiConsumer<MinecraftServer, class_4310> changeCallback, boolean serverBound
	) {
		class_4314<class_4310> type = class_4310.method_20760(initialValue, changeCallback);
		((ServerBoundAccessor)type).unruled_setServerBound(serverBound);
		return type;
	}

	/**
	 * Creates a vanilla Boolean gamerule with the provided default value.
	 * @param initialValue The new gamerule's default value.
	 * @return The newly created gamerule setting.
	 */
	@Contract(value = "_ -> new", pure = true)
	@NotNull public static class_4314<class_4310> createBoolean(boolean initialValue) {
		return class_4310.method_20759(initialValue);
	}

	/**
	 * Creates a possibly server-bound vanilla Boolean gamerule with the provided default value.
	 * @param initialValue The new gamerule's default value.
	 * @param serverBound Whether the new gamerule should be server-bound.
	 * @return The newly created gamerule setting.
	 */
	@Contract(value = "_, _ -> new", pure = true)
	@NotNull public static class_4314<class_4310> createBoolean(boolean initialValue, boolean serverBound) {
		class_4314<class_4310> type = class_4310.method_20759(initialValue);
		((ServerBoundAccessor)type).unruled_setServerBound(serverBound);
		return type;
	}

	/**
	 * Creates and registers a vanilla Boolean gamerule with the provided default
	 * value and change callback, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param initialValue The new gamerule's default value.
	 * @param changeCallback The new gamerule's change callback.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _, _ -> new")
	@NotNull public static class_4313<class_4310> registerBoolean(
			String name, class_5198 category, boolean initialValue, BiConsumer<MinecraftServer, class_4310> changeCallback
	) {
		return register(name, category, createBoolean(initialValue, changeCallback));
	}

	/**
	 * Creates and registers a possibly server-bound vanilla Boolean gamerule with the
	 * provided default value and change callback, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param initialValue The new gamerule's default value.
	 * @param changeCallback The new gamerule's change callback.
	 * @param serverBound Whether the new gamerule should be server-bound.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _, _, _ -> new")
	@NotNull public static class_4313<class_4310> registerBoolean(
			String name, class_5198 category, boolean initialValue, BiConsumer<MinecraftServer, class_4310> changeCallback,
			boolean serverBound
	) {
		return register(name, category, createBoolean(initialValue, changeCallback, serverBound));
	}

	/**
	 * Creates and registers a vanilla Boolean gamerule with the provided default
	 * value, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param initialValue The new gamerule's default value.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _ -> new")
	@NotNull public static class_4313<class_4310> registerBoolean(String name, class_5198 category, boolean initialValue) {
		return register(name, category, createBoolean(initialValue));
	}

	/**
	 * Creates and registers a possibly server-bound vanilla Boolean gamerule with the
	 * provided default value, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param initialValue The new gamerule's default value.
	 * @param serverBound Whether the new gamerule should be server-bound.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _, _ -> new")
	@NotNull public static class_4313<class_4310> registerBoolean(String name, class_5198 category, boolean initialValue, boolean serverBound) {
		return register(name, category, createBoolean(initialValue, serverBound));
	}

	/**
	 * Creates a vanilla integer gamerule with the provided default value
	 * and change callback.
	 * @param initialValue The new gamerule's default value.
	 * @param changeCallback The new gamerule's change callback.
	 * @return The newly created gamerule setting.
	 * @apiNote Better use {@link #registerInt(String, class_5198, int, BiConsumer)} directly.
	 */
	@Contract(value = "_, _ -> new", pure = true)
	@NotNull public static class_4314<class_4312> createInt(int initialValue, BiConsumer<MinecraftServer, class_4312> changeCallback
	) {
		return class_4312.method_20766(initialValue, changeCallback);
	}

	/**
	 * Creates a vanilla integer gamerule with the provided default value.
	 * @param initialValue The new gamerule's default value.
	 * @return The newly created gamerule setting.
	 * @apiNote Better use {@link #registerInt(String, class_5198, int)} directly.
	 */
	@Contract(value = "_ -> new", pure = true)
	@NotNull public static class_4314<class_4312> createInt(int initialValue) {
		return class_4312.method_20768(initialValue);
	}

	/**
	 * Creates and registers a new vanilla integer gamerule with the
	 * provided default value and change callback, at the specified
	 * name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param initialValue The new gamerule's default value.
	 * @param changeCallback The new gamerule's change callback.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _, _ -> new")
	@NotNull public static class_4313<class_4312> registerInt(
			String name, class_5198 category, int initialValue, BiConsumer<MinecraftServer, class_4312> changeCallback
	) {
		return register(name, category, createInt(initialValue, changeCallback));
	}

	/**
	 * Creates and registers a new vanilla integer gamerule with the
	 * provided default value, at the specified name and in the
	 * specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param initialValue The new gamerule's default value.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _ -> new")
	@NotNull public static class_4313<class_4312> registerInt(String name, class_5198 category, int initialValue) {
		return register(name, category, createInt(initialValue));
	}

	public static RuleBuilder<class_4312, Integer> intRuleBuilder(int initialValue) {
		return new RuleBuilder.IntRuleBuilder(initialValue);
	}

	/**
	 * Creates and registers a new double gamerule with the provided
	 * default value, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param initialValue The new gamerule's default value.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _ -> new")
	@NotNull public static class_4313<DoubleRule> registerDoubleRule(String name, class_5198 category, double initialValue) {
		return doubleRuleBuilder(initialValue).register(name, category);
	}

	/**
	 * Creates and registers a new double gamerule with the provided
	 * default value, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param initialValue The new gamerule's default value.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _ -> new")
	@NotNull public static class_4313<EntitySelectorRule> registerEntitySelectorRule(
			String name, class_5198 category, String initialValue
	) {
		return entitySelectorRuleBuilder(initialValue).register(name, category);
	}

	/**
	 * Creates and registers a new double gamerule with the provided
	 * default value, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param enumClass The enum class to link the new rule to.
	 * @param initialValue The new gamerule's default value.
	 * @param <T> The enum type of the rule.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _, _ -> new")
	@NotNull public static <T extends Enum<T>> class_4313<EnumRule<T>> registerEnumRule(
			String name, class_5198 category, Class<T> enumClass, T initialValue
	) {
		return enumRuleBuilder(enumClass, initialValue).register(name, category);
	}

	/**
	 * Creates and registers a new double gamerule with the provided
	 * default value, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param initialValue The new gamerule's default value.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _ -> new")
	@NotNull public static class_4313<FloatRule> registerFloatRule(String name, class_5198 category, float initialValue) {
		return floatRuleBuilder(initialValue).register(name, category);
	}

	/**
	 * Creates and registers a new double gamerule with the provided
	 * default value, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param initialValue The new gamerule's default value.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _ -> new")
	@NotNull public static class_4313<LongRule> registerLongRule(String name, class_5198 category, long initialValue) {
		return longRuleBuilder(initialValue).register(name, category);
	}

	/**
	 * Creates and registers a new double gamerule with the provided
	 * default value, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param registry The registry to which to attach the new rule.
	 * @param initialValue The new gamerule's default value.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _, _ -> new")
	@NotNull public static <T> class_4313<RegistryEntryRule<T>> registerRegistryEntryRule(
			String name, class_5198 category, class_2378<T> registry, T initialValue
	) {
		return registryEntryRuleBuilder(registry, initialValue).register(name, category);
	}

	/**
	 * Creates and registers a new double gamerule with the provided
	 * default value, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param maxLength The max length of the gamerule.
	 *                  Must be between 1 and 128 included.
	 * @param initialValue The new gamerule's default value.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _, _ -> new")
	@NotNull public static class_4313<StringRule> registerStringRule(
			String name, class_5198 category, String initialValue, int maxLength
	) {
		return stringRuleBuilder(initialValue, maxLength).register(name, category);
	}

	/**
	 * Creates and registers a new double gamerule with the provided
	 * default value, at the specified name and in the specified category.
	 * @param name The name of the rule to register.
	 * @param category The category in which to register the gamerule.
	 * @param initialValue The new gamerule's default value.
	 * @return The new rule's registration key.
	 */
	@Contract("_, _, _ -> new")
	@NotNull public static class_4313<TextRule> registerTextRule(String name, class_5198 category, String initialValue) {
		return textRuleBuilder(initialValue).register(name, category);
	}

	/**
	 * Creates a new double gamerule builder with the provided
	 * default value.
	 * @param initialValue The builder's built rule's default value.
	 * @return The new gamerule builder.
	 */
	public static DoubleRule.Builder doubleRuleBuilder(double initialValue) {
		return new DoubleRule.Builder(initialValue);
	}

	/**
	 * Creates a new entity selector gamerule builder with the
	 * provided default value.
	 * @param initialValue The builder's built rule's default value.
	 * @return The new gamerule builder.
	 * @throws EncapsulatedException If the provided default value
	 *         cannot be parsed to an entity selector.
	 */
	public static EntitySelectorRule.Builder entitySelectorRuleBuilder(String initialValue)
			throws EncapsulatedException {
		return new EntitySelectorRule.Builder(initialValue);
	}

	/**
	 * Creates a new enum gamerule builder for the provided
	 * enum class with the specified default value.
	 * @param enumClass The enum class which to base the builder on.
	 * @param initialValue The builder's built rule's default value.
	 * @return The new gamerule builder.
	 * @param <T> The enum type of the new builder.
	 */
	public static <T extends Enum<T>> EnumRule.Builder<T> enumRuleBuilder(Class<T> enumClass, T initialValue) {
		return new EnumRule.Builder<>(enumClass, initialValue);
	}

	/**
	 * Creates a new float gamerule builder with the provided
	 * default value.
	 * @param initialValue The builder's built rule's default value.
	 * @return The new gamerule builder.
	 */
	public static FloatRule.Builder floatRuleBuilder(float initialValue) {
		return new FloatRule.Builder(initialValue);
	}

	/**
	 * Creates a new long gamerule builder with the provided
	 * default value.
	 * @param initialValue The builder's built rule's default value.
	 * @return The new gamerule builder.
	 */
	public static LongRule.Builder longRuleBuilder(long initialValue) {
		return new LongRule.Builder(initialValue);
	}

	/**
	 * Creates a new registry entry gamerule builder for the
	 * provided registry with the specified default value.
	 * @param registry The registry to base the builder on.
	 * @param initialValue The builder's built rule's default value.
	 * @return The new gamerule builder.
	 * @param <T> The type of the provided registry's values.
	 */
	public static <T> RegistryEntryRule.Builder<T> registryEntryRuleBuilder(class_2378<T> registry, T initialValue) {
		return new RegistryEntryRule.Builder<>(registry, initialValue);
	}

	/**
	 * Creates a new registry entry gamerule builder for the
	 * provided registry key with the specified default value key.
	 * It allows to fetch registries at a later point in the game's
	 * lifecycle, such as for dynamic registries.
	 * (e.g. structures, dimensions, etc.)
	 * @param key The key of the registry to base the builder on.
	 * @param valueId The ID of the built rule's default value.
	 * @return The new gamerule builder.
	 * @param <T> The type of the provided registry's values.
	 */
	public static <T> RegistryEntryRule.Builder<T> registryEntryRuleBuilder(
			class_5321<? extends class_2378<T>> key, class_2960 valueId
	) {
		return new RegistryEntryRule.Builder<>(key, valueId);
	}

	/**
	 * Creates a new string gamerule builder with the provided
	 * default value and max length.
	 * @param initialValue The builder's built rule's default value.
	 * @param maxLength The builder's built rule's max length. Must
	 *                  be between 0 excluded and 128 included.
	 * @return The new gamerule builder.
	 * @throws UnsupportedOperationException If the specified max length
	 * 	       is outside the [1 ; 128] interval.
	 */
	public static StringRule.Builder stringRuleBuilder(String initialValue, int maxLength) throws UnsupportedOperationException {
		return new StringRule.Builder(initialValue, maxLength);
	}

	/**
	 * Creates a new text gamerule builder with the provided
	 * default value.
	 * @param initialValue The builder's built rule's default value.
	 * @return The new gamerule builder.
	 */
	public static TextRule.Builder textRuleBuilder(String initialValue) {
		return new TextRule.Builder(initialValue);
	}

	public static boolean getBoolean(class_1928 gameRules, class_4313<class_4310> key) {
		return gameRules.method_8355(key);
	}

	public static int getInt(class_1928 gamerules, class_4313<class_4312> key) {
		return gamerules.method_8356(key);
	}

	public static long getLong(class_1928 gameRules, class_4313<LongRule> key) {
		return ((IGameRulesProvider)gameRules).unruled_getLong(key);
	}

	public static float getFloat(class_1928 gameRules, class_4313<FloatRule> key) {
		return ((IGameRulesProvider)gameRules).unruled_getFloat(key);
	}

	public static double getDouble(class_1928 gameRules, class_4313<DoubleRule> key) {
		return ((IGameRulesProvider)gameRules).unruled_getDouble(key);
	}

	public static <T extends Enum<T>> T getEnum(class_1928 gameRules, class_4313<EnumRule<T>> key) {
		return ((IGameRulesProvider)gameRules).unruled_getEnum(key);
	}

	public static String getString(class_1928 gameRules, class_4313<StringRule> key) {
		return ((IGameRulesProvider)gameRules).unruled_getString(key);
	}

	public static String getText(class_1928 gameRules, class_4313<TextRule> key) {
		return ((IGameRulesProvider)gameRules).unruled_getText(key);
	}

	public static class_2300 getEntitySelector(class_1928 gameRules, class_4313<EntitySelectorRule> key) {
		return ((IGameRulesProvider)gameRules).unruled_getEntitySelector(key);
	}

	public static <T> T getRegistryEntry(class_1928 gameRules, class_4313<RegistryEntryRule<T>> key) {
		return ((IGameRulesProvider)gameRules).unruled_getRegistryEntry(key);
	}
}
