package mc.recraftors.unruled_api.mixin;

import com.llamalad7.mixinextras.injector.wrapoperation.Operation;
import com.llamalad7.mixinextras.injector.wrapoperation.WrapOperation;
import com.llamalad7.mixinextras.sugar.Share;
import com.llamalad7.mixinextras.sugar.ref.LocalRef;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import mc.recraftors.unruled_api.rules.OverridesManager;
import mc.recraftors.unruled_api.utils.EncapsulatedException;
import mc.recraftors.unruled_api.utils.IGameruleOverridesProvider;
import mc.recraftors.unruled_api.utils.LangFallbacks;
import mc.recraftors.unruled_api.utils.Utils;
import net.minecraft.command.argument.DimensionArgumentType;
import net.minecraft.server.command.CommandManager;
import net.minecraft.server.command.GameRuleCommand;
import net.minecraft.server.command.ServerCommandSource;
import net.minecraft.server.world.ServerWorld;
import net.minecraft.text.Text;
import net.minecraft.world.GameRules;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;

import java.util.Set;

@Mixin(GameRuleCommand.class)
public abstract class GameRuleCommandMixin {
    @Inject(method = "register", at = @At("HEAD"))
    private static void setupOverridesCommand(
            CommandDispatcher<ServerCommandSource> dispatcher, CallbackInfo ci,
            @Share("gamerule-overrides") LocalRef<LiteralArgumentBuilder<ServerCommandSource>> ref
    ) {
		LiteralArgumentBuilder<ServerCommandSource> builder = CommandManager.literal("gamerule-override")
			.requires(source -> source.hasPermissionLevel(2));
        ref.set(builder);
        Utils.argumentBuilderThreadLocal.set(builder);
        builder.executes(GameRuleCommandMixin::unruled_listOverridesSimple);
        builder.then(CommandManager.argument("dimension", DimensionArgumentType.dimension())
                .executes(GameRuleCommandMixin::unruled_listOverridesInWorld));
    }

    @Inject(method = "register", at = @At("TAIL"))
    private static void finallizeOverridesCommand(
            CommandDispatcher<ServerCommandSource> dispatcher, CallbackInfo ci,
            @Share("gamerule-overrides") LocalRef<LiteralArgumentBuilder<ServerCommandSource>> ref
    ) {
        dispatcher.register(ref.get());
        Utils.argumentBuilderThreadLocal.remove();
    }

    @WrapOperation(
            method = "executeSet",
            at = @At(
                    value = "INVOKE",
                    target = "Lnet/minecraft/world/GameRules$Rule;set(Lcom/mojang/brigadier/context/CommandContext;Ljava/lang/String;)V"
            )
    )
    private static void executeSetGameRuleSetWrapper(
            GameRules.Rule<?> instance, CommandContext<ServerCommandSource> context, String name,
            Operation<Void> original
    ) throws Exception {
        try {
            original.call(instance, context, name);
        } catch (EncapsulatedException ex) {
            throw ex.exception;
        }
    }

    @Unique
    private static int unruled_listOverridesSimple(CommandContext<ServerCommandSource> context) {
        return unruled_listOverrides(context, context.getSource().getWorld());
    }

    @Unique
    private static int unruled_listOverridesInWorld(CommandContext<ServerCommandSource> context) throws CommandSyntaxException {
        return unruled_listOverrides(context, DimensionArgumentType.getDimensionArgument(context, "dimension"));
    }

    @Unique
    private static int unruled_listOverrides(
            CommandContext<ServerCommandSource> context, ServerWorld world
    ) {
        OverridesManager overrides = ((IGameruleOverridesProvider)world).unruled_getOverridesManager();
        Set<GameRules.Key<?>> set = overrides.getOverrides().getOverrides();
        StringBuilder sb = new StringBuilder();
        set.forEach(k -> sb.append("\n - ").append(k.getName()).append(": ").append(overrides.get(k).serialize()));
        context.getSource().sendFeedback(() -> Text.translatableWithFallback(
                "commands.gamerule_override.list",
                LangFallbacks.OVERRIDE_LIST.format(world.getDimensionKey().getValue(), set.size(), sb.toString()),
                world.getDimensionKey(), set.size(), sb.toString()), false);
        return set.size();
    }
}
