package mc.recraftors.unruled_api.mixin;

import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import mc.recraftors.unruled_api.rules.OverridesManager;
import mc.recraftors.unruled_api.utils.IGameruleOverridesProvider;
import mc.recraftors.unruled_api.utils.LangFallbacks;
import mc.recraftors.unruled_api.utils.Utils;
import net.minecraft.command.argument.DimensionArgumentType;
import net.minecraft.server.command.CommandManager;
import net.minecraft.server.command.ServerCommandSource;
import net.minecraft.server.world.ServerWorld;
import net.minecraft.text.Text;
import net.minecraft.util.Identifier;
import net.minecraft.world.GameRules;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;

import static net.minecraft.server.command.CommandManager.literal;

@Mixin(targets = "net/minecraft/server/command/GameRuleCommand$1")
public abstract class GameruleCommandVisitorMixin {
    @Inject(method = "visit", at = @At("TAIL"))
    private <T extends GameRules.Rule<T>> void setupOverridesSubArgs(
            GameRules.Key<T> key, GameRules.Type<T> type, CallbackInfo ci
    ) {
        LiteralArgumentBuilder<ServerCommandSource> builder = Utils.argumentBuilderThreadLocal.get();
        if (builder == null) {
            return;
        }
        builder.then(literal("get").then(literal(key.getName()).executes(c -> unruled_queryOverrideSimple(c, key))));
        builder.then(literal("set").then(literal(key.getName()).then(type.argument("value")
                .executes(c -> unruled_setOverrideSimple(c, key)))));
        builder.then(literal("unset").then(literal(key.getName()).executes(c -> unruled_unsetOverrideSimple(c, key))));

        builder.then(literal("in").then(CommandManager.argument("dimension", DimensionArgumentType.dimension())
                .then(literal("get").then(literal(key.getName()).executes(c -> unruled_queryOverrideInWorld(c, key))))
                .then(literal("set").then(literal(key.getName()).then(type.argument("value")
                        .executes(c -> unruled_setOverrideInWorld(c, key)))))
                .then(literal("unset").then(literal(key.getName()).executes(c -> unruled_unsetOverrideInWorld(c, key))))
        ));
    }

    @Unique
    private static <T extends GameRules.Rule<T>> int unruled_queryOverrideSimple(
            CommandContext<ServerCommandSource> context, GameRules.Key<T> key
    ) {
        ServerWorld world = context.getSource().getWorld();
        return unruled_queryOverride(context, key, world);
    }

    @Unique
    private static <T extends GameRules.Rule<T>> int unruled_queryOverrideInWorld(
            CommandContext<ServerCommandSource> context, GameRules.Key<T> key
    ) throws CommandSyntaxException {
        return unruled_queryOverride(context, key, DimensionArgumentType.getDimensionArgument(context, "dimension"));
    }

    @Unique
    private static <T extends GameRules.Rule<T>> int unruled_setOverrideSimple(
            CommandContext<ServerCommandSource> context, GameRules.Key<T> key
    ) {
        ServerWorld world = context.getSource().getWorld();
        return unruled_setOverride(context, key, world);
    }

    @Unique
    private static <T extends GameRules.Rule<T>> int unruled_setOverrideInWorld(
            CommandContext<ServerCommandSource> context, GameRules.Key<T> key
    ) throws CommandSyntaxException {
        return unruled_setOverride(context, key, DimensionArgumentType.getDimensionArgument(context, "dimension"));
    }

    @Unique
    private static <T extends GameRules.Rule<T>> int unruled_unsetOverrideSimple(
            CommandContext<ServerCommandSource> context, GameRules.Key<T> key
    ) {
        ServerWorld world = context.getSource().getWorld();
        return unruled_unsetOverride(context, key, world);
    }

    @Unique
    private static <T extends GameRules.Rule<T>> int unruled_unsetOverrideInWorld(
            CommandContext<ServerCommandSource> context, GameRules.Key<T> key
    ) throws CommandSyntaxException {
        return unruled_unsetOverride(context, key, DimensionArgumentType.getDimensionArgument(context, "dimension"));
    }

    @Unique
    private static <T extends GameRules.Rule<T>> int unruled_queryOverride(
            CommandContext<ServerCommandSource> context, GameRules.Key<T> key, ServerWorld world
    ) {
        OverridesManager manager = ((IGameruleOverridesProvider)world).unruled_getOverridesManager();
        String keyName = key.getName();
        Identifier worldId = world.getDimensionKey().getValue();
        if (!manager.hasOverride(key)) {
            context.getSource().sendFeedback(() -> Text.translatableWithFallback(
                    "commands.gamerule_override.none",
                    LangFallbacks.OVERRIDE_NONE.format(keyName, worldId),
                    keyName, worldId), false);
            return 0;
        }
        T rule = manager.get(key);
        context.getSource().sendFeedback(() -> Text.translatableWithFallback(
                "commands.gamerule_override.query",
                LangFallbacks.OVERRIDE_QUERY.format(keyName, worldId, rule.toString()),
                keyName, worldId, rule.toString()), false);
        return rule.getCommandResult();
    }

    @Unique
    private static <T extends GameRules.Rule<T>> int unruled_setOverride(
            CommandContext<ServerCommandSource> context, GameRules.Key<T> key, ServerWorld world
    ) {
        OverridesManager manager = ((IGameruleOverridesProvider)world).unruled_getOverridesManager();
        String keyName = key.getName();
        Identifier worldId = world.getDimensionKey().getValue();
        manager.override(key, context);
        T rule = manager.get(key);
        context.getSource().sendFeedback(() -> Text.translatableWithFallback(
                "commands.gamerule_override.set",
                LangFallbacks.OVERRIDE_SET.format(keyName, worldId, rule.toString()),
                keyName, worldId, rule.toString()), true);
        return rule.getCommandResult();
    }

    @Unique
    private static <T extends GameRules.Rule<T>> int unruled_unsetOverride(
            CommandContext<ServerCommandSource> context, GameRules.Key<T> key, ServerWorld world
    ) {
        OverridesManager manager = ((IGameruleOverridesProvider)world).unruled_getOverridesManager();
        String keyName = key.getName();
        Identifier worldId = world.getDimensionKey().getValue();
        boolean b = manager.removeOverride(key);
        String tr = "command.gamerule_override." + (b ? "unset" : "none");
        LangFallbacks fb = b ? LangFallbacks.OVERRIDE_UNSET : LangFallbacks.OVERRIDE_NONE;
        T rule = manager.get(key);
        context.getSource().sendFeedback(() -> Text.translatableWithFallback(
                tr, fb.format(keyName, worldId), keyName, worldId), true);
        return rule.getCommandResult();
    }
}
