package mc.recraftors.unruled_api.rules;

import com.mojang.brigadier.context.CommandContext;
import net.minecraft.nbt.NbtCompound;
import net.minecraft.nbt.NbtElement;
import net.minecraft.nbt.NbtString;
import net.minecraft.server.command.ServerCommandSource;
import net.minecraft.world.GameRules;
import net.minecraft.world.PersistentState;

import java.util.HashMap;
import java.util.Map;

public class OverridesManager extends PersistentState {
    private final GameRulesOverrides overrides;
    private final Map<String, String> rawMap;

    public static PersistentState.Type<OverridesManager> getPersistentStateType(GameRules rules) {
        return new Type<>(() -> new OverridesManager(rules), nbt -> fromNbt(nbt, rules), null);
    }

    private OverridesManager(GameRules base, Map<String, String> rawMap) {
        this.overrides = new GameRulesOverrides(base, rawMap);
        this.rawMap = rawMap;
    }

    private OverridesManager(GameRules base) {
        this(base, new HashMap<>());
    }

    private static OverridesManager fromNbt(NbtCompound nbt, GameRules base) {
        Map<String, String> map = new HashMap<>();
        nbt.getKeys().forEach(s -> {
            NbtElement e = nbt.get(s);
            if (!(e instanceof NbtString ns)) return;
            map.put(s, ns.asString());
        });
        return new OverridesManager(base, map);
    }

    @Override
    public NbtCompound writeNbt(NbtCompound nbt) {
        this.cache();
        this.rawMap.forEach(nbt::putString);
        return nbt;
    }

    @Override
    public boolean isDirty() {
        this.cache();
        return super.isDirty();
    }

    public <T extends GameRules.Rule<T>> boolean hasOverride(GameRules.Key<T> key) {
        return this.overrides.hasOverride(key);
    }

    public <T extends GameRules.Rule<T>> void override(GameRules.Key<T> key, CommandContext<ServerCommandSource> context) {
        this.overrides.override(key, context);
        this.markDirty();
    }

    public <T extends GameRules.Rule<T>> boolean removeOverride(GameRules.Key<T> key) {
        if (this.overrides.removeOverride(key)) {
            this.markDirty();
            this.rawMap.remove(key.getName());
            return true;
        }
        return false;
    }

    public <T extends GameRules.Rule<T>> T get(GameRules.Key<T> key) {
        return this.overrides.get(key);
    }

    private void cache() {
        this.overrides.getOverrides().forEach(k -> {
            String s = k.getName();
            String v = this.overrides.get(k).serialize();
            if (!this.rawMap.getOrDefault(s, "").equals(v)) {
                this.rawMap.put(s, v);
            }
        });
    }

    public boolean isEmpty() {
        return this.overrides == null || this.overrides.isEmpty();
    }

    public GameRulesOverrides getOverrides() {
        return this.overrides;
    }
}
