package mc.recraftors.unruled_api.rules;

import com.mojang.brigadier.arguments.ArgumentType;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.DynamicCommandExceptionType;
import mc.recraftors.unruled_api.UnruledApi;
import mc.recraftors.unruled_api.impl.FullRegistryWrapperLookup;
import mc.recraftors.unruled_api.utils.*;
import net.fabricmc.api.EnvType;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.command.argument.RegistryEntryReferenceArgumentType;
import net.minecraft.command.argument.RegistryPredicateArgumentType;
import net.minecraft.registry.DynamicRegistryManager;
import net.minecraft.registry.Registry;
import net.minecraft.registry.RegistryKey;
import net.minecraft.registry.entry.RegistryEntry;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.command.ServerCommandSource;
import net.minecraft.text.Text;
import net.minecraft.util.Identifier;
import net.minecraft.world.GameRules.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Optional;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

public class RegistryEntryRule <T> extends Rule<RegistryEntryRule<T>> implements GameruleAccessor<T> {
    private static final DynamicCommandExceptionType UNKNOWN_REGISTRY_ENTRY = new DynamicCommandExceptionType(
            id -> Text.translatableWithFallback("unruled.commands.unknown_registry_entry", "Unknown entry %s in registry", id)
    );

    private ProviderEntry<T> provider;
    private Registry<T> registry;
    private T value;
    private IGameruleValidator<T> validator;
    private IGameruleAdapter<T> adapter;

    public RegistryEntryRule(Type<RegistryEntryRule<T>> type, Registry<T> registry, T initialValue,
                             IGameruleValidator<T> validator, IGameruleAdapter<T> adapter) throws UnsupportedOperationException {
        super(type);
        if (registry.getId(initialValue) == null) {
            throw new UnsupportedOperationException("initial value must be a registry member");
        }
        this.registry = requireNonNull(registry);
        this.provider = new ProviderEntry<>(registry.getKey(), requireNonNull(registry.getId(initialValue)));
        this.value = requireNonNull(initialValue);
        this.validator = requireNonNull(validator);
        this.adapter = requireNonNull(adapter);
    }

    public RegistryEntryRule(Type<RegistryEntryRule<T>> type, RegistryKey<? extends Registry<T>> registryKey,
                             Identifier valueKey, IGameruleValidator<T> validator, IGameruleAdapter<T> adapter) {
        super(type);
        this.registry = null;
        this.provider = new ProviderEntry<>(requireNonNull(registryKey), requireNonNull(valueKey));
        this.validator = requireNonNull(validator);
        this.adapter = requireNonNull(adapter);
    }

    @SuppressWarnings("unused")
    public RegistryEntryRule(Type<RegistryEntryRule<T>> type, Registry<T> registry, T initialValue) {
        this(type, registry, initialValue, IGameruleValidator::alwaysTrue, Optional::of);
    }

    private static <V> Supplier<ArgumentType<?>> argSupplierBuilder(RegistryKey<? extends Registry<V>> key) {
        return () -> {
            try {
                return RegistryEntryReferenceArgumentType.registryEntry(
                        new FullRegistryWrapperLookup(Utils.registryAccessThreadLocal.get()).toCommandRegAccessAccess(), key
                );
            } catch (Exception ex) {
                return RegistryPredicateArgumentType.registryPredicate(key);
            }
        };
    }

    public void provide(DynamicRegistryManager registryManager) {
        requireNonNull(registryManager);
        if (this.registry == null) this.registry = requireNonNull(registryManager.get(this.provider.regKey()));
        T t = this.registry.get(this.provider.entryId());
        if (t != null) this.set(t);
    }

    public T get() {
        return this.value;
    }

    public void set(T value, @Nullable MinecraftServer server) {
        this.bump(value, server);
    }

    private void bump(T value, MinecraftServer server) {
        boolean b = this.validator.validate(value);
        if (!b) {
            Optional<T> o = this.adapter.adapt(value);
            b = (o.isPresent() && this.validator.validate(value = o.get()));
        }
        if (b) {
            this.value = value;
            this.provider = requireNonNull(this.provider.withId(this.registry.getId(value)));
            this.changed(server);
        }
    }

    private void set(T t) {
        if (this.validator.validate(t)) {
            this.value = t;
            return;
        }
        Optional<T> o = this.adapter.adapt(t);
        if (o.isEmpty() || !this.validator.validate(o.get())) return;
        this.value = o.get();
    }

    public boolean validate(String input) {
        if (this.registry == null && FabricLoader.getInstance().getEnvironmentType() == EnvType.CLIENT) {
            this.registry = ClientUtils.getClientWorldRegistryOrThrow(this.provider.regKey());
        }
        if (this.registry == null) return false;
        Optional<T> o = parseInput(input, this.registry);
        if (o.isPresent() && this.validator.validate(o.get())) {
            this.value = o.get();
            return true;
        }
        return false;
    }

    private static <U> Optional<U> parseInput(String input, Registry<U> registry) {
        Identifier id = Identifier.tryParse(input);
        if (id == null || !registry.containsId(id)) {
            return Optional.empty();
        }
        return Optional.ofNullable(registry.get(id));
    }

    @Override
    protected void setFromArgument(CommandContext<ServerCommandSource> context, String name) {
        try {
            //noinspection unchecked
            RegistryEntry.Reference<T> entry = RegistryEntryReferenceArgumentType
                    .getRegistryEntry(context, name, (net.minecraft.registry.RegistryKey<Registry<T>>) this.registry.getKey());
            T t = this.registry.get(entry.registryKey().getValue());
            this.set(requireNonNull(t));
        } catch (IllegalArgumentException e) {
            try {
                //noinspection unchecked
                RegistryPredicateArgumentType.RegistryPredicate<T> predicate = RegistryPredicateArgumentType
                        .getPredicate(context, name, (RegistryKey<Registry<T>>) this.registry.getKey(), UNKNOWN_REGISTRY_ENTRY);
                RegistryEntry.Reference<T> entry = this.registry.streamEntries().filter(predicate).findFirst()
                        .orElseThrow(() -> new IllegalArgumentException("No entry found"));
                this.set(entry.value());
            } catch (CommandSyntaxException ex) {
                throw new IllegalArgumentException(e);
            }
        } catch (CommandSyntaxException|NullPointerException|IllegalStateException e) {
            throw new IllegalArgumentException(e);
        }
    }

    @Override
    protected void deserialize(String value) {
        if (this.registry == null) {
            return;
        }
        parseInput(value, this.registry).ifPresentOrElse(
                this::set,
                () -> {
                    if (this.registry != null) {
                        UnruledApi.LOGGER.warn("Unable to find an entry of {} matching '{}'", this.registry.getKey(), value);
                    } else {
                        this.provider = this.provider.withId(Identifier.tryParse(value));
                    }
                }
        );
    }

    @Override
    public String serialize() {
        if (this.registry == null) {
            return this.provider.entryId.toString();
        }
        return requireNonNull(this.registry.getId(this.value)).toString();
    }

    @Override
    public int getCommandResult() {
        return this.registry.getRawId(this.value);
    }

    @Override
    protected RegistryEntryRule<T> getThis() {
        return this;
    }

    @Override
    public RegistryEntryRule<T> copy() {
        if (this.registry == null) {
            return new RegistryEntryRule<>(this.type, this.provider.regKey, this.provider.entryId, this.validator, this.adapter);
        }
        return new RegistryEntryRule<>(this.type, this.registry, this.value, this.validator, this.adapter);
    }

    @Override
    public void setValue(RegistryEntryRule<T> rule, @Nullable MinecraftServer server) {
        this.bump(rule.value, server);
    }

    @Override
    public IGameruleValidator<T> unruled_getValidator() {
        return this.validator;
    }

    @Override
    public void unruled_setValidator(IGameruleValidator<T> validator) {
        this.validator = requireNonNull(validator);
    }

    @Override
    public IGameruleAdapter<T> unruled_getAdapter() {
        return this.adapter;
    }

    @Override
    public void unruled_setAdapter(IGameruleAdapter<T> adapter) {
        this.adapter = requireNonNull(adapter);
    }

    public static class Builder <V> extends RuleBuilder<RegistryEntryRule<V>, V> {
        private final Registry<V> registry;
        private final ProviderEntry<V> provider;
        public Builder(Registry<V> registry, V initialValue) {
            super(argSupplierBuilder(registry.getKey()), Builder::acceptor, initialValue);
            this.registry = requireNonNull(registry);
            this.provider = null;
        }

        public Builder(RegistryKey<? extends Registry<V>> regKey, Identifier valueId) {
            super(argSupplierBuilder(regKey), Builder::acceptor, null);
            this.registry = null;
            this.provider = new ProviderEntry<>(regKey, valueId);
        }

        @Override
        @NotNull
        protected RegistryEntryRule<V> ruleBuilder(Type<RegistryEntryRule<V>> type) {
            if (this.registry == null) {
                return new RegistryEntryRule<>(type, this.provider.regKey, this.provider.entryId, super.validator, super.adapter);
            }
            return new RegistryEntryRule<>(type, this.registry, super.initialValue, super.validator, super.adapter);
        }

        static <U> void acceptor(Visitor consumer, Key<RegistryEntryRule<U>> key, Type<RegistryEntryRule<U>> type) {
            ((IGameRulesVisitor)consumer).unruled_visitRegistryEntry(key, type);
        }
    }

    private record ProviderEntry <T> (RegistryKey<? extends Registry<T>> regKey, Identifier entryId) {
        ProviderEntry<T> withId(Identifier id) {
            return new ProviderEntry<>(this.regKey, id);
        }
    }
}
