package de.z0rdak.yawp.commands;

import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import de.z0rdak.yawp.api.commands.CommandConstants;
import de.z0rdak.yawp.api.core.RegionManager;
import de.z0rdak.yawp.api.events.region.RegionEvent;
import de.z0rdak.yawp.api.permission.Permissions;
import de.z0rdak.yawp.commands.arguments.region.ContainingOwnedRegionArgumentType;
import de.z0rdak.yawp.constants.Constants;
import de.z0rdak.yawp.core.area.CuboidArea;
import de.z0rdak.yawp.core.area.SphereArea;
import de.z0rdak.yawp.core.flag.BooleanFlag;
import de.z0rdak.yawp.core.flag.RegionFlag;
import de.z0rdak.yawp.core.region.IMarkableRegion;
import de.z0rdak.yawp.core.region.IProtectedRegion;
import de.z0rdak.yawp.core.stick.MarkerStick;
import de.z0rdak.yawp.data.region.LevelRegionData;
import de.z0rdak.yawp.data.region.RegionDataManager;
import de.z0rdak.yawp.platform.Services;
import de.z0rdak.yawp.util.LocalRegions;
import de.z0rdak.yawp.util.StickUtil;
import de.z0rdak.yawp.util.visualization.BlockDisplayProperty;
import java.util.Collections;
import java.util.Optional;
import java.util.Set;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2172;
import net.minecraft.class_2487;
import net.minecraft.class_2561;

import static de.z0rdak.yawp.api.commands.CommandConstants.*;
import static de.z0rdak.yawp.commands.DimensionCommands.getRandomExample;
import static de.z0rdak.yawp.commands.arguments.ArgumentUtil.*;
import static de.z0rdak.yawp.util.ChatLinkBuilder.buildRegionInfoLink;
import static de.z0rdak.yawp.api.MessageSender.sendCmdFeedback;
import static net.minecraft.class_124.field_1061;
import static net.minecraft.class_9334.field_49628;

public final class MarkerCommands {

    private MarkerCommands() {
    }

    static LiteralArgumentBuilder<class_2168> build() {
        return literal(MARKER)
                .then(literal(GIVE)
                        .executes(MarkerCommands::giveMarkerStick))
                .then(literal(RESET)
                        .executes(MarkerCommands::resetStick))
                .then(literal(CREATE)
                        .then(class_2170.method_9244(CommandConstants.NAME.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> class_2172.method_9265(Collections.singletonList(getRandomExample()), builder))
                                .executes(ctx -> createMarkedRegion(ctx, getRegionNameArgument(ctx), null))
                                .then(class_2170.method_9244(PARENT.toString(), StringArgumentType.word())
                                        .suggests((ctx, builder) -> ContainingOwnedRegionArgumentType.owningRegions().listSuggestionsWithMarker(ctx, builder))
                                        .executes(ctx -> createMarkedRegion(ctx, getRegionNameArgument(ctx), getContainingOwnedRegionArgumentWithMarker(ctx))))));
    }

    public static IMarkableRegion fromMarkedBlocks(CommandContext<class_2168> ctx, class_1657 player, String regionName) throws CommandSyntaxException {
        class_1799 maybeStick = player.method_6047();
        if (StickUtil.isMarker(maybeStick)) {
            class_2487 stickNBT = StickUtil.getStickNBT(maybeStick);
            if (stickNBT != null) {
                MarkerStick marker = new MarkerStick(stickNBT);
                if (!marker.isValidArea()) {
                    sendCmdFeedback(ctx.getSource(), class_2561.method_48321("cli.msg.dim.info.region.create.stick.area.invalid", "Marked area is not valid").method_27692(field_1061));
                    return null;
                }
                return LocalRegions.regionFrom(player, marker, regionName);
            } else {
                sendCmdFeedback(ctx.getSource(), class_2561.method_48321("cli.msg.dim.info.region.create.stick.invalid", "Invalid RegionMarker data, sorry. Get a new one and try again."));
                return null;
            }
            
        } else {
            sendCmdFeedback(ctx.getSource(), class_2561.method_48321("cli.msg.dim.info.region.create.stick.missing", "Put a valid(*) RegionMarker in your main hand to create a region!").method_27692(field_1061));
            return null;
        }
    }

    private static int createMarkedRegion(CommandContext<class_2168> ctx, String regionName, IProtectedRegion parentRegion) {
        try {
            LevelRegionData levelData = RegionDataManager.getOrCreate(ctx.getSource().method_9225().method_27983().method_29177());
            int res = levelData.isValidRegionName(regionName);
            if (res == -1) {
                sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.dim.info.region.create.name.invalid", "Invalid region name supplied: '%s'", regionName));
                return res;
            }
            if (res == 1) {
                sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.dim.info.region.create.name.exists", "Dimension %s already contains region with name %s", buildRegionInfoLink(levelData.getDim()), buildRegionInfoLink(levelData.getLocal(regionName))));
                return res;
            }
            class_1657 player = ctx.getSource().method_9207();
            IMarkableRegion newRegion = fromMarkedBlocks(ctx, player, regionName);
            if (newRegion == null) {
                sendCmdFeedback(ctx.getSource(), class_2561.method_48321("cli.msg.dim.info.region.create.stick.area.invalid", "Marked area is not valid").method_27692(field_1061));
                return -1;
            }
            IProtectedRegion parent = parentRegion == null ? levelData.getDim() : parentRegion;
            return createRegion(ctx, player, levelData, newRegion, parent);
        } catch (CommandSyntaxException e) {
            Constants.LOGGER.error(e);
            sendCmdFeedback(ctx.getSource(), class_2561.method_48321("cli.msg.dim.info.region.create.stick.no-player", "This command can only be executed as a player!").method_27692(field_1061));
            return -1;
        }
    }

    private static int createRegion(CommandContext<class_2168> ctx, class_1657 player, LevelRegionData dimCache, IMarkableRegion region, IProtectedRegion parentRegion) {
        if (Services.EVENT.post(new RegionEvent.Create(region, player)) ) {
            return 1;
        }
        //if (RegionEvents.CREATE_REGION.invoker().createRegion(new RegionEvent.CreateRegionEvent(region, player))) {
        //    return 1;
        //}
        boolean hasConfigPermission = Permissions.get().hasConfigPermission(player);
        boolean hasRegionPermission = Permissions.get().hasGroupPermission(parentRegion, player, Permissions.OWNER);
        if (hasConfigPermission || hasRegionPermission) {
            Services.REGION_CONFIG.getDefaultFlags().stream()
                    .map(RegionFlag::fromId)
                    .forEach(flag -> region.addFlag(new BooleanFlag(flag)));
            dimCache.addLocal(parentRegion, region);
            LocalRegions.ensureHigherRegionPriorityFor(region, Services.REGION_CONFIG.getDefaultPriority());
            RegionDataManager.save();
            sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.dim.info.region.create.success", "Successfully created region %s (parent: %s)", buildRegionInfoLink(region), buildRegionInfoLink(parentRegion)));
            return 0;
        } else {
            sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.dim.info.region.create.local.deny", "You don't have the permission to create a region in the region %s!", buildRegionInfoLink(parentRegion)));
            return 1;
        }
    }

    private static int resetStick(CommandContext<class_2168> ctx) {
        try {
            class_1657 player = ctx.getSource().method_9207();
            class_1799 mainHandItem = player.method_6047();
            if (!mainHandItem.equals(class_1799.field_8037) && StickUtil.isMarker(mainHandItem)) {
                StickUtil.resetMarkerNbt(mainHandItem, player.method_37908().method_27983());
                sendCmdFeedback(ctx.getSource(), class_2561.method_48321("cli.msg.dim.info.region.create.stick.reset", "RegionMarker successfully reset!"));
                return 0;
            } else {
                sendCmdFeedback(ctx.getSource(), class_2561.method_48321("cli.msg.dim.info.region.create.stick.missing", "Put a valid(*) RegionMarker in your main hand to create a region!").method_27692(field_1061));
                return 1;
            }
        } catch (CommandSyntaxException e) {
            sendCmdFeedback(ctx.getSource(), class_2561.method_48321("cli.msg.dim.info.region.create.stick.no-player", "This command can only be executed as a player!").method_27692(field_1061));
            return 1;
        }
    }

    private static int giveMarkerStick(CommandContext<class_2168> ctx) {
        try {
            class_1657 targetPlayer = ctx.getSource().method_9207();
            class_1799 marker = class_1802.field_8600.method_7854();
            StickUtil.initMarkerNbt(marker, targetPlayer.method_37908().method_27983());
            targetPlayer.method_7270(marker);
            sendCmdFeedback(ctx.getSource(), class_2561.method_48321("cli.msg.dim.info.region.create.stick.success", "RegionMarker added to your inventory!"));
        } catch (CommandSyntaxException e) {
            sendCmdFeedback(ctx.getSource(), class_2561.method_48321("cli.msg.dim.info.region.create.stick.no-player", "This command can only be executed as a player!").method_27692(field_1061));
            return 1;
        }
        return 0;
    }
}
