package de.z0rdak.yawp.util.text.messages.multiline;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import de.z0rdak.yawp.core.region.*;
import de.z0rdak.yawp.data.region.LevelRegionData;
import de.z0rdak.yawp.data.region.RegionDataManager;
import de.z0rdak.yawp.util.text.Messages;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import net.minecraft.class_2558;
import net.minecraft.class_2561;
import net.minecraft.class_5250;

import static de.z0rdak.yawp.util.ChatComponentBuilder.*;
import static de.z0rdak.yawp.util.ChatLinkBuilder.*;

import static de.z0rdak.yawp.util.text.messages.multiline.RegionStateMessage.buildRegionStateLink;
import static de.z0rdak.yawp.util.text.messages.pagination.TeleportAnchorPagination.buildRegionTeleportAnchorListLink;
import static net.minecraft.class_124.field_1065;

public class RegionInfoMessage implements MultiLineMessage<IProtectedRegion> {

    private final IProtectedRegion region;
    private final List<class_2561> regionInfoLines;

    public RegionInfoMessage(IProtectedRegion region) {
        this.region = region;
        this.regionInfoLines = new ArrayList<>();
    }

    // approximation to the real maximum value
    private final static int MAX_ENCODER_LIMIT = 30000;

    private static Optional<JsonObject> regionJson(IProtectedRegion region) {
        switch (region.getRegionType()) {
            case GLOBAL -> {
                GlobalRegion global = (GlobalRegion) region;
                var json = new JsonObject();
                DataResult<JsonElement> jsonResult = GlobalRegion.CODEC.encode(global, JsonOps.INSTANCE, json);
                if (jsonResult.result().isPresent()) {
                    JsonElement orThrow = jsonResult.result().get();
                    return Optional.ofNullable(orThrow.getAsJsonObject());
                }
                return Optional.of(json);
            }
            case DIMENSION -> {
                DimensionalRegion dim = (DimensionalRegion) region;
                var json = new JsonObject();
                DataResult<JsonElement> jsonResult = DimensionalRegion.CODEC.encode(dim, JsonOps.INSTANCE, json);
                if (jsonResult.result().isPresent()) {
                    JsonElement orThrow = jsonResult.result().get();
                    return Optional.ofNullable(orThrow.getAsJsonObject());
                }
                return Optional.of(json);
            }
            case LOCAL -> {
                MarkedRegion local = (MarkedRegion) region;
                var json = new JsonObject();
                DataResult<JsonElement> jsonResult = MarkedRegion.CODEC.encode(local, JsonOps.INSTANCE, json);
                if (jsonResult.result().isPresent()) {
                    JsonElement orThrow = jsonResult.result().get();
                    return Optional.ofNullable(orThrow.getAsJsonObject());
                }
                return Optional.of(json);
            }
        }
        return Optional.empty();
    }

    public static class_5250 buildRegionOverviewHeader(IProtectedRegion region) {
        Optional<JsonObject> jsonObject = regionJson(region);
        String clipBoardText = "";
        if (jsonObject.isPresent()) {
            JsonObject regionJson = jsonObject.get();
            var jsonStr = regionJson.toString();
            clipBoardText = jsonStr.getBytes(StandardCharsets.UTF_8).length > MAX_ENCODER_LIMIT
                    ? "Region data to big to send to client, sorry"
                    : regionJson.toString();
        }
        switch (region.getRegionType()) {
            case GLOBAL: {
                GlobalRegion global = (GlobalRegion) region;
                var json = new JsonObject();
                DataResult<JsonElement> jsonResult = GlobalRegion.CODEC.encode(global, JsonOps.INSTANCE, json);
                if (jsonResult.result().isPresent()) {
                    JsonElement orThrow = jsonResult.result().get();
                    int length = orThrow.toString().getBytes(StandardCharsets.UTF_8).length;
                    class_5250 dumpLinkText = class_2561.method_48321("cli.msg.global.overview.header.dump.link.text", "Global overview");
                    class_5250 dumpLinkHover = class_2561.method_48321("cli.msg.global.overview.header.dump.link.hover", "Copy Global Region NBT to clipboard");
                    class_5250 clipBoardDumpLink = buildExecuteCmdComponent(dumpLinkText, dumpLinkHover, clipBoardText, class_2558.class_2559.field_21462, field_1065);
                    return buildHeader(class_2561.method_48322("cli.msg.info.header.for", "== %s for %s ==", clipBoardDumpLink, buildRegionInfoLink(region)));
                }
            }
            case DIMENSION: {
                class_5250 dumpLinkText = class_2561.method_48321("cli.msg.dim.overview.header.dump.link.text", "Dimension overview");
                class_5250 dumpLinkHover = class_2561.method_48321("cli.msg.dim.overview.header.dump.link.hover", "Copy Dimensional Region NBT to clipboard");
                class_5250 clipBoardDumpLink = buildExecuteCmdComponent(dumpLinkText, dumpLinkHover,  clipBoardText, class_2558.class_2559.field_21462, field_1065);
                return buildHeader(class_2561.method_48322("cli.msg.info.header.for", "== %s for %s ==", clipBoardDumpLink, buildRegionInfoLink(region)));
            }
            case LOCAL: {
                class_5250 dumpLinkText = class_2561.method_48321("cli.msg.local.overview.header.dump.link.text", "Region overview");
                class_5250 dumpLinkHover = class_2561.method_48321("cli.msg.local.overview.header.dump.link.hover", "Copy Local Region NBT to clipboard");
                class_5250 clipBoardDumpLink = buildExecuteCmdComponent(dumpLinkText, dumpLinkHover, clipBoardText, class_2558.class_2559.field_21462, field_1065);
                return buildHeader(class_2561.method_48322("cli.msg.info.header.for", "== %s for %s ==", clipBoardDumpLink, buildRegionInfoLink(region)));
            }
            default:
                throw new IllegalStateException("Unexpected value: " + region.getRegionType());
        }
    }

    public static class_5250 buildFlagsListLink(IProtectedRegion region) {
        if (region.getRegionType() == RegionType.GLOBAL) {
            return buildRegionFlagListLink(region);
        }
        return Messages.substitutable("%s | %s", buildResponsibleFlagListLink(region), buildRegionFlagListLink(region));
    }

    public static class_5250 buildGroupLinks(IProtectedRegion region) {
        return getGroupsForRegion(region).stream()
                .map(group -> buildGroupLink(region, group, getGroupSize(region, group)))
                .reduce(class_2561.method_43470(""), (link1, link2) -> link1.method_27693(" ").method_10852(link2));
    }

    private static class_5250 buildRegionHierarchyComponent(IProtectedRegion region) {
        class_5250 listChildrenLink = buildRegionListChildrenLink(region);
        switch (region.getRegionType()) {
            case GLOBAL: {
                // Dimensions: [n dimensions(s)]
                return buildInfoComponent("cli.msg.info.dimensions", "Dimensions", listChildrenLink);
            }
            case DIMENSION: {
                // Parent: [global], [n children], [n regions] [+]
                class_5250 globalRegionLink = buildRegionInfoLink(region.getParent(), class_2561.method_48321("cli.msg.info.region.global.link.hover", "Show global region info"));
                LevelRegionData levelData = RegionDataManager.getOrCreate(region.getDim());
                class_5250 hierarchyLinks = Messages.substitutable("%s, %s, %s", globalRegionLink, buildDimRegionsLink(levelData), listChildrenLink);
                return buildInfoComponent("cli.msg.info.region.hierarchy", "Hierarchy", hierarchyLinks);
            }
            case LOCAL: {
                // Parent: [parent] [x], [n children] [+]
                class_5250 parentRemoveThisChildLink = buildRegionRemoveChildLink(region.getParent(), region);
                class_5250 hierarchyLinks = class_2561.method_43470("");
                if (region.getParent().getRegionType() == RegionType.DIMENSION) {
                    // don't show removal link, since it's not possible to remove the parent
                    hierarchyLinks = Messages.substitutable("%s, %s", buildRegionInfoLink(region.getParent()), listChildrenLink);
                }
                if (region.getParent().getRegionType() == RegionType.LOCAL) {
                    hierarchyLinks = Messages.substitutable("%s %s, %s", buildRegionInfoLink(region.getParent()), parentRemoveThisChildLink, listChildrenLink);
                }
                return buildInfoComponent("cli.msg.info.region.parent", "Parent", hierarchyLinks);
            }
            default:
                throw new IllegalStateException("Unexpected value: " + region.getRegionType());
        }
    }

    @Override
    public IProtectedRegion getContent() {
        return this.region;
    }

    @Override
    public List<class_2561> getLines() {
        // == Region [<name>] overview ==
        class_5250 header = buildRegionOverviewHeader(region);
        regionInfoLines.add(header);

        // Flags: [n] responsible flag(s) | [m] flag(s)] [+]
        class_5250 flagsText = buildInfoComponent("cli.msg.info.region.flag", "Flags", buildFlagsListLink(region));
        regionInfoLines.add(flagsText);

        if (region.getRegionType() == RegionType.LOCAL) {
            var local = (IMarkableRegion) region;
            // Area: [Area]
            class_5250 areaText = buildInfoComponent("cli.msg.info.region.area", "Area", buildRegionAreaLink(local));
            regionInfoLines.add(areaText);

            // Visualization: [Visualization] | [Settings]
            var subject = class_2561.method_48321("cli.msg.info.region.visualization", "Visualization");
            var actions = Messages.substitutable("%s | %s", buildRegionVisualizationLink(local), buildDisplaySettingsLink(local));
            var visualization = buildInfoComponent(subject, actions);
            regionInfoLines.add(visualization);

            // Teleport Anchors: [m] teleport anchor(s) [+]
            var tpAnchorSubject = class_2561.method_48321("cli.msg.info.region.tp-anchor", "Teleport Anchors");
            var tpAnchor = buildInfoComponent(tpAnchorSubject, buildRegionTeleportAnchorListLink(local));
            regionInfoLines.add(tpAnchor);
        }

        // Groups: [owners], [members], [<listGroups>]
        class_5250 groupsText = buildInfoComponent("cli.msg.info.region.group", "Groups", buildGroupLinks(region));
        regionInfoLines.add(groupsText);

        // Regions: [global], [n children], [n regions][+],
        // Dimensions: [n dimensions(s)]
        // Parent: [parent][x], [n children][+]
        class_5250 hierarchy = buildRegionHierarchyComponent(region);
        regionInfoLines.add(hierarchy);

        // State: [State]
        class_5250 state = buildInfoComponent("cli.msg.info.region.state", "State", buildRegionStateLink(region));
        regionInfoLines.add(state);

        return regionInfoLines;
    }
}
