package de.z0rdak.yawp.core.area;

import de.z0rdak.yawp.constants.serialization.RegionNbtKeys;
import de.z0rdak.yawp.util.AreaUtil;
import de.z0rdak.yawp.util.NbtCompatHelper;
import org.apache.commons.lang3.NotImplementedException;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2512;
import net.minecraft.class_3341;

import static de.z0rdak.yawp.util.AreaUtil.blocksBetweenOnAxis;
import static de.z0rdak.yawp.util.AreaUtil.distanceManhattan;

/**
 * Represents and wraps a simple AxisAlignedBB.
 * This area is marked by two positions and thus spans a cuboid shape
 */
public class CuboidArea extends AbstractArea {

    private class_3341 area;
    private class_2338 p1;
    private class_2338 p2;

    public CuboidArea(class_3341 area) {
        super(AreaType.CUBOID);
        this.area = area;
    }

    public CuboidArea(class_2338 p1, class_2338 p2) {
        this(class_3341.method_34390(p1, p2));
        this.p1 = AreaUtil.getLowerPos(p1, p2);
        this.p2 = AreaUtil.getHigherPos(p1, p2);
    }

    public CuboidArea(class_2487 nbt) {
        super(nbt);
        this.deserializeNBT(nbt);
    }

    public static CuboidArea expand(CuboidArea area, int min, int max) {
        class_2338 p1 = area.getAreaP1();
        class_2338 p2 = area.getAreaP2();
        return new CuboidArea(new class_2338(p1.method_10263(), min, p1.method_10260()),
                new class_2338(p2.method_10263(), max, p2.method_10260()));
    }

    private static boolean isInFacePlane(class_2338 point, class_2338 corner1, class_2338 corner2, class_2338 corner3, class_2338 corner4) {
        // Check if the point is within the plane defined by the four corners
        // This can be done by checking if the point is within the bounding box of the face
        return point.method_10263() >= corner1.method_10263() && point.method_10263() <= corner2.method_10263()
                && point.method_10264() >= corner1.method_10264() && point.method_10264() <= corner3.method_10264()
                && point.method_10260() >= corner1.method_10260() && point.method_10260() <= corner4.method_10260();
    }

    private static List<class_2338> getBlocksInFace(class_2338 corner1, class_2338 corner2, class_2338 corner3, class_2338 corner4) {
        List<class_2338> blocksInFace = new ArrayList<>();
        // Determine the min and max coordinates along each axis
        int minX = Math.min(Math.min(corner1.method_10263(), corner2.method_10263()), Math.min(corner3.method_10263(), corner4.method_10263()));
        int minY = Math.min(Math.min(corner1.method_10264(), corner2.method_10264()), Math.min(corner3.method_10264(), corner4.method_10264()));
        int minZ = Math.min(Math.min(corner1.method_10260(), corner2.method_10260()), Math.min(corner3.method_10260(), corner4.method_10260()));
        int maxX = Math.max(Math.max(corner1.method_10263(), corner2.method_10263()), Math.max(corner3.method_10263(), corner4.method_10263()));
        int maxY = Math.max(Math.max(corner1.method_10264(), corner2.method_10264()), Math.max(corner3.method_10264(), corner4.method_10264()));
        int maxZ = Math.max(Math.max(corner1.method_10260(), corner2.method_10260()), Math.max(corner3.method_10260(), corner4.method_10260()));
        // Iterate through the grid defined by the min and max coordinates
        for (int x = minX; x <= maxX; x++) {
            for (int y = minY; y <= maxY; y++) {
                for (int z = minZ; z <= maxZ; z++) {
                    class_2338 currentPos = new class_2338(x, y, z);
                    if (isInFacePlane(currentPos, corner1, corner2, corner3, corner4)) {
                        blocksInFace.add(currentPos);
                    }
                }
            }
        }
        return blocksInFace;
    }

    @Override
    public boolean contains(class_2338 pos) {
        // INFO: this.area.contains(x,y,z); does not work, because the max checks are exclusive by default.
        // TODO: Maybe replace with net.minecraft.util.math.MutableBoundingBox::intersectsWith which has inclusive checks
        return pos.method_10263() >= area.method_35415() && pos.method_10263() <= area.method_35418()
                && pos.method_10264() >= this.area.method_35416() && pos.method_10264() <= this.area.method_35419()
                && pos.method_10260() >= this.area.method_35417() && pos.method_10260() <= this.area.method_35420();
    }

    public boolean contains(CuboidArea inner) {
        return this.area.method_35415() <= inner.area.method_35415() && this.area.method_35418() >= inner.area.method_35418()
                && this.area.method_35416() <= inner.area.method_35416() && this.area.method_35419() >= inner.area.method_35419()
                && this.area.method_35417() <= inner.area.method_35417() && this.area.method_35420() >= inner.area.method_35420();
    }

    public boolean contains(SphereArea inner) {
        int sphereRadius = inner.getRadius();
        class_2338 center = inner.center;
        // Bounding box check
        if (!this.intersects(inner)) {
            return false; // Cuboid and sphere do not intersect, sphere cannot be contained
        }

        // Bounding sphere check
        int maxDistanceToCorner = maxDistanceToCorners(center);
        if (maxDistanceToCorner > sphereRadius) {
            return false; // Maximum distance to cuboid corner is greater than sphere's radius
        }

        // Iterate over a cube around the sphere to generate points
        for (int x = center.method_10263() - sphereRadius; x <= center.method_10263() + sphereRadius; x++) {
            for (int y = center.method_10264() - sphereRadius; y <= center.method_10264() + sphereRadius; y++) {
                for (int z = center.method_10260() - sphereRadius; z <= center.method_10260() + sphereRadius; z++) {
                    class_2338 currentPos = new class_2338(x, y, z);
                    int distance = distanceManhattan(center, currentPos);
                    // Check if the point is within or on the surface of the sphere
                    if (distance <= sphereRadius) {
                        // Check if the point is within the cuboid
                        if (!this.contains(currentPos)) {
                            return false; // Point is outside cuboid, sphere is not contained
                        }
                    }
                }
            }
        }
        return true; // All points within or on the sphere are contained in the cuboid
    }

    private int maxDistanceToCorners(class_2338 center) {
        List<class_2338> corners = getVertices();
        int maxDistance = Integer.MIN_VALUE;
        for (class_2338 corner : corners) {
            int distance = distanceManhattan(center, corner);
            if (distance > maxDistance) {
                maxDistance = distance;
            }
        }
        return maxDistance;
    }

    /**
     * Returns the vertices of the cuboid area as a list of BlockPos.
     * Z+
     * p7-----p8
     * /|      /|
     * Y+  p5------p6|
     * | |     | |
     * | p3----|-p4
     * |/      |/
     * p1------p2  X+
     *
     * @return [p1, p2, p3, p4, p5, p6, p7, p8] as list of BlockPos
     */
    public List<class_2338> getVertices() {
        class_2338 p1 = new class_2338(this.area.method_35415(), this.area.method_35416(), this.area.method_35417());
        class_2338 p2 = new class_2338(this.area.method_35418(), this.area.method_35416(), this.area.method_35417());
        class_2338 p3 = new class_2338(this.area.method_35415(), this.area.method_35416(), this.area.method_35420());
        class_2338 p4 = new class_2338(this.area.method_35418(), this.area.method_35416(), this.area.method_35420());
        class_2338 p5 = new class_2338(this.area.method_35415(), this.area.method_35419(), this.area.method_35417());
        class_2338 p6 = new class_2338(this.area.method_35418(), this.area.method_35419(), this.area.method_35417());
        class_2338 p7 = new class_2338(this.area.method_35415(), this.area.method_35419(), this.area.method_35420());
        class_2338 p8 = new class_2338(this.area.method_35418(), this.area.method_35419(), this.area.method_35420());
        return Arrays.asList(p1, p2, p3, p4, p5, p6, p7, p8);
    }

    /**
     * Returns the hull of the cuboid area as a set of BlockPos.
     * The hull is the outermost layer of blocks of the cuboid area.
     * The hull is calculated by iterating through the faces of the cuboid area and collecting the blocks in each face.
     * @return hull as set of BlockPos of cuboid area
     */
    @Override
    public Set<class_2338> getHull() {
        List<class_2338> vertices = getVertices();
        List<class_2338> face1 = getBlocksInFace(vertices.get(0), vertices.get(1), vertices.get(2), vertices.get(3));
        List<class_2338> face2 = getBlocksInFace(vertices.get(4), vertices.get(5), vertices.get(6), vertices.get(7));
        List<class_2338> face3 = getBlocksInFace(vertices.get(0), vertices.get(2), vertices.get(4), vertices.get(6));
        List<class_2338> face4 = getBlocksInFace(vertices.get(1), vertices.get(3), vertices.get(5), vertices.get(7));
        List<class_2338> face5 = getBlocksInFace(vertices.get(0), vertices.get(1), vertices.get(4), vertices.get(5));
        List<class_2338> face6 = getBlocksInFace(vertices.get(2), vertices.get(3), vertices.get(6), vertices.get(7));
        return Stream.of(face1, face2, face3, face4, face5, face6)
                .flatMap(List::stream)
                .collect(Collectors.toSet());
    }

    private boolean intersects(CuboidArea other) {
        return this.area.method_14657(other.area);
    }

    public boolean intersects(SphereArea other) {
        return other.intersects(this);
    }

    public class_3341 getArea() {
        return area;
    }

    public int getXsize() {
        return Math.max(this.area.method_35414(), 1);
    }

    public int getZsize() {
        return Math.max(this.area.method_14663(), 1);
    }

    public int getYsize() {
        return Math.max(this.area.method_14660(), 1);
    }

    public class_2338 getAreaP1() {
        return this.p1;
    }

    public class_2338 getAreaP2() {
        return this.p2;
    }

    @Override
    public class_2487 serializeNBT() {
        class_2487 nbt = super.serializeNBT();
        nbt.method_10566(RegionNbtKeys.P1, class_2512.method_10692(this.p1));
        nbt.method_10566(RegionNbtKeys.P2, class_2512.method_10692(this.p2));
        return nbt;
    }

    @Override
    public void deserializeNBT(class_2487 nbt) {
        super.deserializeNBT(nbt);
        this.p1 = NbtCompatHelper.toBlockPos(nbt, RegionNbtKeys.P1).orElseThrow();
        this.p2 = NbtCompatHelper.toBlockPos(nbt, RegionNbtKeys.P2).orElseThrow();
        this.area = class_3341.method_34390(p1, p2);
    }

    @Override
    public String toString() {
        return getAreaType().areaType + " " + AreaUtil.blockPosStr(this.p1) + " <-> " + AreaUtil.blockPosStr(this.p2) +
                "\n" + "Size: " + "X=" + this.getXsize() + ", Y=" + this.getYsize() + ", Z=" + this.getZsize() +
                "\n" + "Blocks: " + AreaUtil.blockPosStr(this.p1) + ", " + AreaUtil.blockPosStr(this.p2);
    }

    @Override
    public List<class_2338> markedBlocks() {
        return Arrays.asList(this.p1, this.p2);
    }

    /**
     * Returns the outer frame of the cuboid area as a set of BlockPos.
     *
     * @return outer frame as set of BlockPos of cuboid area
     */
    public Set<class_2338> getFrame() {
        List<class_2338> vertices = getVertices();
        Set<class_2338> p12 = blocksBetweenOnAxis(vertices.get(0), vertices.get(1), class_2350.class_2351.field_11048);
        Set<class_2338> p34 = blocksBetweenOnAxis(vertices.get(2), vertices.get(3), class_2350.class_2351.field_11048);
        Set<class_2338> p56 = blocksBetweenOnAxis(vertices.get(4), vertices.get(5), class_2350.class_2351.field_11048);
        Set<class_2338> p78 = blocksBetweenOnAxis(vertices.get(6), vertices.get(7), class_2350.class_2351.field_11048);
        Set<class_2338> p15 = blocksBetweenOnAxis(vertices.get(0), vertices.get(4), class_2350.class_2351.field_11052);
        Set<class_2338> p26 = blocksBetweenOnAxis(vertices.get(1), vertices.get(5), class_2350.class_2351.field_11052);
        Set<class_2338> p37 = blocksBetweenOnAxis(vertices.get(2), vertices.get(6), class_2350.class_2351.field_11052);
        Set<class_2338> p48 = blocksBetweenOnAxis(vertices.get(3), vertices.get(7), class_2350.class_2351.field_11052);
        Set<class_2338> p13 = blocksBetweenOnAxis(vertices.get(0), vertices.get(2), class_2350.class_2351.field_11051);
        Set<class_2338> p24 = blocksBetweenOnAxis(vertices.get(1), vertices.get(3), class_2350.class_2351.field_11051);
        Set<class_2338> p57 = blocksBetweenOnAxis(vertices.get(4), vertices.get(6), class_2350.class_2351.field_11051);
        Set<class_2338> p68 = blocksBetweenOnAxis(vertices.get(5), vertices.get(7), class_2350.class_2351.field_11051);
        return Stream.of(p12, p34, p56, p78, p15, p26, p37, p48, p13, p24, p57, p68)
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
    }

    @Override
    public boolean containsOther(IMarkableArea inner) {
        switch (inner.getAreaType()) {
            case CUBOID:
                return this.contains((CuboidArea) inner);
            case SPHERE:
                return this.contains((SphereArea) inner);
            default:
                throw new NotImplementedException("Area type not implemented yet");
        }
    }

    @Override
    public boolean intersects(IMarkableArea other) {
        switch (other.getAreaType()) {
            case CUBOID:
                return this.intersects((CuboidArea) other);
            case SPHERE:
                return this.intersects((SphereArea) other);
            default:
                throw new NotImplementedException("Area type not implemented yet");
        }
    }
}
