package de.z0rdak.yawp.core.flag;

import de.z0rdak.yawp.constants.Constants;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import net.minecraft.class_2487;

import static de.z0rdak.yawp.constants.serialization.RegionNbtKeys.FLAG_TYPE;

/**
 * Represents a simple map holding the flag values to the corresponding flag names for a region and providing methods handling the flags. <br>
 * [Key] FlagName -> [Value] IFlag <br>
 * E.g. "break_blocks" -> BooleanFlag {"value": false, ...}
 */
public class FlagContainer extends HashMap<String, IFlag> implements IFlagContainer {


    public FlagContainer(class_2487 nbt) {
        this();
        this.deserializeNBT(nbt);
    }

    public FlagContainer() {
        super();
    }

    @Override
    public class_2487 serializeNBT() {
        class_2487 nbt = new class_2487();
        this.forEach((flagName, iFlag) -> {
            if (RegionFlag.contains(flagName)) {
                nbt.method_10566(flagName, iFlag.serializeNBT());
            }
        });
        return nbt;
    }

    @Override
    public void deserializeNBT(class_2487 nbt) {
        Set<String> flagKeys = nbt.method_10541();
        flagKeys.stream()
                .filter(RegionFlag::contains)
                .forEach(key -> {
                    class_2487 flagNbt = nbt.method_10562(key);
                    FlagType flagType = FlagType.of(flagNbt.method_10558(FLAG_TYPE));
                    if (flagType != null) {
                        switch (flagType) {
                            case BOOLEAN_FLAG:
                                this.put(key, new BooleanFlag(flagNbt));
                                break;
                            case LIST_FLAG:
                                this.put(key, new ListFlag(flagNbt));
                                break;
                            case INT_FLAG:
                                this.put(key, new IntFlag(flagNbt));
                                break;
                        }
                    } else {
                        // TODO: Throw error to give error message in abstract region to be more precise with error
                        Constants.LOGGER.warn("Error reading entry for flag '{}'.", key);
                    }
                });
    }

    public FlagContainer deepCopy() {
        return new FlagContainer(this.serializeNBT());
    }

    public void put(IFlag flag) {
        this.put(flag.getName(), flag);
    }

    public FlagState flagState(String flagName) {
        if (this.contains(flagName)) {
            return this.get(flagName).getState();
        } else
            return FlagState.UNDEFINED;
    }

    public Map<String, IFlag> getActiveFlags() {
        Map<String, IFlag> activeFlags = new HashMap<>();
        this.forEach((k, v) -> {
            if (v.isActive()) {
                activeFlags.put(k, v);
            }
        });
        return activeFlags;
    }

    public List<IFlag> getFlags(FlagState state) {
        return this.values().stream().filter(flag -> flag.getState() == state).toList();
    }

    public boolean contains(String flag) {
        return this.containsKey(flag);
    }

    public void updateFlag(IFlag flag) {
        this.put(flag);
    }

    public void toggleFlag(String flag, boolean enable) {
        if (this.contains(flag)) {
            this.get(flag).setState(enable ? FlagState.ALLOWED : FlagState.DENIED);
        }
    }
}
