package de.z0rdak.yawp.core.region;

import de.z0rdak.yawp.api.permission.Permissions;
import de.z0rdak.yawp.core.flag.FlagContainer;
import de.z0rdak.yawp.core.flag.IFlag;
import de.z0rdak.yawp.core.flag.RegionFlag;
import de.z0rdak.yawp.core.group.PlayerContainer;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.stream.Collectors;
import net.minecraft.class_1657;
import net.minecraft.class_1937;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2519;
import net.minecraft.class_2520;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_7924;

import static de.z0rdak.yawp.constants.serialization.RegionNbtKeys.*;

/**
 * An abstract region represents the basic implementation of a IProtectedRegion.
 * This abstraction can be used for markable regions as well as regions without
 * an area (dimensions). <br>
 */
public abstract class AbstractRegion implements IProtectedRegion {
    protected class_5321<class_1937> dimension;
    protected IProtectedRegion parent;
    protected String parentName;
    private String name;
    private RegionType regionType;
    private FlagContainer flags;
    private Map<String, PlayerContainer> groups;
    private boolean isActive;
    private boolean isMuted;
    private Map<String, IProtectedRegion> children;
    private Set<String> childrenNames;

    protected AbstractRegion(class_2487 nbt) {
        this.childrenNames = new HashSet<>(0);
        this.children = new HashMap<>(0);
        this.parentName = null;
        this.parent = null;
        this.flags = new FlagContainer();
        this.groups = new HashMap<>();
        this.groups.put(Permissions.MEMBER, new PlayerContainer(Permissions.MEMBER));
        this.groups.put(Permissions.OWNER, new PlayerContainer(Permissions.OWNER));
        this.deserializeNBT(nbt);
    }

    protected AbstractRegion(String name, class_5321<class_1937> dimension, RegionType type) {
        this.name = name;
        this.dimension = dimension;
        this.regionType = type;
        this.flags = new FlagContainer();
        this.groups = new HashMap<>();
        this.groups.put(Permissions.MEMBER, new PlayerContainer(Permissions.MEMBER));
        this.groups.put(Permissions.OWNER, new PlayerContainer(Permissions.OWNER));
        this.children = new HashMap<>();
        this.isActive = true;
        this.childrenNames = new HashSet<>();
    }

    protected AbstractRegion(String name, class_5321<class_1937> dimension, RegionType regionType, class_1657 owner) {
        this(name, dimension, regionType);
        if (owner != null) {
            this.groups.get(Permissions.OWNER).addPlayer(owner.method_5667(), owner.method_5820());
        }
    }

    @Override
    public String getParentName() {
        return parentName;
    }

    @Override
    public String getName() {
        return name;
    }

    protected void setName(String name) {
        this.name = name;
    }

    public void setGroups(Map<String, PlayerContainer> groups) {
        this.groups = groups;
    }

    @Override
    public class_5321<class_1937> getDim() {
        return dimension;
    }

    @Override
    public RegionType getRegionType() {
        return this.regionType;
    }

    @Override
    public void addFlag(IFlag flag) {
        this.flags.put(flag);
    }

    @Override
    public void removeFlag(String flag) {
        this.flags.remove(flag);
    }

    public boolean containsFlag(RegionFlag flag) {
        return this.flags.contains(flag.name);
    }

    @Override
    public boolean containsFlag(String flag) {
        return this.flags.contains(flag);
    }

    @Override
    public Collection<IFlag> getFlags() {
        return Collections.unmodifiableList(new ArrayList<>(this.flags.values()));
    }

    public void setFlags(FlagContainer flags) {
        this.flags = flags;
    }

    @Override
    public FlagContainer getFlagContainer() {
        return flags;
    }

    @Nullable
    @Override
    public IFlag getFlag(String flagName) {
        if (this.flags.contains(flagName)) {
            return this.flags.get(flagName);
        } else {
            return null;
        }
    }

    @Override
    public boolean isActive() {
        return this.isActive;
    }

    @Override
    public void setIsActive(boolean isActive) {
        this.isActive = isActive;
    }

    @Override
    public boolean isMuted() {
        return this.isMuted;
    }

    @Override
    public void setIsMuted(boolean isMuted) {
        this.isMuted = isMuted;
    }

    @Override
    public void addPlayer(class_1657 player, String group) {
        this.getGroup(group).addPlayer(player.method_5667(), player.method_5820());
    }

    @Override
    public void addPlayer(UUID uuid, String playerName, String group) {
        this.getGroup(group).addPlayer(uuid, playerName);
    }


    @Override
    public void addTeam(String teamName, String group) {
        this.getGroup(group).addTeam(teamName);
    }

    @Override
    public void removeTeam(String teamName, String group) {
        this.getGroup(group).removeTeam(teamName);
    }

    public void resetGroups() {
        this.groups.clear();
        this.groups.put(Permissions.MEMBER, new PlayerContainer(Permissions.MEMBER));
        this.groups.put(Permissions.OWNER, new PlayerContainer(Permissions.OWNER));
    }

    @Override
    public void removePlayer(UUID playerUuid, String group) {
        if (group.equals("*")) {
            for (String g : this.groups.keySet()) {
                this.getGroup(g).removePlayer(playerUuid);
            }
            return;
        }
        this.getGroup(group).removePlayer(playerUuid);
    }

    @Override
    public boolean hasTeam(String teamName, String group) {
        return this.getGroup(group).hasTeam(teamName);
    }

    @Override
    public boolean hasPlayer(UUID playerUuid, String group) {
        return this.getGroup(group).hasPlayer(playerUuid);
    }

    /**
     * Gets the container for the provided group. Creates a new one if none is existent.
     */
    @Override
    public PlayerContainer getGroup(String group) {
        if (!this.groups.containsKey(group)) {
            return this.groups.put(group, new PlayerContainer(group));
        }
        return this.groups.get(group);
    }

    /**
     * Checks if the player is defined in the regions player list OR whether the player is an operator.
     * Usually this check is needed when an event occurs, and it needs to be checked whether
     * the player has a specific permission to perform an action in the region.
     *
     * @param player to be checked
     * @return true if player is in region list or is an operator, false otherwise
     */
    @Override
    public boolean permits(class_1657 player) {
        return isInGroup(player, Permissions.OWNER) || isInGroup(player, Permissions.MEMBER);
    }

    public boolean isInGroup(class_1657 player, String group) {
        return this.groups.get(group).hasPlayer(player.method_5667()) || (player.method_5781() != null && this.groups.get(group).hasTeam(player.method_5781().method_1197()));
    }

    /**
     * Will always be called by IMarkableRegion to remove child of type IMarkableRegion
     */
    @Override
    public void removeChild(IProtectedRegion child) {
        this.children.remove(child.getName());
        this.childrenNames.remove(child.getName());
    }

    @Override
    public void clearChildren() {
        this.children.clear();
        this.childrenNames.clear();
    }

    @Override
    public Map<String, IProtectedRegion> getChildren() {
        return Collections.unmodifiableMap(this.children);
    }

    @Override
    public Set<String> getChildrenNames() {
        return this.childrenNames;
    }

    @Override
    public boolean hasChild(IProtectedRegion maybeChild) {
        return this.children.containsKey(maybeChild.getName());
    }

    @Override
    public boolean addChild(IProtectedRegion child) {
        this.children.put(child.getName(), child);
        this.childrenNames.add(child.getName());
        ((AbstractRegion) child).setParent(this);
        return true;
    }

    protected boolean setParent(IProtectedRegion parent) {
        this.parent = parent;
        this.parentName = parent.getName();
        return true;
    }


    public IProtectedRegion getParent() {
        return parent;
    }

    @Override
    public class_2487 serializeNBT() {
        class_2487 nbt = new class_2487();
        nbt.method_10582(NAME, this.name);
        nbt.method_10582(DIM, dimension.method_29177().toString());
        nbt.method_10582(REGION_TYPE, this.regionType.type);
        nbt.method_10556(ACTIVE, this.isActive);
        nbt.method_10556(MUTED, this.isMuted);
        nbt.method_10566(FLAGS, this.flags.serializeNBT());
        nbt.method_10566(OWNERS, this.groups.get(OWNERS).serializeNBT());
        nbt.method_10566(MEMBERS, this.groups.get(MEMBERS).serializeNBT());
        if (this.parent != null) {
            nbt.method_10582(PARENT, this.parent.getName());
        } else {
            nbt.method_10582(PARENT, "");
        }
        if (this.children != null) {
            class_2499 childrenList = new class_2499();
            childrenList.addAll(this.children.keySet().stream().map(class_2519::method_23256).collect(Collectors.toSet()));
            nbt.method_10566(CHILDREN, childrenList);
        } else {
            nbt.method_10566(CHILDREN, new class_2499());
        }
        return nbt;
    }

    @Override
    public void deserializeNBT(class_2487 nbt) {
        this.name = nbt.method_10558(NAME);
        this.dimension = class_5321.method_29179(class_7924.field_41223, class_2960.method_60654(nbt.method_10558(DIM)));
        this.isActive = nbt.method_10577(ACTIVE);
        this.isMuted = nbt.method_10577(MUTED);
        this.regionType = RegionType.of(nbt.method_10558(REGION_TYPE));
        this.flags = new FlagContainer(nbt.method_10562(FLAGS));
        this.groups = new HashMap<>();
        this.groups.put(OWNERS, new PlayerContainer(nbt.method_10562(OWNERS)));
        this.groups.put(MEMBERS, new PlayerContainer(nbt.method_10562(MEMBERS)));
        if (this.parent == null && nbt.method_10573(PARENT, class_2520.field_33258)) {
            String parentName = nbt.method_10558(PARENT);
            if (!parentName.isEmpty()) {
                this.parentName = nbt.method_10558(PARENT);
            } else {
                this.parentName = null;
            }
        }
        if (this.children != null && this.children.isEmpty()) {
            if (nbt.method_10573(CHILDREN, class_2520.field_33259)) {
                class_2499 childrenNbt = nbt.method_10554(CHILDREN, class_2520.field_33258);
                if (!childrenNbt.isEmpty()) {
                    this.children = new HashMap<>(childrenNbt.size());
                    this.childrenNames = new HashSet<>(childrenNbt.size());
                    for (int i = 0; i < childrenNbt.size(); i++) {
                        this.childrenNames.add(childrenNbt.method_10608(i));
                    }
                }
            }
        }
    }
}
