package de.z0rdak.yawp.data.region;

import de.z0rdak.yawp.api.permission.Permissions;
import de.z0rdak.yawp.constants.Constants;
import de.z0rdak.yawp.core.INbtSerializable;
import de.z0rdak.yawp.core.area.AreaType;
import de.z0rdak.yawp.core.group.PlayerContainer;
import de.z0rdak.yawp.core.region.*;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.stream.Collectors;
import net.minecraft.class_1657;
import net.minecraft.class_1937;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_5321;

import static de.z0rdak.yawp.constants.serialization.RegionNbtKeys.*;

public class DimensionRegionCache implements INbtSerializable<class_2487> {

    private Map<String, IMarkableRegion> regionsInDimension;
    private DimensionalRegion dimensionalRegion;

    public DimensionRegionCache(class_5321<class_1937> dim) {
        this(new DimensionalRegion(dim, RegionDataManager.get().getGlobalRegion()));
    }

    public DimensionRegionCache(class_2487 nbt) {
        this.deserializeNBT(nbt);
    }

    protected DimensionRegionCache(DimensionalRegion dimensionalRegion) {
        this.dimensionalRegion = dimensionalRegion;
        this.regionsInDimension = new HashMap<>();
    }

    private static String getDataName(DimensionalRegion dim) {
        return getDataName(dim.getName());
    }

    private static String getDataName(String dim) {
        return Constants.MOD_ID + "-" + dim.replace(':', '-');
    }

    private static IMarkableRegion deserializeLocalRegion(AreaType areaType, class_2487 regionNbt) {
        switch (areaType) {
            case CUBOID:
                return new CuboidRegion(regionNbt);
            case CYLINDER:
                return new CylinderRegion(regionNbt);
            case SPHERE:
                return new SphereRegion(regionNbt);
            case POLYGON_3D:
                return new PolygonRegion(regionNbt);
            case PRISM:
                return new PrismRegion(regionNbt);
            default:
                throw new IllegalArgumentException("Unable to read area type of region '" + regionNbt.method_10558(NAME) + "'!");
        }
    }

    public class_5321<class_1937> dimensionKey() {
        return this.dimensionalRegion.getDim();
    }

    public DimensionalRegion getDimensionalRegion() {
        return dimensionalRegion;
    }

    public Map<String, IMarkableRegion> getRegionsInDimension() {
        return Collections.unmodifiableMap(regionsInDimension);
    }

    public Collection<IMarkableRegion> getAllLocal() {
        return regionsInDimension.values().stream().collect(Collectors.toUnmodifiableList());
    }

    public void addRegion(IProtectedRegion parent, IMarkableRegion child) {
        parent.addChild(child);
        this.regionsInDimension.put(child.getName(), child);
    }

    public void addRegion(IMarkableRegion child) {
        this.dimensionalRegion.addChild(child);
        this.regionsInDimension.put(child.getName(), child);
    }

    public int getRegionCount() {
        return regionsInDimension.size();
    }

    public Collection<String> getRegionNames() {
        return new ArrayList<>(regionsInDimension.keySet());
    }

    @Nullable
    public IMarkableRegion getRegion(String regionName) {
        return this.regionsInDimension.get(regionName);
    }

    public void removeRegion(IMarkableRegion region) {
        if (this.contains(region.getName())) {
            this.regionsInDimension.remove(region.getName());
            if (region.getParent().getRegionType() == RegionType.DIMENSION) {
                region.getParent().removeChild(region);
            }
        }
    }

    public void clearRegions() {
        this.regionsInDimension.clear();
        this.dimensionalRegion.clearChildren();
    }

    public boolean contains(String regionName) {
        return regionsInDimension.containsKey(regionName);
    }

    @Override
    public class_2487 serializeNBT() {
        class_2487 nbt = new class_2487();
        nbt.method_10566(DIM_REGION, this.dimensionalRegion.serializeNBT());
        class_2487 regions = new class_2487();
        this.regionsInDimension.forEach((name, region) -> {
            regions.method_10566(name, region.serializeNBT());
        });
        nbt.method_10566(REGIONS, regions);
        return nbt;
    }

    public void renameRegion(IMarkableRegion region, String regionName) {
        if (this.regionsInDimension.containsKey(regionName)) {
            throw new IllegalArgumentException("Region with name '" + regionName + "' already exists in dimension '" + this.dimensionalRegion.getName() + "'!");
        }
        IMarkableRegion currentRegion = this.regionsInDimension.get(region.getName());
        IProtectedRegion parent = currentRegion.getParent();
        this.removeRegion(currentRegion);
        currentRegion.rename(regionName);
        this.addRegion(parent, currentRegion);
    }

    @Override
    public void deserializeNBT(class_2487 nbt) {
        if (nbt.method_10573(DIM_REGION, class_2520.field_33260)) {
            this.dimensionalRegion = new DimensionalRegion(nbt.method_10562(DIM_REGION));
        } else {
            throw new IllegalArgumentException("Unable to load dimensional region data from NBT");
        }
        this.regionsInDimension = new HashMap<>();
        class_2487 regionsNbt = nbt.method_10562(REGIONS);
        regionsNbt.method_10541().forEach(regionName -> {
            class_2487 regionNbt = regionsNbt.method_10562(regionName);
            AreaType areaType = AreaType.of(regionNbt.method_10558(AREA_TYPE));
            if (areaType != null) {
                Constants.LOGGER.debug("Loading region data for region '{}'", regionName);
                IMarkableRegion newRegion = DimensionRegionCache.deserializeLocalRegion(areaType, regionNbt);
                this.addRegion(this.getDimensionalRegion(), newRegion);
            } else {
                Constants.LOGGER.error("Unable to read region type for region '{}'!", regionName);
            }
        });
    }

    public boolean hasOwner(class_1657 player) {
        PlayerContainer owners = this.dimensionalRegion.getGroup(Permissions.OWNER);
        return owners.hasPlayer(player.method_5667())
                || (player.method_5781() != null && owners.hasTeam(player.method_5781().method_1197()));
    }

    public boolean hasMember(class_1657 player) {
        PlayerContainer members = this.dimensionalRegion.getGroup(Permissions.MEMBER);
        return members.hasPlayer(player.method_5667())
                || (player.method_5781() != null && members.hasTeam(player.method_5781().method_1197()));
    }
}
