package de.z0rdak.yawp.data.region;

import com.mojang.datafixers.types.Type;
import de.z0rdak.yawp.api.commands.CommandConstants;
import de.z0rdak.yawp.commands.arguments.region.RegionArgumentType;
import de.z0rdak.yawp.constants.Constants;
import de.z0rdak.yawp.core.flag.BooleanFlag;
import de.z0rdak.yawp.core.flag.IFlag;
import de.z0rdak.yawp.core.flag.RegionFlag;
import de.z0rdak.yawp.core.region.GlobalRegion;
import de.z0rdak.yawp.core.region.IMarkableRegion;
import de.z0rdak.yawp.core.region.IProtectedRegion;
import de.z0rdak.yawp.platform.Services;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_18;
import net.minecraft.class_1937;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_2561;
import net.minecraft.class_26;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_4284;
import net.minecraft.class_5321;
import net.minecraft.class_7225;
import net.minecraft.class_7924;
import net.minecraft.server.MinecraftServer;
import org.apache.commons.lang3.NotImplementedException;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.stream.Collectors;

import static de.z0rdak.yawp.api.commands.CommandConstants.values;
import static de.z0rdak.yawp.constants.serialization.RegionNbtKeys.*;
import static de.z0rdak.yawp.handler.HandlerUtil.isServerSide;

public class RegionDataManager extends class_18 {

    /**
     * Name which is used for the file to store the NBT data: yawp-dimensions.dat
     */
    private static final String DATA_NAME = Constants.MOD_ID + "-dimensions";
    public static MinecraftServer serverInstance;
    /**
     * The global region of this mod which all sets common rules/flags for all regions.
     */
    private static GlobalRegion globalRegion = new GlobalRegion();
    /**
     * Singleton used to access methods to manage region data.
     */

    private static RegionDataManager regionDataCache = new RegionDataManager();
    /**
     * Map which holds the mod region information. Each dimension has its on DimensionRegionCache.
     */
    private final Map<class_5321<class_1937>, DimensionRegionCache> dimCacheMap = new HashMap<>();
    private final Set<String> dimensionDataNames = new HashSet<>();

    private RegionDataManager() {
    }

    public static void save() {
        Constants.LOGGER.debug(class_2561.method_48321("data.nbt.dimensions.save", "Save for RegionDataManager called. Attempting to save region data...").getString());
        regionDataCache.method_80();
    }

    /**
     * Returns the name of all dimension tracked by the region data manager.
     *
     * @return the set of dimension names which are tracked by the region data manager.
     */
    @SuppressWarnings("unused")
    public static Set<String> getDimensionDataNames() {
        return Collections.unmodifiableSet(regionDataCache.dimensionDataNames);
    }

    public static List<DimensionRegionCache> getDimensionCaches() {
        return Collections.unmodifiableList(new ArrayList<>(regionDataCache.dimCacheMap.values()));
    }

    public static RegionDataManager get() {
        if (regionDataCache == null) {
            if (serverInstance != null) {
                class_3218 overworld = serverInstance.method_30002();
                if (!overworld.field_9236) {
                    class_26 storage = overworld.method_17983();
                    class_8645<RegionDataManager> rdmt = new class_8645<>(RegionDataManager::new, RegionDataManager::load, class_4284.field_45079);
                    regionDataCache = storage.method_20786(rdmt, DATA_NAME);
                }
            }
        }
        return regionDataCache;
    }

    /**
     * Server startup hook for loading the region data from the yawp-dimension.dat file by creating an instance of RegionDataManager.
     *
     * @param server which is fired upon server start and acts as trigger to load region data from disk.
     */
    public static void initServerInstance(MinecraftServer server) {
        serverInstance = server;
    }

    /**
     * Server startup hook for loading the region data from the yawp-dimension.dat file by creating an instance of RegionDataManager.
     *
     * @param minecraftServer
     * @param serverWorld
     */
    public static void loadRegionDataForWorld(MinecraftServer minecraftServer, class_3218 serverWorld) {
        try {
            if (serverInstance == null) {
                serverInstance = minecraftServer;
            }
            var isOverworld = serverWorld.method_27983().method_29177().equals(class_3218.field_25179.method_29177());
            if (isServerSide(serverWorld) && isOverworld) {
                class_26 storage = serverWorld.method_17983();
                class_8645<RegionDataManager> rdmt = new class_8645<>(RegionDataManager::new, RegionDataManager::load, class_4284.field_45079);
                RegionDataManager data = storage.method_17924(rdmt, DATA_NAME);
                storage.method_123(DATA_NAME, data);
                regionDataCache = data;
                Constants.LOGGER.info(class_2561.method_48322("data.nbt.dimensions.load.success", "Loaded %s region(s) for %s dimension(s)", data.getTotalRegionAmount(), data.getDimensionAmount()).getString());
            }
        } catch (NullPointerException npe) {
            Constants.LOGGER.error(class_2561.method_48321("data.nbt.dimensions.load.failure", "Loading regions failed!").getString());
        }
    }

    /**
     * Method which gets called when a new RegionDataManager instance is created by loadRegionData.
     *
     * @param nbt compound region data read from disk to be deserialized for the region cache.
     */
    public static RegionDataManager load(class_2487 nbt, class_7225.class_7874 registries) {
        RegionDataManager rdm = new RegionDataManager();
        rdm.dimCacheMap.clear();
        if (!nbt.method_10545(GLOBAL) || nbt.method_10562(GLOBAL).method_33133()) {
            Constants.LOGGER.info(class_2561.method_43471("Missing global region data. Initializing new data. (Ignore this for the first server start)").getString());
            globalRegion = new GlobalRegion();
        } else {
            class_2487 globalNbt = nbt.method_10562(GLOBAL);
            globalRegion = new GlobalRegion();
            globalRegion.deserializeNBT(globalNbt);
            Constants.LOGGER.info(class_2561.method_43471("Loaded global region data").getString());
        }

        if (!nbt.method_10545(DIMENSIONS) || nbt.method_10562(DIMENSIONS).method_33133()) {
            Constants.LOGGER.info(class_2561.method_43471("No region data found for dimensions. Initializing new data...").getString());
            rdm.dimCacheMap.clear();
        } else {
            class_2487 dimensionRegions = nbt.method_10562(DIMENSIONS);
            Constants.LOGGER.info(class_2561.method_43471("Loading region(s) for " + dimensionRegions.method_10546() + " dimension(s)").getString());
            rdm.dimCacheMap.clear();
            // deserialize all region without parent and child references
            for (String dimKey : dimensionRegions.method_10541()) {
                if (dimensionRegions.method_10573(dimKey, class_2520.field_33260)) {
                    class_5321<class_1937> dimension = class_5321.method_29179(class_7924.field_41223, class_2960.method_60654(dimKey));
                    class_2487 dimCacheNbt = dimensionRegions.method_10562(dimKey);
                    if (dimCacheNbt.method_10573(REGIONS, class_2520.field_33260)) {
                        Constants.LOGGER.info(class_2561.method_43471("Loading " + dimCacheNbt.method_10562(REGIONS).method_10546() + " region(s) for dimension '" + dimKey + "'").getString());
                    } else {
                        Constants.LOGGER.info(class_2561.method_43471("No region data for dimension '" + dimKey + "' found").getString());
                    }
                    DimensionRegionCache dimCache = new DimensionRegionCache(dimCacheNbt);
                    globalRegion.addChild(dimCache.getDimensionalRegion());
                    rdm.dimCacheMap.put(dimension, dimCache);
                    rdm.dimensionDataNames.add(dimKey);
                    Constants.LOGGER.info(class_2561.method_48322("data.nbt.dimensions.loaded.dim.amount", "Loaded %s region(s) for dimension '%s'", dimCache.getRegionCount(), dimCache.getDimensionalRegion().getName()).getString());
                }
            }
        }
        // restore parent/child hierarchy
        rdm.dimCacheMap.forEach((dimKey, cache) -> {
            if (cache.getRegionCount() > 0) {
                Constants.LOGGER.info(class_2561.method_43471("Restoring region hierarchy for regions in dimension '" + dimKey.method_29177() + "'").getString());
                ArrayList<IMarkableRegion> regions = new ArrayList<>(cache.getRegionsInDimension().values());
                regions.forEach(region -> {
                    // set child reference
                    region.getChildrenNames().forEach(childName -> {
                        if (!cache.contains(childName)) {
                            Constants.LOGGER.error(class_2561.method_43471("Corrupt save data. Child region '" + childName + "' not found in dimension '" + dimKey + "'!").getString());
                        } else {
                            IMarkableRegion child = cache.getRegion(childName);
                            if (child != null) {
                                cache.getDimensionalRegion().removeChild(child);
                                region.addChild(child);
                            }
                        }
                    });
                });
            }
        });
        return rdm;
    }

    /**
     * An event which is called after a player has been moved to a different world.
     * Event handler which creates a new DimensionRegionCache when a dimension is created the first time, by a player loading the dimension.
     */
    public static void addDimKeyOnDimensionChange(class_1657 Player, class_1937 origin, class_1937 destination) {
        if (isServerSide(destination)) {
            if (!regionDataCache.dimCacheMap.containsKey(destination.method_27983())) {
                DimensionRegionCache cache = RegionDataManager.get().newCacheFor(destination.method_27983());
                Constants.LOGGER.info("Init region data for dimension '{}'..", cache.dimensionKey().method_29177());
                save();
            }
        }
    }

    /**
     * Event handler which is used to initialize the dimension cache with first dimension entry when a player logs in.
     */
    public static void addDimKeyOnPlayerLogin(class_1297 entity, class_1937 serverWorld) {
        if (isServerSide(serverWorld) && entity instanceof class_1657) {
            class_5321<class_1937> dim = serverWorld.method_27983();
            if (!regionDataCache.dimCacheMap.containsKey(dim)) {
                DimensionRegionCache cache = regionDataCache.newCacheFor(dim);
                Constants.LOGGER.info("Player joining to server in dimension without region data. This should only happen the first time a player is joining.");
                Constants.LOGGER.info("Init region data for dimension '{}'..", cache.dimensionKey().method_29177());
                save();
            }
        }
    }

    public static void addFlags(Set<String> flags, IProtectedRegion region) {
        flags.stream()
                .map(RegionFlag::fromId)
                .forEach(flag -> {
                    switch (flag.type) {
                        case BOOLEAN_FLAG:
                            region.addFlag(new BooleanFlag(flag));
                            break;
                        case LIST_FLAG:
                        case INT_FLAG:
                            throw new NotImplementedException("");
                    }
                });
    }

    /**
     * Method which gets called the region data is marked as dirty via the save/markDirty method.
     *
     * @param compound nbt data to be filled with the region information.
     * @return the compound region nbt data to be saved to disk.
     */
    @Override
    public class_2487 method_75(class_2487 compound, class_7225.class_7874 var2) {
        compound.method_10566(GLOBAL, globalRegion.serializeNBT());
        class_2487 dimRegionNbtData = new class_2487();
        // Constants.LOGGER.info(new TranslationTextComponent("data.nbt.dimensions.save.amount", this.getTotalRegionAmount(), dimCacheMap.keySet().size()).getString());
        Constants.LOGGER.info(class_2561.method_43471("Saving " + this.getTotalRegionAmount() + " region(s) for " + dimCacheMap.keySet().size() + " dimensions").getString());
        for (Map.Entry<class_5321<class_1937>, DimensionRegionCache> entry : dimCacheMap.entrySet()) {
            // Constants.LOGGER.info(new TranslationTextComponent("data.nbt.dimensions.save.dim.amount", this.getRegionAmount(entry.getKey()), entry.getKey().location().toString()).getString());
            Constants.LOGGER.info(class_2561.method_43471("Saving " + this.getRegionAmount(entry.getKey()) + " region(s) for dimension '" + entry.getKey().method_29177() + "'").getString());
            String dimensionName = entry.getValue().getDimensionalRegion().getName();
            dimRegionNbtData.method_10566(dimensionName, entry.getValue().serializeNBT());
        }
        compound.method_10566(DIMENSIONS, dimRegionNbtData);
        return compound;
    }

    public int getTotalRegionAmount() {
        return dimCacheMap.values().stream()
                .mapToInt(DimensionRegionCache::getRegionCount)
                .sum();
    }

    public int getRegionAmount(class_5321<class_1937> dim) {
        return cacheFor(dim).getRegionCount();
    }

    public Set<String> getDimensionList() {
        return dimCacheMap.keySet().stream()
                .map(entry -> entry.method_29177().toString())
                .collect(Collectors.toSet());
    }

    public Set<class_5321<class_1937>> getDimKeys() {
        return new HashSet<>(dimCacheMap.keySet());
    }

    public GlobalRegion getGlobalRegion() {
        return globalRegion;
    }

    public void resetDimensionCache(class_5321<class_1937> dim) {
        dimCacheMap.remove(dim);
    }

    public void resetGlobalRegion() {
        List<IProtectedRegion> collect = new ArrayList<>(globalRegion.getChildren().values());
        globalRegion = new GlobalRegion();
        collect.forEach(dr -> {
            globalRegion.addChild(dr);
        });
        save();
    }

    public int getDimensionAmount() {
        return dimCacheMap.keySet().size();
    }

    @Nullable
    public Optional<IMarkableRegion> getRegionIn(class_5321<class_1937> dim, String regionName) {
        if (dimCacheMap.containsKey(dim)) {
            DimensionRegionCache cache = dimCacheMap.get(dim);
            if (cache.contains(regionName)) {
                IMarkableRegion region = cache.getRegion(regionName);
                return region == null ? Optional.empty() : Optional.of(region);
            }
        }
        return Optional.empty();
    }

    public Collection<IMarkableRegion> getRegionsFor(class_5321<class_1937> dim) {
        return cacheFor(dim).getAllLocal();
    }

    public DimensionRegionCache cacheFor(class_5321<class_1937> dim) {
        if (!dimCacheMap.containsKey(dim)) {
            newCacheFor(dim);
            save();
        }
        return dimCacheMap.get(dim);
    }

    public Optional<DimensionRegionCache> getCache(class_5321<class_1937> dim) {
        if (!dimCacheMap.containsKey(dim)) {
            return Optional.empty();
        }
        return Optional.of(dimCacheMap.get(dim));
    }

    /**
     * Method to check if a region name is valid for a given dimension. <br>
     * A region name is valid if it matches the pattern and is not already used in the dimension.
     *
     * @param dim        the dimension to be checked.
     * @param regionName the name of the region to be checked.
     * @return -1 if the region name is invalid, 0 if the region name is valid, 1 if the region name is already used in the dimension.
     */
    public int isValidRegionName(class_5321<class_1937> dim, String regionName) {
        List<String> commandStrings = Arrays.stream(values()).map(CommandConstants::toString).collect(Collectors.toList());
        if (!regionName.matches(RegionArgumentType.VALID_NAME_PATTERN.pattern())
                || commandStrings.contains(regionName.toLowerCase())) {
            return -1;
        }
        if (cacheFor(dim).contains(regionName)) {
            return 1;
        }
        return 0;
    }

    /**
     * Method to check if a region name is valid for a given dimension. <br>
     * A region name is valid if it matches the pattern and is not already used in the dimension.
     *
     * @param dim        the dimension to be checked.
     * @param regionName the name of the region to be checked.
     * @return -1 if the region name is invalid, 0 if the region name is valid, 1 if the region name is already used in the dimension.
     */
    public boolean isAvailableForLocal(class_5321<class_1937> dim, String regionName) {
        if (cacheFor(dim).contains(regionName)) {
            return false;
        }
        return isValidRegionName(dim, regionName) == 0;
    }

    public boolean containsCacheFor(class_5321<class_1937> dim) {
        return dimCacheMap.containsKey(dim);
    }

    public List<String> getFlagsIdsForDim(DimensionRegionCache dimCache) {
        if (dimCache != null) {
            return dimCache.getDimensionalRegion().getFlags()
                    .stream()
                    .map(IFlag::getName)
                    .collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    public DimensionRegionCache newCacheFor(class_5321<class_1937> dim) {
        DimensionRegionCache cache = new DimensionRegionCache(dim);
        Set<String> defaultDimFlags = Services.REGION_CONFIG.getDefaultDimFlags();
        addFlags(defaultDimFlags, cache.getDimensionalRegion());
        cache.getDimensionalRegion().setIsActive(Services.REGION_CONFIG.shouldActivateNewDimRegion());
        globalRegion.addChild(cache.getDimensionalRegion());
        dimCacheMap.put(dim, cache);
        dimensionDataNames.add(cache.getDimensionalRegion().getName());
        return cache;
    }
}
