package de.z0rdak.yawp.util;

import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;
import com.mojang.authlib.GameProfile;
import com.mojang.brigadier.context.CommandContext;
import de.z0rdak.yawp.constants.Constants;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Consumer;
import net.minecraft.class_2168;
import net.minecraft.class_3312;

import static java.nio.charset.StandardCharsets.UTF_8;

public class MojangApiHelper {

    private MojangApiHelper() {
    }

    @Nullable
    public static GameProfile getGameProfileInfo(UUID uuid, Consumer<GameProfile> onResult) {
        try (CloseableHttpClient httpclient = HttpClients.createDefault()) {
            String uuidWithOutDashes = uuid.toString().replace("-", "");
            String uri = "https://sessionserver.mojang.com/session/minecraft/profile/" + uuidWithOutDashes;
            HttpGet httpGet = new HttpGet(uri);
            try {
                CloseableHttpResponse response = httpclient.execute(httpGet);
                GameProfile gameProfile = deserializeGameProfile(response);
                if (gameProfile != null) {
                    return gameProfile;
                }
                onResult.accept(null);
                return null;
            } catch (IOException e) {
                Constants.LOGGER.error("Error fetching game profile info for player '{}': {}", uuid.toString(), e);
                throw e;
            }
        } catch (IOException e) {
            onResult.accept(null);
            Constants.LOGGER.error("Error fetching game profile info for player '{}': {}", uuid.toString(), e);
            return null;
        }
    }

    @Nullable
    private static GameProfile deserializeGameProfile(CloseableHttpResponse response) throws IOException {
        if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
            Reader reader = new InputStreamReader(response.getEntity().getContent(), UTF_8);
            MojangProfileResponse profileResponse = new Gson().fromJson(reader, MojangProfileResponse.class);
            String uuidStr = profileResponse.id.replaceAll(
                    "(\\w{8})(\\w{4})(\\w{4})(\\w{4})(\\w{12})", "$1-$2-$3-$4-$5");
            return new GameProfile(UUID.fromString(uuidStr), profileResponse.name);
        }
        return null;
    }

    @Nullable
    public static GameProfile getGameProfileInfo(String username, Consumer<GameProfile> onResult) {
        try (CloseableHttpClient httpclient = HttpClients.createDefault()) {
            String uri = "https://api.mojang.com/users/profiles/minecraft/" + username;
            HttpGet httpGet = new HttpGet(uri);
            try {
                CloseableHttpResponse response = httpclient.execute(httpGet);
                if (response.getStatusLine().getStatusCode() == HttpStatus.SC_NO_CONTENT) {
                    // if status code is 204, then the player does not exist
                    Constants.LOGGER.error("Could not retrieve game profile for player {}", username);
                    return null;
                }
                GameProfile gameProfile = deserializeGameProfile(response);
                if (gameProfile != null) {
                    return gameProfile;
                }
                onResult.accept(null);
                return null;
            } catch (IOException e) {
                Constants.LOGGER.error("Error fetching game profile info for player '{}': {}", username, e);
                throw e;
            }
        } catch (IOException e) {
            Constants.LOGGER.error("Error fetching game profile info for player '{}': {}", username, e);
            onResult.accept(null);
            return null;
        }
    }

    public static Optional<GameProfile> lookupGameProfileInCache(CommandContext<class_2168> ctx, String playerName) {
        class_3312 profileCache = ctx.getSource().method_9211().method_3793();
        // Uses Mojang's API to retrieve info from player repo. It invokes
        // YggdrasilGameProfileRepository.findProfilesByNames through the PlayerProfileCache
        // which itself makes an HTTP request to Mojang's API
        return profileCache.method_14515(playerName);
    }

    public static Optional<GameProfile> lookupGameProfileInCache(CommandContext<class_2168> ctx, UUID uuid) {
        class_3312 profileCache = ctx.getSource().method_9211().method_3793();
        // This in contrast to the name search does not make an HTTP request
        // It just looks up the profile in the cache
        return profileCache.method_14512(uuid);
    }

    private static class MojangProfileResponse {
        @SerializedName("name")
        public String name;
        @SerializedName("id")
        public String id;
    }
}
