package de.z0rdak.yawp.data.region;

import de.z0rdak.yawp.api.core.RegionManager;
import de.z0rdak.yawp.constants.Constants;
import de.z0rdak.yawp.core.flag.*;
import de.z0rdak.yawp.core.region.*;
import de.z0rdak.yawp.platform.Services;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_1937;
import net.minecraft.class_2561;
import net.minecraft.class_26;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_5218;
import net.minecraft.class_5321;
import net.minecraft.server.MinecraftServer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.stream.Collectors;

import static de.z0rdak.yawp.handler.HandlerUtil.isServerSide;

public class RegionDataManager {

    public static final Logger LOGGER = LogManager.getLogger(Constants.MOD_ID.toUpperCase(Locale.ROOT) + "-DataManager");
    private static MinecraftServer serverInstance;
    private static LevelListData savedLevelData = new LevelListData();
    private static GlobalRegionData globalRegionData = new GlobalRegionData();
    private static final Map<class_2960, LevelRegionData> dimRegionStorage = new  HashMap<>();

    public static LevelListData getSavedLevelData() {
        return savedLevelData;
    }

    public static Set<class_2960> getLevels() {
        return new HashSet<>(savedLevelData.getLevels());
    }

    public static boolean hasLevel(class_2960 level) {
        return savedLevelData.hasDimEntry(level);
    }

    // TODO: Move to API
    public static Set<String> getLevelNames() {
        return getLevels().stream().map(class_2960::toString).collect(Collectors.toSet());
    }

    public static GlobalRegionData getGlobalRegionData() {
        return globalRegionData;
    }
    public static GlobalRegion getGlobalRegion() {
        return getGlobalRegionData().getGlobal();
    }

    private RegionDataManager() {
    }


    public static void save() {
        save(true);
    }

    public static void save(boolean force) {
        if (force) {
            saveDimList(serverInstance);
            saveGlobalData(serverInstance);
            saveTrackedLevels(serverInstance);
        } else {
            savedLevelData.method_80();
            globalRegionData.method_80();
            dimRegionStorage.forEach((key, value) -> value.method_80());
        }
    }

    public static LevelListData getSavedDims() {
        if (savedLevelData == null) {
            if (serverInstance != null) {
                class_3218 overworld = serverInstance.method_30002();
                if (!overworld.method_8608()) {
                    class_26 storage = overworld.method_17983();
                    savedLevelData = storage.method_17924(LevelListData.TYPE);
                }
            }
        }
        return savedLevelData;
    }

    public static void onServerStarting(MinecraftServer server) {
        LOGGER.info(class_2561.method_48321("data.region.init","Initializing RegionDataManager...").getString());
        serverInstance = server;
        checkYawpDir(server);
    }

    private static void saveTrackedLevels(MinecraftServer server){
        server.method_3738().forEach(level -> {
            class_2960 levelRl = level.method_27983().method_29177();
            if (savedLevelData.hasDimEntry(levelRl)) {
                saveLevelData(server, level);
            }
        });
    }

    public static void save(MinecraftServer server, boolean flush, boolean force) {
        LOGGER.info(class_2561.method_48321("data.region.levels.save.forced", "Requested save. Saving region data for all levels").getString());
        if (serverInstance == null) serverInstance = server;
        save(force);
    }

    private static void saveDimList(MinecraftServer server) {
        class_26 dataStorage = server.method_30002().method_17983();
        dataStorage.method_123(LevelListData.TYPE, savedLevelData);
    }

    private static void saveGlobalData(MinecraftServer server) {
        class_26 dataStorage = server.method_30002().method_17983();
        dataStorage.method_123(GlobalRegionData.TYPE, globalRegionData);
    }

    private static void saveLevelData(MinecraftServer server, class_1937 level) {
        class_26 storage = server.method_30002().method_17983();
        class_2960 levelRl = level.method_27983().method_29177();
        LevelRegionData levelRegionData = dimRegionStorage.get(levelRl);
        LOGGER.info(class_2561.method_48322("data.region.level.save", "Saving region data for level '%s'", levelRl.toString()).getString());
        storage.method_123(LevelRegionData.buildSavedDataType(levelRl), levelRegionData);
        levelRegionData.method_80();
    }

    private static LevelRegionData loadLevelData(MinecraftServer server, class_1937 level) {
        class_26 storage = server.method_30002().method_17983();
        class_2960 dimLoc = level.method_27983().method_29177();
        return storage.method_20786(LevelRegionData.buildSavedDataType(dimLoc));
    }

    public static void saveOnStop(MinecraftServer server) {
        if (serverInstance == null) serverInstance = server;
        LOGGER.info(class_2561.method_48321("data.region.levels.save.stopped", "Stopping server. Saving region data for all levels").getString());
        save(true);
    }

    public static void saveOnUnload(MinecraftServer server, class_3218 level) {
        if (savedLevelData.hasDimEntry(level.method_27983().method_29177())) {
            LOGGER.info(class_2561.method_48322("data.region.level.save.unload", "Unloading level '%s'. Saving region data", level.method_27983().method_29177().toString()).getString());
            saveLevelData(server, level);
        }
    }

    public static void loadLevelListData(MinecraftServer server) {
        try {
            if (serverInstance == null)
                serverInstance = server;
            class_26 dataStorage = server.method_30002().method_17983();
            savedLevelData = dataStorage.method_20786(LevelListData.TYPE);
            if (savedLevelData == null) {
                LOGGER.info(class_2561.method_48321("data.region.levels.load.missing", "Missing level list for region data (ignore on first startup). Initializing...").getString());
                savedLevelData = new LevelListData();
                saveDimList(server);
            }
            LOGGER.info(class_2561.method_48322("data.region.levels.load.success", "Found region data for %s dimension(s)", savedLevelData.getLevels().size()).getString());
            globalRegionData = dataStorage.method_20786(GlobalRegionData.TYPE);
            if (globalRegionData == null) {
                LOGGER.info(class_2561.method_48321("data.region.global.missing", "Missing global region data (ignore on first startup). Initializing...").getString());
                globalRegionData = new GlobalRegionData();
                saveGlobalData(server);
            }
        } catch (NullPointerException npe) {
            LOGGER.error(class_2561.method_48321("data.region.level.local.load.failed", "Loading level region list failed!").getString(), npe);
        }
    }

    public static void worldLoad(MinecraftServer server, class_3218 level) {
        try {
            if (serverInstance == null)
                serverInstance = server;
            class_2960 levelRl = level.method_27983().method_29177();
            // init level data
            if (savedLevelData.hasDimEntry(levelRl)) {
                LevelRegionData levelRegionData = loadLevelData(server, level);
                if (levelRegionData == null) {
                    levelRegionData = new LevelRegionData(levelRl);
                    LOGGER.info(class_2561.method_48322("data.region.level.local.missing", "Initializing region data for '%s'", levelRl.toString()).getString());
                    dimRegionStorage.put(levelRl, levelRegionData);
                    saveLevelData(server, level);
                } else {
                    LOGGER.info(class_2561.method_48322("data.region.level.local.load.success", "Loaded %s region(s) for '%s'", levelRegionData.regionCount(), levelRl.toString()).getString());
                    dimRegionStorage.put(levelRl, levelRegionData);
                    savedLevelData.addDimEntry(levelRl);
                }
                // restoring region hierarchy
                LOGGER.info(class_2561.method_48322("data.region.level.local.load.restore", "Restoring region hierarchy for '%s'.", levelRl.toString()).getString());

                // restore dim <-> global hierarchy
                DimensionalRegion dimensionalRegion = levelRegionData.getDim();
                RegionManager.get().getGlobalRegion().addChild(dimensionalRegion);
                restoreHierarchy(levelRegionData, dimensionalRegion);

                // restore dim <-> local <-> local hierarchy
                levelRegionData.getLocals().forEach((regionName, region) -> {
                    restoreHierarchy(dimRegionStorage.get(levelRl), region);
                });
            }
        } catch (NullPointerException npe) {
            LOGGER.error(class_2561.method_48321("data.region.level.local.load.failed", "Loading regions failed!").getString(), npe);
        }
    }
    
    private static void restoreHierarchy(LevelRegionData levelRegionData, IProtectedRegion region) {
        ArrayList<String> childNames = new ArrayList<>(region.getChildrenNames());
        childNames.forEach(childName -> {
            if (!levelRegionData.hasLocal(childName)) {
                LOGGER.warn(class_2561.method_48322("data.region.level.local.load.restore.failed", "No region with name '%s' found in save data of '%s'! Your region data is most likely corrupt.", childName, levelRegionData.getId().toString()).getString());
            } else {
                IMarkableRegion child = levelRegionData.getLocal(childName);
                if (child != null) {
                    levelRegionData.getDim().removeChild(child);
                    region.addChild(child);
                }
            }
        });
    }

    private static void checkYawpDir(MinecraftServer server) {
        Path worldRootPath = server.method_27050(class_5218.field_24188).normalize();
        Path dataDirPath = worldRootPath.resolve("data/" + Constants.MOD_ID);
        if (Files.notExists(dataDirPath)) {
            try {
                Files.createDirectories(dataDirPath);
                LOGGER.info(class_2561.method_48322("data.region.env.init", "Created region data directory '%s'", dataDirPath.toString()).getString());
            } catch (IOException e) {
                LOGGER.error(class_2561.method_48321("data.region.env.error", "Failed to create directory for region data: '%s'").getString(), e);
                throw new RuntimeException(e);
            }
        }
    }


    public static void initLevelDataOnLogin(class_1297 entity, class_1937 level) {
        if (isServerSide(level) && entity instanceof class_1657) {
            initLevelData(level.method_27983().method_29177());
        }
    }

    public static void initLevelDataOnChangeWorld(class_1657 player, class_1937 srcLvl, class_1937 dstLvl) {
        if (isServerSide(srcLvl)) {
            initLevelData(dstLvl.method_27983().method_29177());
        }
    }

    private static LevelRegionData initLevelData(class_2960 rl){
        if (!savedLevelData.hasDimEntry(rl)) {
            LevelRegionData levelRegionData = new LevelRegionData(rl);
            DimensionalRegion dimensionalRegion = levelRegionData.getDim();
            // add default flags from config
            Set<String> defaultDimFlags = Services.REGION_CONFIG.getDefaultDimFlags();
            defaultDimFlags.stream()
                    .map(RegionFlag::fromId)
                    .forEach(flag -> dimensionalRegion.addFlag(new BooleanFlag(flag)));
            // set state from config
            dimensionalRegion.setIsActive(Services.REGION_CONFIG.shouldActivateNewDimRegion());
            // add as child of global
            RegionManager.get().getGlobalRegion().addChild(dimensionalRegion);

            dimRegionStorage.put(rl, levelRegionData);
            savedLevelData.addDimEntry(rl);
            LOGGER.info(class_2561.method_48322("data.region.level.init", "Initializing region data for level '%s'", rl.toString()).getString());
            save(false);
            return levelRegionData;
        }
        return dimRegionStorage.get(rl);
    }

    public static Optional<LevelRegionData> getLevelRegionData(class_2960 rl) {
        if (!savedLevelData.hasDimEntry(rl)) {
            return Optional.empty();
        }
        return Optional.of(dimRegionStorage.get(rl));
    }

    public static Optional<LevelRegionData> getLevelRegionData(class_5321<class_1937> dim) {
        return getLevelRegionData(dim.method_29177());
    }

    public static LevelRegionData getOrCreate(class_2960 rl) {
        if (!savedLevelData.hasDimEntry(rl)) {
            return initLevelData(rl);
        }
        return dimRegionStorage.get(rl);
    }

    public static LevelRegionData getOrCreate(class_1937 level)  {
        return getOrCreate(level.method_27983().method_29177());
    }


    // TODO: Move to API?
    // RegionManager.get().getLevelRegionData(region.getDim()).get().getLocalList()
    // Instead
    // RegionManager.get().getOrCreate(region.getDim()).getLocalList()
    public static Collection<IMarkableRegion> getLocalsFor(class_5321<class_1937> dim) {
        return getOrCreate(dim.method_29177()).getLocalList();
    }

    // TODO: Change API to offer a getOrCreate method
    public static LevelRegionData getOrCreate(class_5321<class_1937> dim) {
       return getOrCreate(dim.method_29177());
    }

    public static void resetLevelData(class_2960 rl) {
        dimRegionStorage.remove(rl);
    }

    public static void resetLevelData(class_5321<class_1937> dim) {
        resetLevelData(dim.method_29177());
    }

}
