package de.z0rdak.yawp.commands.arguments.flag;

import com.mojang.brigadier.StringReader;
import com.mojang.brigadier.arguments.ArgumentType;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.DynamicCommandExceptionType;
import com.mojang.brigadier.exceptions.SimpleCommandExceptionType;
import com.mojang.brigadier.suggestion.Suggestions;
import com.mojang.brigadier.suggestion.SuggestionsBuilder;
import de.z0rdak.yawp.api.commands.CommandConstants;
import de.z0rdak.yawp.commands.arguments.region.RegionArgumentType;
import de.z0rdak.yawp.constants.Constants;
import de.z0rdak.yawp.core.flag.IFlag;
import de.z0rdak.yawp.core.flag.RegionFlag;
import de.z0rdak.yawp.core.region.IProtectedRegion;
import de.z0rdak.yawp.core.region.RegionType;
import de.z0rdak.yawp.util.ChatLinkBuilder;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import net.minecraft.class_2168;
import net.minecraft.class_2172;
import net.minecraft.class_2561;
import net.minecraft.class_5250;

import static de.z0rdak.yawp.api.MessageSender.overLayMessage;
import static de.z0rdak.yawp.api.commands.CommandConstants.ADD;
import static de.z0rdak.yawp.api.commands.CommandConstants.REMOVE;
import static de.z0rdak.yawp.api.commands.Commands.buildAddFlagCommand;
import static de.z0rdak.yawp.util.ChatLinkBuilder.buildRegionInfoLink;
import static de.z0rdak.yawp.api.MessageSender.sendCmdFeedback;

public class IFlagArgumentType implements ArgumentType<String> {

    public static final Pattern VALID_FLAG_PATTERN = Pattern.compile("^[A-Za-z][A-Za-z\\-][A-Za-z]$");
    private static final Collection<String> EXAMPLES = RegionFlag.getFlagNames();
    private static final SimpleCommandExceptionType ERROR_AREA_INVALID = new SimpleCommandExceptionType(class_2561.method_48321("cli.arg.flag.parse.invalid", "Unable to parse flag identifier!"));
    private static final DynamicCommandExceptionType ERROR_INVALID_VALUE = new DynamicCommandExceptionType(
            flag -> class_2561.method_48322("cli.arg.flag.invalid", "Invalid flag identifier: '%s'", flag)
    );

    private IFlagArgumentType() {
    }

    @Nullable
    public static IFlag getFlag(CommandContext<class_2168> context, String argName) throws CommandSyntaxException {
        RegionType regionType = RegionArgumentType.getRegionType(context);
        String flagIdentifier = context.getArgument(argName, String.class);
        if (RegionFlag.contains(flagIdentifier) && regionType != null) {
            IProtectedRegion region = RegionArgumentType.getRegion(context, regionType);
            if (region.containsFlag(flagIdentifier)) {
                return region.getFlag(flagIdentifier);
            } else {
                class_5250 flagAddHint = class_2561.method_48322("cli.msg.info.region.flag.add-hint", "Add flag by clicking: %s", ChatLinkBuilder.buildAddFlagLink(region, flagIdentifier));
                class_5250 flagNotPresentInfo = class_2561.method_48322("cli.msg.info.region.flag.not-present", "Region %s does not contain flag '%s'. %s", buildRegionInfoLink(region), flagIdentifier, flagAddHint);
                sendCmdFeedback(context.getSource(), flagNotPresentInfo);
                return null;
            }
        } else {
            sendCmdFeedback(context.getSource(), class_2561.method_43470("Invalid flag identifier: '" + flagIdentifier + "'!"));
            throw ERROR_INVALID_VALUE.create(flagIdentifier);
        }
    }

    /**
     * Using this as an actual argument does not work on a server-side only mod,
     * because it needs to be registered in the corresponding registry.
     */
    public static IFlagArgumentType flag() {
        return new IFlagArgumentType();
    }

    @Override
    public String parse(StringReader reader) throws CommandSyntaxException {
        int i = reader.getCursor();

        while (reader.canRead() && String.valueOf(reader.peek()).matches(Pattern.compile("^[A-Za-z\\d\\-]$").pattern())) {
            reader.skip();
        }
        String s = reader.getString().substring(i, reader.getCursor());

        try {
            boolean isValidName = s.matches(VALID_FLAG_PATTERN.pattern());
            if (isValidName) {
                return s;
            } else {
                throw new IllegalArgumentException("Invalid flag identifier supplied");
            }
        } catch (IllegalArgumentException argumentException) {
            reader.setCursor(i);
            Constants.LOGGER.error("Error parsing flag identifier");
            throw ERROR_AREA_INVALID.createWithContext(reader);
        }
    }

    @Override
    public Collection<String> getExamples() {
        return EXAMPLES;
    }

    private <S> FlagEditType getEditType(CommandContext<S> context) {
        List<String> nodes = context.getNodes()
                .stream()
                .map(node -> node.getNode().getName())
                .collect(Collectors.toList());
        if (nodes.get(1).equals(CommandConstants.FLAG.toString())) {
            return FlagEditType.INFO;
        }
        if (nodes.contains(ADD.toString())) {
            return FlagEditType.ADD;
        }
        if (nodes.contains(REMOVE.toString())) {
            return FlagEditType.REMOVE;
        }
        return FlagEditType.UNSET;
    }

    private <S> List<String> getSuggestionFlags(FlagEditType flagEditType, IProtectedRegion region) throws CommandSyntaxException {
        List<String> flagsInRegion = region.getFlags().flags()
                .stream()
                .map(IFlag::getName)
                .distinct()
                .collect(Collectors.toList());
        switch (flagEditType) {
            case ADD: // show flags not in region
                List<String> flags = RegionFlag.getFlagNames();
                flags.removeAll(flagsInRegion);
                return flags;
            case REMOVE: // Only show existing flags
            case INFO:
                return flagsInRegion;
            case UNSET:
            default:
                return new ArrayList<>();
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public <S> CompletableFuture<Suggestions> listSuggestions(CommandContext<S> ctx, SuggestionsBuilder builder) {
        RegionType regionType = RegionArgumentType.getRegionType(ctx);
        boolean isCommandSource = ctx.getSource() instanceof class_2168;
        if (regionType == null) {
            if (isCommandSource) {
                sendCmdFeedback((class_2168) ctx.getSource(), class_2561.method_43470("Invalid region type supplied"));
            }
            return Suggestions.empty();
        }
        if (isCommandSource) {
            class_2168 src = (class_2168) ctx.getSource();
            try {
                IProtectedRegion region = RegionArgumentType.getRegion((CommandContext<class_2168>) ctx, regionType);
                FlagEditType flagEditType = getEditType(ctx);
                List<String> flagToSuggest = getSuggestionFlags(flagEditType, region);
                if ((flagEditType == FlagEditType.REMOVE || flagEditType == FlagEditType.INFO) && flagToSuggest.isEmpty()) {
                    if (src.method_43737()) {
                        overLayMessage(src.method_44023(), class_2561.method_48322("cli.msg.info.region.flag.no-flags-plain", "No flags defined in region %s!", buildRegionInfoLink(region)));
                        return Suggestions.empty();
                    }
                    class_5250 hint = class_2561.method_48322("cli.msg.info.region.flag.add-hint", "Add flag by clicking: %s", ChatLinkBuilder.buildSuggestAddFlagLink(region));
                    sendCmdFeedback(src, class_2561.method_48322("cli.msg.info.region.flag.no-flags", "No flags defined in region %s! %s", buildRegionInfoLink(region), hint));
                    return Suggestions.empty();
                }
                if (flagEditType == FlagEditType.ADD && flagToSuggest.isEmpty()) {
                    sendCmdFeedback(src, class_2561.method_48322("cli.msg.info.region.flag.all-flags", "Region %s already contains all flags!", buildRegionInfoLink(region)));
                    return Suggestions.empty();
                }
                return class_2172.method_9265(flagToSuggest, builder);
            } catch (CommandSyntaxException e) {
                throw new RuntimeException(e);
            }

        } else {
            return Suggestions.empty();
        }
    }
}
