package de.z0rdak.yawp.handler;

import com.mojang.brigadier.ParseResults;
import com.mojang.brigadier.context.CommandContextBuilder;
import com.mojang.brigadier.context.ParsedArgument;
import com.mojang.brigadier.context.ParsedCommandNode;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import de.z0rdak.yawp.api.commands.CommandConstants;
import de.z0rdak.yawp.api.core.RegionManager;
import de.z0rdak.yawp.api.permission.Permissions;
import de.z0rdak.yawp.commands.CommandSourceType;
import de.z0rdak.yawp.constants.Constants;
import de.z0rdak.yawp.core.region.GlobalRegion;
import de.z0rdak.yawp.core.region.IMarkableRegion;
import de.z0rdak.yawp.core.region.IProtectedRegion;
import de.z0rdak.yawp.data.region.LevelRegionData;
import de.z0rdak.yawp.data.region.RegionDataManager;
import de.z0rdak.yawp.platform.Services;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import net.minecraft.class_124;
import net.minecraft.class_1657;
import net.minecraft.class_1937;
import net.minecraft.class_2168;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_5321;
import net.minecraft.class_7924;

import static de.z0rdak.yawp.api.commands.CommandConstants.*;
import static de.z0rdak.yawp.util.ChatLinkBuilder.buildRegionInfoLink;
import static de.z0rdak.yawp.api.MessageSender.sendCmdFeedback;

public class CommandInterceptor {
    private final static int CANCEL_CMD = 1;
    private final static int ALLOW_CMD = 0;

    /**
     * Handler for managing different command permissions.
     */
    public static int handleModCommands(ParseResults<class_2168> parseResults, String command) {
        CommandContextBuilder<class_2168> cmdContext = parseResults.getContext();
        class_2168 src = cmdContext.getSource();
        List<ParsedCommandNode<class_2168>> cmdNodes = cmdContext.getNodes();
        List<String> nodeNames = cmdNodes.stream()
                .map(node -> node.getNode().getName())
                .collect(Collectors.toList());
        try {
            CommandSourceType cmdSrcType = CommandSourceType.of(src);
            if (!hasModBaseCmd(nodeNames)) {
                return ALLOW_CMD;
            }
            if (nodeNames.size() > 2) {
                Constants.LOGGER.debug("Executed command: '{}' by '{}'.", command, cmdContext.getSource().method_9214());
                String subCmd = nodeNames.get(1);
                int cancelExecutionResultCode = ALLOW_CMD;
                switch (subCmd) {
                    case "local":
                        if (!nodeNames.contains(CommandConstants.LOCAL.toString())) {
                            cancelExecutionResultCode = CANCEL_CMD;
                            break;
                        }
                        cancelExecutionResultCode = handleRegionCmdExecution(cmdContext, nodeNames, cmdSrcType);
                        break;
                    case "dim":
                        if (!nodeNames.contains(DIM.toString())) {
                            cancelExecutionResultCode = CANCEL_CMD;
                            break;
                        }
                        cancelExecutionResultCode = handleDimCommandExecution(cmdContext, cmdSrcType);
                        break;
                    case "global":
                        if (!nodeNames.contains(GLOBAL.toString())) {
                            cancelExecutionResultCode = CANCEL_CMD;
                            break;
                        }
                        cancelExecutionResultCode = verifyGlobalCommandPermission(cmdContext, cmdSrcType);
                        break;
                    case "flag":
                        if (!nodeNames.contains(FLAG.toString())) {
                            cancelExecutionResultCode = CANCEL_CMD;
                            break;
                        }
                        cancelExecutionResultCode = verifyFlagCommandPermission(cmdContext, nodeNames, cmdSrcType);
                        break;
                    case "marker":
                        if (!nodeNames.contains(MARKER.toString())) {
                            cancelExecutionResultCode = CANCEL_CMD;
                            break;
                        }
                        cancelExecutionResultCode = verifyMarkerCommandPermission(cmdContext, nodeNames, cmdSrcType);
                        break;
                    case "info":
                        if (!nodeNames.contains(CommandConstants.INFO.toString())) {
                            cancelExecutionResultCode = CANCEL_CMD;
                            break;
                        }
                        cancelExecutionResultCode = handleShortcutCmdExecution(cmdContext, cmdSrcType);
                        break;
                    case "create":
                        if (!nodeNames.contains(CommandConstants.CREATE.toString())) {
                            cancelExecutionResultCode = CANCEL_CMD;
                            break;
                        }
                        cancelExecutionResultCode = handleCreateShortcutCmdExecution(cmdContext, nodeNames, cmdSrcType);
                        break;
                    case "delete":
                        if (!nodeNames.contains(CommandConstants.DELETE.toString())) {
                            cancelExecutionResultCode = CANCEL_CMD;
                            break;
                        }
                        cancelExecutionResultCode = handleShortcutCmdExecution(cmdContext, cmdSrcType);
                        break;
                }
                return cancelExecutionResultCode;
            }
        } catch (RuntimeException e) {
            Constants.LOGGER.error(e);
            return CANCEL_CMD;
        }
        return ALLOW_CMD;
    }

    private static int handleShortcutCmdExecution(CommandContextBuilder<class_2168> cmdContext, CommandSourceType cmdSrcType) {
        class_2168 src = cmdContext.getSource();
        IProtectedRegion region = checkValidLocalRegionShortcut(cmdContext, LOCAL);
        if (region == null) {
            return CANCEL_CMD;
        }
        try {
            boolean hasPermission = hasCmdPermission(cmdContext, cmdSrcType, Permissions.OWNER, region);
            handlePermission(src, region, hasPermission);
            return hasPermission ? ALLOW_CMD : CANCEL_CMD;
        } catch (CommandSyntaxException e) {
            Constants.LOGGER.error(e);
            return CANCEL_CMD;
        }
    }

    private static int handleCreateShortcutCmdExecution(CommandContextBuilder<class_2168> cmdContext, List<String> nodeNames, CommandSourceType cmdSrcType) {
        class_2168 src = cmdContext.getSource();
        var maybeDimApi = RegionManager.get().getDimRegionApi(src.method_9225().method_27983());
        if (maybeDimApi.isEmpty()) {
            return CANCEL_CMD;
        }
        try {
            var dimApi = maybeDimApi.get();
            IProtectedRegion region = dimApi.getCache().getDim();
            boolean hasPermission = hasCmdPermission(cmdContext, cmdSrcType, Permissions.OWNER, region);
            handlePermission(src, region, hasPermission);
            return hasPermission ? ALLOW_CMD : CANCEL_CMD;
        } catch (CommandSyntaxException e) {
            Constants.LOGGER.error(e);
            return CANCEL_CMD;
        }
    }

    /**
     * Verifies the permission for the given marker command. <br>
     * Syntax: /wp marker reset|give|create [...] <br>
     */
    private static int verifyMarkerCommandPermission(CommandContextBuilder<class_2168> cmdContext, List<String> nodeNames, CommandSourceType cmdSrcType) {
        try {
            boolean isMarkerSubCmd = checkSubCmdAtIndex(nodeNames, 2, RESET, GIVE, CREATE);
            if (!isMarkerSubCmd) {
                return ALLOW_CMD;
            }
            if (cmdSrcType == CommandSourceType.PLAYER) {
                boolean isCreateCmd = checkSubCmdAtIndex(nodeNames, 2, CREATE);
                // 0   1      2       3      4
                // wp marker create <name> >parent>
                boolean isParentArgProvided = nodeNames.size() >= 5 && nodeNames.get(4) != null;
                class_1657 player = cmdContext.getSource().method_9207();
                LevelRegionData dimCache = RegionDataManager.getOrCreate(player.method_37908().method_27983());
                boolean hasPermission = Services.PERMISSION_CONFIG.hasConfigPermission(cmdContext.getSource(), cmdSrcType);
                boolean hasRegionPermission = false;
                if (isCreateCmd) {
                    if (isParentArgProvided) {
                        ParsedArgument<class_2168, ?> commandSourceParsedArgument = cmdContext.getArguments().get(nodeNames.get(4));
                        if (commandSourceParsedArgument.getResult() instanceof String parentName) {
                            IMarkableRegion parent = dimCache.getLocal(parentName);
                            if (parent != null) {
                                hasRegionPermission = Permissions.get().hasGroupPermission(parent, player, Permissions.OWNER);
                            }
                        }
                    } else { // assuming dimensional regions as parent
                        hasRegionPermission = Permissions.get().hasGroupPermission(dimCache.getDim(), player, Permissions.OWNER);
                    }
                } else {
                    hasRegionPermission = Permissions.get().hasGroupPermission(dimCache.getDim(), player, Permissions.OWNER);
                }
                hasPermission = hasPermission || hasRegionPermission;
                handlePermission(cmdContext.getSource(), hasPermission);
                return hasPermission ? ALLOW_CMD : CANCEL_CMD;
            }
            Constants.LOGGER.error("A player is required to execute this command.");
            return CANCEL_CMD;
        } catch (CommandSyntaxException e) {
            Constants.LOGGER.error(e);
            return CANCEL_CMD;
        }
    }


    /**
     * Verifies the permission for the given flag command. <br>
     * Syntax for Local Regions:        /wp flag local &lt;dim&gt; &lt;region&gt; &lt;flag&gt; enable|msg ... <br>
     * Syntax for Dimensional Regions:  /wp flag dim &lt;dim&gt; &lt;flag&gt; enable|msg ... <br>
     * Syntax for Global Regions:       /wp flag global &lt;flag&gt; enable|msg ... <br>
     * Note: Implementation can still be improved through generalization or fusing the flag command with the sub-flag command. <br>
     */
    private static int verifyFlagCommandPermission(CommandContextBuilder<class_2168> cmdContext, List<String> nodeNames, CommandSourceType cmdSrcType) {
        class_2168 src = cmdContext.getSource();
        try {
            boolean isRegionTypeCmd = checkSubCmdAtIndex(nodeNames, 2, CommandConstants.LOCAL, DIM, GLOBAL);
            if (!isRegionTypeCmd) {
                return ALLOW_CMD;
            }
            String regionTypeCmd = nodeNames.get(2);
            switch (regionTypeCmd) {
                case "local": {
                    IProtectedRegion region = checkValidLocalRegion(cmdContext);
                    if (region == null) {
                        return ALLOW_CMD;
                    }
                    Function<List<String>, Boolean> subCmdPermission = (nodes) -> {
                        //  0   1    2       3      4    5      6
                        // /wp flag local <dim> <region> <flag> enable|msg ...
                        int subCmdIdx = 6;
                        boolean isReadOnlyCmd = checkSubCmdAtIndex(nodes, subCmdIdx, INFO);
                        boolean isFlagShortCmd = nodes.size() == subCmdIdx;
                        return (isFlagShortCmd || isReadOnlyCmd) && Services.PERMISSION_CONFIG.isReadOnlyAllowed();
                    };
                    boolean hasPermission = hasCmdPermission(cmdContext, cmdSrcType, Permissions.OWNER, region, subCmdPermission);
                    handlePermission(src, region, hasPermission);
                    return hasPermission ? ALLOW_CMD : CANCEL_CMD;
                }
                case "dim": {
                    LevelRegionData dimCache = checkValidDimRegion(cmdContext);
                    if (dimCache == null) {
                        return ALLOW_CMD;
                    }
                    IProtectedRegion region = dimCache.getDim();
                    Function<List<String>, Boolean> subCmdPermission = (nodes) -> {
                        //  0   1    2     3    4    5          6
                        // /wp flag dim <dim> <flag> enable|msg ...
                        int subCmdIdx = 5;
                        boolean isReadOnlyCmd = checkSubCmdAtIndex(nodes, subCmdIdx, INFO);
                        boolean isFlagShortCmd = nodes.size() == subCmdIdx;
                        return (isFlagShortCmd || isReadOnlyCmd) && Services.PERMISSION_CONFIG.isReadOnlyAllowed();
                    };
                    boolean hasPermission = hasCmdPermission(cmdContext, cmdSrcType, Permissions.OWNER, region, subCmdPermission);
                    handlePermission(src, region, hasPermission);
                    return hasPermission ? ALLOW_CMD : CANCEL_CMD;
                }
                case "global": {
                    GlobalRegion region = RegionManager.get().getGlobalRegion();
                    Function<List<String>, Boolean> subCmdPermission = (nodes) -> {
                        //  0   1    2       3      4         5
                        // /wp flag global <flag> enable|msg ...
                        int subCmdIdx = 4;
                        boolean isReadOnlyCmd = checkSubCmdAtIndex(nodes, subCmdIdx, INFO);
                        boolean isFlagShortCmd = nodes.size() == subCmdIdx;
                        return (isFlagShortCmd || isReadOnlyCmd) && Services.PERMISSION_CONFIG.isReadOnlyAllowed();
                    };
                    boolean hasPermission = hasCmdPermission(cmdContext, cmdSrcType, Permissions.OWNER, region, subCmdPermission);
                    handlePermission(src, region, hasPermission);
                    return hasPermission ? ALLOW_CMD : CANCEL_CMD;
                }
                default:
                    break;
            }
            return ALLOW_CMD;
        } catch (CommandSyntaxException e) {
            Constants.LOGGER.error(e);
            return CANCEL_CMD;
        }
    }

    /**
     * Verifies the permission for the given flag command. <br>
     * Syntax: /wp global info|clear|add|remove|list|state.
     */
    private static int verifyGlobalCommandPermission(CommandContextBuilder<class_2168> cmdContext, CommandSourceType cmdSrcType) {
        class_2168 src = cmdContext.getSource();
        GlobalRegion region = RegionManager.get().getGlobalRegion();
        try {
            Function<List<String>, Boolean> subCmdPermission = (nodes) -> {
                //  0  1      2         3
                // /wp global info|list ...
                int subCmdIdx = 2;
                boolean isReadOnlyCmd = checkSubCmdAtIndex(nodes, subCmdIdx, INFO, LIST);
                boolean isExtendedInfoCmd = checkSubCmdAtIndex(nodes, subCmdIdx, STATE) && nodes.size() == subCmdIdx + 1;
                boolean isRegionShortCmd = nodes.size() == subCmdIdx;
                return (isRegionShortCmd || isReadOnlyCmd || isExtendedInfoCmd) && Services.PERMISSION_CONFIG.isReadOnlyAllowed();
            };
            boolean hasPermission = hasCmdPermission(cmdContext, cmdSrcType, Permissions.OWNER, region, subCmdPermission);
            handlePermission(src, region, hasPermission);
            return hasPermission ? ALLOW_CMD : CANCEL_CMD;
        } catch (CommandSyntaxException e) {
            Constants.LOGGER.error(e);
            return CANCEL_CMD;
        }
    }

    public static int handleRegionCmdExecution(CommandContextBuilder<class_2168> cmdContext, List<String> nodeNames, CommandSourceType cmdSrcType) {
        class_2168 src = cmdContext.getSource();
        IProtectedRegion region = checkValidLocalRegion(cmdContext);
        if (region == null) {
            return CANCEL_CMD;
        }
        try {
            int subCmdIndex = nodeNames.indexOf(AREA.toString());
            subCmdIndex = 4;
            Function<List<String>, Boolean> subCmdPermission = (nodes) -> {
                int subCmdIdx = nodes.indexOf(AREA.toString());
                subCmdIdx = 4;
                // nodes.size() > 5
                //  0   1      2      3      4    5
                // /wp local <dim> <region> area ...
                boolean isAreaWriteCmd = checkSubCmdAtIndex(nodes, subCmdIdx, AREA) && nodes.size() > subCmdIdx + 1;
                // nodes.size() == 6
                //  0   1      2      3      4    5     6
                // /wp local <dim> <region> area  tp [player]
                boolean isRegionTpCmd = isAreaWriteCmd && checkSubCmdAtIndex(nodes, subCmdIdx + 1, TELEPORT) && nodes.size() >= subCmdIdx + 2;
                // nodes.size() == 4 || (  nodes.size() == 5 + INFO || LIST  )
                //  0   1      2      3        4
                // /wp local <dim> <region> info|list ...
                boolean isReadOnlyCmd = checkSubCmdAtIndex(nodes, subCmdIdx, INFO, LIST);
                boolean isExtendedInfoCmd = checkSubCmdAtIndex(nodes, subCmdIdx, STATE, AREA) && nodes.size() == subCmdIdx + 1;
                boolean isRegionShortCmd = nodes.size() == subCmdIdx;
                boolean isReadOnlyAndAllowed = (isRegionShortCmd || isReadOnlyCmd || isExtendedInfoCmd) && Services.PERMISSION_CONFIG.isReadOnlyAllowed();
                boolean isRegionTpAndAllowed = isRegionTpCmd && Services.PERMISSION_CONFIG.allowRegionTp();
                return isReadOnlyAndAllowed || isRegionTpAndAllowed;
            };
            // nodes.size() > 5
            //  0   1      2      3      4    5
            // /wp local <dim> <region> area ...
            boolean isAreaWriteCmd = checkSubCmdAtIndex(nodeNames, subCmdIndex, AREA) && nodeNames.size() > subCmdIndex + 1;
            //  0    1      2      3       4     5     6
            // /wp local <dim> <region> area set|expand ...
            // needs to be handled separately, because the player also needs permission to modify the region parent
            boolean isAreaSetCmd = isAreaWriteCmd && checkSubCmdAtIndex(nodeNames, subCmdIndex + 1, SET) && nodeNames.size() > subCmdIndex + 2;
            boolean isAreaExpandCmd = isAreaWriteCmd && checkSubCmdAtIndex(nodeNames, subCmdIndex + 1, EXPAND) && nodeNames.size() > subCmdIndex + 2;
            boolean isAreaModifyCmd = isAreaWriteCmd && (isAreaExpandCmd || isAreaSetCmd);
            boolean hasParentPermission = hasCmdPermission(cmdContext, cmdSrcType, Permissions.OWNER, region.getParent());
            if (isAreaModifyCmd && !hasParentPermission) {
                handlePermission(src, region.getParent(), false);
                return CANCEL_CMD;
            }
            boolean hasPermission = hasCmdPermission(cmdContext, cmdSrcType, Permissions.OWNER, region, subCmdPermission);
            handlePermission(src, region, hasPermission);
            return hasPermission ? ALLOW_CMD : CANCEL_CMD;
        } catch (CommandSyntaxException e) {
            Constants.LOGGER.error(e);
            return CANCEL_CMD;
        }
    }

    public static int handleDimCommandExecution(CommandContextBuilder<class_2168> cmdContext, CommandSourceType cmdSrcType) {
        class_2168 src = cmdContext.getSource();
        LevelRegionData dimCache = checkValidDimRegion(cmdContext);
        if (dimCache == null) {
            return CANCEL_CMD;
        }
        try {
            IProtectedRegion region = dimCache.getDim();
            Function<List<String>, Boolean> subCmdPermission = (nodes) -> {
                //  0   1    2       3      4
                // /wp dim <dim> info|list ...
                int subCmdIdx = 3;
                boolean isReadOnlyCmd = checkSubCmdAtIndex(nodes, subCmdIdx, INFO, LIST);
                boolean isExtendedInfoCmd = checkSubCmdAtIndex(nodes, subCmdIdx, STATE) && nodes.size() == subCmdIdx + 1;
                boolean isRegionShortCmd = nodes.size() == subCmdIdx;
                return (isRegionShortCmd || isReadOnlyCmd || isExtendedInfoCmd) && Services.PERMISSION_CONFIG.isReadOnlyAllowed();
            };
            boolean hasPermission = hasCmdPermission(cmdContext, cmdSrcType, Permissions.OWNER, region, subCmdPermission);
            handlePermission(src, region, hasPermission);
            return hasPermission ? ALLOW_CMD : CANCEL_CMD;
        } catch (CommandSyntaxException e) {
            Constants.LOGGER.error(e);
            return CANCEL_CMD;
        }
    }

    private static boolean hasModBaseCmd(List<String> nodeNames) {
        return !nodeNames.isEmpty() && nodeNames.get(0) != null && nodeNames.get(0).equals(Constants.MOD_ID);
    }

    @Nullable
    private static LevelRegionData checkValidDimRegion(CommandContextBuilder<class_2168> cmdContext) {
        ParsedArgument<class_2168, ?> dimParsedArgument = cmdContext.getArguments().get(DIM.toString());
        if (dimParsedArgument != null && dimParsedArgument.getResult() instanceof class_2960 dimResLoc) {
            class_5321<class_1937> dim = class_5321.method_29179(class_7924.field_41223, dimResLoc);
            Optional<LevelRegionData> levelData = RegionManager.get().getLevelRegionData(dim);
            if (levelData.isEmpty()) {
                sendCmdFeedback(cmdContext.getSource(), class_2561.method_43470("Dimension not found in region data").method_27692(class_124.field_1061));
                return null;
            }
            return levelData.get();
        }
        return null;
    }

    @Nullable
    private static IProtectedRegion checkValidLocalRegionShortcut(CommandContextBuilder<class_2168> cmdContext, CommandConstants argumentKey) {
        ParsedArgument<class_2168, ?> regionArg = cmdContext.getArguments().get(argumentKey.toString());
        if (regionArg != null && regionArg.getResult() instanceof String regionName) {
            class_3218 level = cmdContext.getSource().method_9225();
            LevelRegionData dimCache = RegionDataManager.getOrCreate(level.method_27983());
            if (!dimCache.hasLocal(regionName)) {
                sendCmdFeedback(cmdContext.getSource(), class_2561.method_43470("No region with name '" + regionName + "' defined in dim '" + dimCache.getDim().getName() + "'"));
                return null;
            }
            IMarkableRegion region = dimCache.getLocal(regionName);
            if (region == null) {
                sendCmdFeedback(cmdContext.getSource(), class_2561.method_43470("No region with name '" + regionName + "' defined in dim '" + dimCache.getDim().getName() + "'"));
                return null;
            }
            return region;
        }
        return null;
    }

    @Nullable
    private static IProtectedRegion checkValidLocalRegion(CommandContextBuilder<class_2168> cmdContext) {
        ParsedArgument<class_2168, ?> regionArg = cmdContext.getArguments().get(CommandConstants.LOCAL.toString());
        if (regionArg != null && regionArg.getResult() instanceof String regionName) {
            ParsedArgument<class_2168, ?> dimParsedArgument = cmdContext.getArguments().get(DIM.toString());
            if (dimParsedArgument != null && dimParsedArgument.getResult() instanceof class_2960 dimResLoc) {
                class_5321<class_1937> dim = class_5321.method_29179(class_7924.field_41223, dimResLoc);
                LevelRegionData dimCache = RegionDataManager.getOrCreate(dim);
                if (!dimCache.hasLocal(regionName)) {
                    sendCmdFeedback(cmdContext.getSource(), class_2561.method_43470("No region with name '" + regionName + "' defined in dim '" + dimCache.getDim().getName() + "'"));
                    return null;
                }
                IMarkableRegion region = dimCache.getLocal(regionName);
                if (region == null) {
                    sendCmdFeedback(cmdContext.getSource(), class_2561.method_43470("No region with name '" + regionName + "' defined in dim '" + dimCache.getDim().getName() + "'"));
                    return null;
                }
                return region;
            }
        }
        return null;
    }

    private static boolean checkSubCmdAtIndex(List<String> nodeNames, int index, CommandConstants subCmd, CommandConstants... subCmds) {
        List<String> subCmdList = Arrays.stream(subCmds).map(CommandConstants::toString).collect(Collectors.toList());
        subCmdList.add(subCmd.toString());
        return nodeNames.size() >= index + 1 && nodeNames.get(index) != null && subCmdList.stream().anyMatch(nodeNames.get(index)::equals);
    }

    private static void handlePermission(class_2168 src, IProtectedRegion region, boolean hasPermission) {
        if (!hasPermission) {
            Constants.LOGGER.info("'{}' is not allowed to manage region '{}'", src.method_9214(), region.getName());
            sendCmdFeedback(src, class_2561.method_48322("cli.msg.info.region.modify.deny", "[%s] You don't have the permission to execute this command!", buildRegionInfoLink(region)));
        }
    }

    private static void handlePermission(class_2168 src, boolean hasPermission) {
        if (!hasPermission) {
            Constants.LOGGER.info("'{}' is not allowed to execute this command", src.method_9214());
            sendCmdFeedback(src, class_2561.method_48321("cli.msg.info.cmd.deny", "You don't have permission to execute this command!"));
        }
    }

    @SuppressWarnings("SameParameterValue")
    private static boolean hasCmdPermission(CommandContextBuilder<class_2168> ctx, CommandSourceType cmdSrcType, String permissionGroup, IProtectedRegion region) throws CommandSyntaxException {
        switch (cmdSrcType) {
            case PLAYER: {
                class_1657 player = ctx.getSource().method_9207();
                boolean hasConfigPermission = Permissions.get().hasConfigPermission(player);
                boolean hasRegionPermission = Permissions.get().hasGroupPermission(region, player, permissionGroup);
                return (hasRegionPermission || hasConfigPermission);
            }
            case SERVER:
                return true;
            case COMMAND_BLOCK:
                return Services.PERMISSION_CONFIG.isCommandBlockExecutionAllowed();
            default:
                return false;
        }
    }

    @SuppressWarnings("SameParameterValue")
    private static boolean hasCmdPermission(CommandContextBuilder<class_2168> ctx, CommandSourceType cmdSrcType, String permissionGroup, IProtectedRegion region, Function<List<String>, Boolean> subCmdPermission) throws CommandSyntaxException {
        switch (cmdSrcType) {
            case PLAYER: {
                List<String> nodeNames = ctx.getNodes().stream().map(node -> node.getNode().getName()).collect(Collectors.toList());
                class_1657 player = ctx.getSource().method_9207();
                boolean hasConfigPermission = Permissions.get().hasConfigPermission(player);
                boolean hasRegionPermission = Permissions.get().hasGroupPermission(region, player, permissionGroup);
                boolean hasSubCmdPermission = subCmdPermission.apply(nodeNames);
                return (hasRegionPermission || hasConfigPermission) || hasSubCmdPermission;
            }
            case SERVER:
                return true;
            case COMMAND_BLOCK:
                return Services.PERMISSION_CONFIG.isCommandBlockExecutionAllowed();
            default:
                return false;
        }
    }
}
