package de.z0rdak.yawp.api.core;

import de.z0rdak.yawp.core.area.CuboidArea;
import de.z0rdak.yawp.core.area.SphereArea;
import de.z0rdak.yawp.core.region.GlobalRegion;
import de.z0rdak.yawp.core.region.IMarkableRegion;
import de.z0rdak.yawp.core.region.IProtectedRegion;
import de.z0rdak.yawp.data.region.LevelRegionData;
import de.z0rdak.yawp.data.region.RegionDataManager;
import java.util.*;
import java.util.stream.Collectors;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2382;
import net.minecraft.class_2960;
import net.minecraft.class_3341;
import net.minecraft.class_5321;
import net.minecraft.class_7924;


public final class RegionManager implements IRegionManager {

    private RegionManager() {}

    public static RegionManager get() {
        return new RegionManager();
    }

    @Override
    public GlobalRegion getGlobalRegion() {
        return RegionDataManager.getGlobalRegion();
    }

    @Override
    public void resetGlobal() {
        RegionDataManager.getGlobalRegionData().reset();
        save();
    }

    @Override
    public Optional<IProtectedRegion> getDimensionalRegion(class_5321<class_1937> dim) {
        Optional<LevelRegionData> cache = getLevelRegionData(dim);
        return cache.map(LevelRegionData::getDim);
    }

    @Override
    public Optional<LevelRegionData> getLevelRegionData(class_5321<class_1937> dim) {
        return RegionDataManager.getLevelRegionData(dim);
    }

    @Override
    public void save() {
        RegionDataManager.save(true);
    }

    /**
     * Gets the DimensionalRegion API for the specified dimension key.
     *
     * @param dim the dimension key to get the API for
     * @return the DimensionalRegionApi for the specified dimension key if it exists, otherwise Optional.Empty
     */
    @Override
    public Optional<IDimensionRegionApi> getDimRegionApi(class_5321<class_1937> dim) {
        if (RegionDataManager.hasLevel(dim.method_29177())){
            LevelRegionData levelRegionData = RegionDataManager.getOrCreate(dim.method_29177());
            return Optional.of(new DimensionRegionApi(levelRegionData));
        }

        return Optional.empty();
    }

    /**
     * Gets the DimensionalRegion API for the specified dimension key (E.g. "minecraft:overworld").
     *
     * @param dimKey the dimension key to get the API for
     * @return the DimensionalRegionApi for the specified dimension key if it exists, otherwise Optional.Empty
     */
    @Override
    public Optional<IDimensionRegionApi> getDimRegionApiByKey(String dimKey) {
        return this.getDimRegionApi(getDimApiKey(dimKey));
    }

    @Override
    public class_5321<class_1937> getDimApiKey(String dimKey) {
        return class_5321.method_29179(class_7924.field_41223, class_2960.method_60654(dimKey));
    }

    @Override
    public boolean hasLevelData(class_5321<class_1937> dim) {
        return RegionDataManager.hasLevel(dim.method_29177());
    }

    @Override
    public boolean createDimRegion(class_5321<class_1937> dim) {
        if (hasLevelData(dim)) {
            return false;
        }
        RegionDataManager.getOrCreate(dim);
        save();
        return true;
    }

    @Override
    public Set<class_2960> getLevels() {
        return RegionDataManager.getLevels();
    }

    public Set<String> getLevelNames() {
        return RegionDataManager.getLevelNames();
    }

    @Override
    public void resetLevelData(class_5321<class_1937> dim) {
        RegionDataManager.resetLevelData(dim);
        save();
    }

    public static class DimensionRegionApi implements IDimensionRegionApi {
        private final LevelRegionData levelData;

        private DimensionRegionApi(LevelRegionData levelData) {
            this.levelData = levelData;
        }

        @Override
        public void save() {
            RegionManager.get().save();
        }

        @Override
        public Optional<IMarkableRegion> getLocalRegion(String name) {
            IMarkableRegion region = levelData.getLocal(name);
            return region != null ? Optional.of(region) : Optional.empty();
        }

        @Override
        public class_5321<class_1937> getDimKey() {
            return levelData.getDim().getDim();
        }

        @Override
        public LevelRegionData getCache() {
            return levelData;
        }

        @Override
        public boolean hasLocal(String name) {
            return levelData.hasLocal(name);
        }

        @Override
        public boolean addLocalRegion(IMarkableRegion region) {
            if (hasLocal(region.getName())) return false;
            levelData.addLocal(region);
            return true;
        }

        @Override
        public boolean removeLocal(IMarkableRegion region) {
            return removeLocalRegion(region.getName());
        }

        @Override
        public boolean removeLocalRegion(String regionName) {
            if (hasLocal(regionName)) {
                Optional<IMarkableRegion> localRegion = getLocalRegion(regionName);
                if (localRegion.isPresent()) {
                    levelData.removeLocal(localRegion.get());
                    return true;
                }
            }
            return false;
        }

        @Override
        public boolean hasRegionAt(class_2338 pos) {
            return !getRegionsAt(pos).isEmpty();
        }

        @Override
        public List<IMarkableRegion> getRegionsAt(class_2338 pos) {
            return getAllLocalRegions().stream()
                    .filter(region -> region.contains(pos))
                    .collect(Collectors.toList());
        }

        @Override
        public Collection<IMarkableRegion> getAllLocalRegions() {
            return this.levelData.getLocalList();
        }

        @Override
        public List<IMarkableRegion> getRegionsIn(class_2382 pos1, class_2382 pos2) {
            return getRegionsInBox(class_3341.method_34390(pos1, pos2));
        }

        @Override
        public List<IMarkableRegion> getRegionsInCoords(int x1, int y1, int z1, int x2, int y2, int z2) {
            return getRegionsIn(new class_2338(x1, y1, z1), new class_2338(x2, y2, z2));
        }

        @Override
        public List<IMarkableRegion> getRegionsInBox(class_3341 blockBox) {
            CuboidArea predicateArea = new CuboidArea(blockBox);
            return getAllLocalRegions().stream()
                    .filter(r -> predicateArea.containsOther(r.getArea()))
                    .toList();
        }

        @Override
        public List<IMarkableRegion> getRegionsAround(class_2338 pos, int radius) {
            SphereArea predicateArea = new SphereArea(pos, radius);
            return getAllLocalRegions().stream()
                    .filter(r -> predicateArea.intersects(r.getArea()))
                    .toList();
        }


        @Override
        public List<IMarkableRegion> getIntersectingRegions(class_3341 blockBox) {
            CuboidArea predicateArea = new CuboidArea(blockBox);
            return getAllLocalRegions().stream()
                    .filter(r -> predicateArea.intersects(r.getArea()))
                    .collect(Collectors.toList());
        }

        @Override
        public List<IMarkableRegion> getIntersectingRegions(IMarkableRegion region) {
            return getAllLocalRegions().stream()
                    .filter(r -> !r.equals(region)) // filter input region from the result
                    .filter(r -> r.getArea().intersects(region.getArea()))
                    .collect(Collectors.toList());
        }

        @Override
        public List<IMarkableRegion> getContainingRegions(IMarkableRegion region) {
            return getAllLocalRegions().stream()
                    .filter(r -> !r.equals(region)) // filter input region from the result
                    .filter(r -> r.getArea().containsOther(region.getArea()))
                    .collect(Collectors.toList());
        }

        @Override
        public List<IMarkableRegion> getContainedRegions(IMarkableRegion region) {
            return getAllLocalRegions().stream()
                    .filter(r -> !r.equals(region)) // filter input region from the result
                    .filter(r -> region.getArea().containsOther(r.getArea()))
                    .collect(Collectors.toList());
        }

        /**
         * Gets the region with the highest priority among all involved regions at the given location and dimension. <br>
         * This considers the active state of the region as well. <br>
         *
         * @param position the position to check for involved regions
         * @return the region with the highest priority among all involved regions which contain the given location
         */
        @Override
        public Optional<IMarkableRegion> getInvolvedRegionFor(class_2338 position) {
            List<IMarkableRegion> regionsForPos = getInvolvedRegionsFor(position);
            if (regionsForPos.isEmpty()) {
                return Optional.empty();
            } else {
                return Optional.of(Collections.max(regionsForPos, Comparator.comparing(IMarkableRegion::getPriority)));
            }
        }

        /**
         * Gets all active regions which contain the provided position in the  <br>
         *
         * @param pos the position to check for involved regions
         * @return all active regions which contain the given location and dimension
         */
        private List<IMarkableRegion> getInvolvedRegionsFor(class_2338 pos) {
            return getRegionsAt(pos).stream()
                    .filter(IMarkableRegion::isActive)
                    .collect(Collectors.toList());
        }

        @Override
        public Optional<IProtectedRegion> findResponsibleRegion(class_2338 pos) {
            Optional<IMarkableRegion> maybeRegion = getInvolvedRegionFor(pos);
            if (maybeRegion.isEmpty()) {
                IProtectedRegion dimRegion = levelData.getDim();
                if (dimRegion.isActive()) {
                    return Optional.of(dimRegion);
                } else {
                    return RegionManager.get().getGlobalRegion().isActive()
                            ? Optional.of(RegionManager.get().getGlobalRegion())
                            : Optional.empty();
                }
            }
            IProtectedRegion region = maybeRegion.get();
            return Optional.of(region);
        }
    }

}