package de.z0rdak.yawp.commands.arguments.flag;

import com.mojang.brigadier.StringReader;
import com.mojang.brigadier.arguments.ArgumentType;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.DynamicCommandExceptionType;
import com.mojang.brigadier.exceptions.SimpleCommandExceptionType;
import com.mojang.brigadier.suggestion.Suggestions;
import com.mojang.brigadier.suggestion.SuggestionsBuilder;
import de.z0rdak.yawp.commands.arguments.ArgumentUtil;
import de.z0rdak.yawp.constants.Constants;
import de.z0rdak.yawp.core.flag.RegionFlag;
import de.z0rdak.yawp.core.region.CuboidRegion;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import net.minecraft.class_2168;
import net.minecraft.class_2172;
import net.minecraft.class_2561;

import static de.z0rdak.yawp.api.MessageSender.sendCmdFeedback;

public class RegionFlagArgumentType implements ArgumentType<String> {

    public static final Pattern VALID_FLAG_PATTERN = Pattern.compile("^[A-Za-z][A-Za-z\\-][A-Za-z]$");
    private static final Collection<String> EXAMPLES = RegionFlag.getFlagNames();
    private static final SimpleCommandExceptionType ERROR_AREA_INVALID = new SimpleCommandExceptionType(class_2561.method_48321("cli.arg.flag.parse.invalid", "Unable to parse flag identifier!"));
    private static final DynamicCommandExceptionType ERROR_INVALID_VALUE = new DynamicCommandExceptionType(
            flag -> class_2561.method_48322("cli.arg.flag.invalid", "Invalid flag identifier: '%s'", flag)
    );

    /**
     * Using this as an actual argument does not work on a server-side only mod,
     * because it needs to be registered in the corresponding registry.
     */
    public static RegionFlagArgumentType flag() {
        return new RegionFlagArgumentType();
    }

    public static RegionFlag getFlag(CommandContext<class_2168> ctx, String argName) throws CommandSyntaxException {
        String flagIdentifier = ctx.getArgument(argName, String.class);
        if (RegionFlag.contains(flagIdentifier)) {
            return RegionFlag.fromId(flagIdentifier);
        } else {
            sendCmdFeedback(ctx.getSource(), class_2561.method_43470("Invalid flag identifier: '" + flagIdentifier + "'!"));
            throw ERROR_INVALID_VALUE.create(flagIdentifier);
        }
    }

    public static Set<RegionFlag> getFlags(CommandContext<class_2168> ctx, String argName) throws CommandSyntaxException {
        String flagIdentifiers = ctx.getArgument(argName, String.class);
        Set<String> flagsList = new HashSet<>(Arrays.asList(flagIdentifiers.split(" ")));
        Set<RegionFlag> regionFlags = flagsList.stream()
                .filter(flag -> {
                    if (RegionFlag.contains(flag))
                        return true;
                    else {
                        sendCmdFeedback(ctx.getSource(), class_2561.method_43470("Invalid flag identifier: '" + flag + "'!"));
                        return false;
                    }
                })
                .map(RegionFlag::fromId)
                .collect(Collectors.toSet());
        if (regionFlags.isEmpty()) {
            throw ERROR_INVALID_VALUE.create(flagIdentifiers);
        }
        return regionFlags;
    }

    @Override
    public String parse(StringReader reader) throws CommandSyntaxException {
        int i = reader.getCursor();

        while (reader.canRead() && String.valueOf(reader.peek()).matches(Pattern.compile("^[A-Za-z\\d\\-]$").pattern())) {
            reader.skip();
        }
        String s = reader.getString().substring(i, reader.getCursor());

        try {
            boolean isValidName = s.matches(VALID_FLAG_PATTERN.pattern());
            if (isValidName) {
                return s;
            } else {
                throw new IllegalArgumentException("Invalid flag identifier supplied");
            }
        } catch (IllegalArgumentException argumentException) {
            reader.setCursor(i);
            Constants.LOGGER.error("Error parsing flag identifier");
            throw ERROR_AREA_INVALID.createWithContext(reader);
        }
    }

    @Override
    public Collection<String> getExamples() {
        return EXAMPLES;
    }

    // TODO: Replace usages with IFlagArgumentType where possible
    @Override
    @SuppressWarnings("unchecked")
    public <S> CompletableFuture<Suggestions> listSuggestions(CommandContext<S> ctx, SuggestionsBuilder builder) {
        if (ctx.getSource() instanceof class_2168 src) {
            CuboidRegion region = (CuboidRegion) ArgumentUtil.getRegionArgument((CommandContext<class_2168>) ctx);
            List<String> flagNames = RegionFlag.getFlagNames();

            String input = ctx.getInput();
            if (input.contains("add")) {
                flagNames = flagNames.stream()
                        .filter(flagName -> !region.containsFlag(flagName))
                        .collect(Collectors.toList());
            }
            if (input.contains("remove")) {
                flagNames = flagNames.stream()
                        .filter(region::containsFlag)
                        .collect(Collectors.toList());
            }
            if (flagNames.isEmpty()) {
                if (input.contains("add")) {
                    sendCmdFeedback(src, class_2561.method_43470("There are no flag left to add for this region '" + region.getName() + "'."));
                }
                if (input.contains("remove")) {
                    sendCmdFeedback(src, class_2561.method_43470("Region '" + region.getName() + "' does not contain any flags."));
                }
                return Suggestions.empty();
            }
            return class_2172.method_9265(flagNames, builder);
        } else {
            return Suggestions.empty();
        }
    }
}
