package de.z0rdak.yawp.core.area;

import com.mojang.serialization.Codec;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import de.z0rdak.yawp.util.AreaUtil;
import org.apache.commons.lang3.NotImplementedException;

import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_3341;

import static de.z0rdak.yawp.util.AreaUtil.distance;
import static de.z0rdak.yawp.util.AreaUtil.distanceManhattan;

import D;

public class SphereArea extends CenteredArea {

    public static Codec<SphereArea> CODEC = RecordCodecBuilder.create(instance -> instance.group(
                    class_2338.field_25064.fieldOf("center")
                            .forGetter(SphereArea::getCenterPos),
                    Codec.INT.fieldOf("radius")
                            .forGetter(SphereArea::getRadius),
                    Codec.STRING.fieldOf("areaType")
                            .forGetter(r -> MarkedAreaTypes.areaIdentifier(r.getAreaType()).toString()),
                    BlockDisplayProperties.CODEC.fieldOf("display")
                            .orElse(BlockDisplayProperties.createRndDefault())
                            .forGetter(MarkedArea::getDisplay)
            ).apply(instance, (center, radius, area, display) -> {
                var sphere = new SphereArea(center, radius);
                sphere.updateDisplay(display);
                return sphere;
            })
    );

    private final int radius;

    public SphereArea(class_2338 centerPos, class_2338 scopePos) {
        super(centerPos, AreaType.SPHERE);
        this.radius = (int) (distance(centerPos, scopePos) + 0.5);
    }

    public SphereArea(class_2338 middlePos, int radius) {
        this(middlePos, new class_2338(middlePos).method_10069(0, radius, 0));
    }

    public static SphereArea expand(SphereArea area, int expansion) {
        var expanded = new SphereArea(area.center, Math.max(area.radius + expansion, 0));
        expanded.updateDisplay(area.getDisplay());
        return expanded;
    }

    public int getRadius() {
        return this.radius;
    }

    @Override
    public boolean contains(class_2338 pos) {
        return distance(this.center, pos) < this.radius + 0.5;
    }

    public boolean isHullBlock(class_2338 pos) {
        var d = distance(this.center, pos);
        return d > this.radius - 0.5 && d < this.radius + 0.5;
    }

    @Override
    public Set<class_2338> getHull() {
        class_2338 p1 = this.center.method_10069(-this.radius, -this.radius, -this.radius);
        class_2338 p2 = new class_2338(this.center).method_10069(this.radius, this.radius, this.radius);
        class_3341 cube = class_3341.method_34390(p1, p2);
        Set<class_2338> cubeBlocks = AreaUtil.blocksIn(cube);
        return cubeBlocks.stream().filter(this::isHullBlock).collect(Collectors.toSet());
    }

    @Override
    public Set<class_2338> getFrame() {
        Set<class_2338> frameBlocks = this.getMinimalOutline();

        int halfRadius = this.radius / 2;
        frameBlocks.addAll(AreaUtil.getSliceBlocks(this.center, this.radius, halfRadius, class_2350.class_2351.field_11048, this::isHullBlock));
        frameBlocks.addAll(AreaUtil.getSliceBlocks(this.center, this.radius, -halfRadius, class_2350.class_2351.field_11048, this::isHullBlock));
        frameBlocks.addAll(AreaUtil.getSliceBlocks(this.center, this.radius, halfRadius, class_2350.class_2351.field_11052, this::isHullBlock));
        frameBlocks.addAll(AreaUtil.getSliceBlocks(this.center, this.radius, -halfRadius, class_2350.class_2351.field_11052, this::isHullBlock));
        frameBlocks.addAll(AreaUtil.getSliceBlocks(this.center, this.radius, halfRadius, class_2350.class_2351.field_11051, this::isHullBlock));
        frameBlocks.addAll(AreaUtil.getSliceBlocks(this.center, this.radius, -halfRadius, class_2350.class_2351.field_11051, this::isHullBlock));
        return frameBlocks;
    }

    @Override
    public Set<class_2338> getMinimalOutline() {
        Set<class_2338> frameBlocks = new HashSet<>();
        frameBlocks.addAll(AreaUtil.getSliceBlocks(this.center, this.radius, 0, class_2350.class_2351.field_11048, this::isHullBlock));
        frameBlocks.addAll(AreaUtil.getSliceBlocks(this.center, this.radius, 0, class_2350.class_2351.field_11052, this::isHullBlock));
        frameBlocks.addAll(AreaUtil.getSliceBlocks(this.center, this.radius, 0, class_2350.class_2351.field_11051, this::isHullBlock));
        return frameBlocks;
    }

    public boolean contains(CuboidArea inner) {
        double maxDistance = Double.NEGATIVE_INFINITY;
        // Calculate the maximum distance from the sphere center to cuboid vertices
        for (class_2338 vertex : inner.getVertices()) {
            double distance = distanceManhattan(this.center, vertex);
            if (distance > maxDistance) {
                maxDistance = distance;
            }
        }
        // Check if the maximum distance is less than or equal to the sphere's radius
        return maxDistance <= this.getRadius();
    }

    public boolean contains(SphereArea inner) {
        return distanceManhattan(this.center, inner.center) + inner.radius <= this.radius;
    }

    public boolean intersects(CuboidArea other) {
        // Check if sphere's center is inside the cuboid
        if (other.contains(this.center)) {
            return true;
        }
        // Compute the closest point on the cuboid to the sphere
        int closestX = Math.max(other.getArea().method_35415(), Math.min(this.center.method_10263(), other.getArea().method_35418()));
        int closestY = Math.max(other.getArea().method_35416(), Math.min(this.center.method_10264(), other.getArea().method_35419()));
        int closestZ = Math.max(other.getArea().method_35417(), Math.min(this.center.method_10260(), other.getArea().method_35420()));
        // Calculate the distance between the sphere center and the closest point
        return distanceManhattan(this.center, new class_2338(closestX, closestY, closestZ)) <= this.radius;
    }

    public boolean intersects(SphereArea other) {
        return distanceManhattan(this.center, other.center) <= this.radius + other.radius;
    }

    @Override
    public boolean containsOther(IMarkableArea inner) {
        switch (inner.getAreaType()) {
            case CUBOID:
                return this.contains((CuboidArea) inner);
            case SPHERE:
                return this.contains((SphereArea) inner);
            default:
                throw new NotImplementedException("Area type not implemented yet");
        }
    }

    @Override
    public boolean intersects(IMarkableArea other) {
        switch (other.getAreaType()) {
            case CUBOID:
                return this.intersects((CuboidArea) other);
            case SPHERE:
                return this.intersects((SphereArea) other);
            default:
                throw new NotImplementedException("Area type not implemented yet");
        }
    }

    @Override
    public MarkedAreaType<?> getType() {
        return MarkedAreaTypes.SPHERE_AREA;
    }

    @Override
    public String toString() {
        return "Sphere " + AreaUtil.blockPosStr(this.center) + ", r=" + radius;
    }
}
