package de.z0rdak.yawp.commands;

import com.mojang.brigadier.arguments.BoolArgumentType;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.builder.RequiredArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import de.z0rdak.yawp.api.commands.CommandConstants;
import de.z0rdak.yawp.api.core.RegionManager;
import de.z0rdak.yawp.api.events.flag.FlagEvent;
import de.z0rdak.yawp.commands.arguments.ArgumentUtil;
import de.z0rdak.yawp.commands.arguments.flag.IFlagArgumentType;
import de.z0rdak.yawp.commands.arguments.region.RegionArgumentType;
import de.z0rdak.yawp.core.flag.FlagMessage;
import de.z0rdak.yawp.core.flag.FlagState;
import de.z0rdak.yawp.core.flag.IFlag;
import de.z0rdak.yawp.core.region.IProtectedRegion;
import de.z0rdak.yawp.platform.Services;
import de.z0rdak.yawp.util.text.Messages;
import de.z0rdak.yawp.util.text.messages.multiline.MultiLineMessage;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2172;
import net.minecraft.class_2181;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_5250;

import static de.z0rdak.yawp.api.commands.CommandConstants.*;
import static de.z0rdak.yawp.commands.arguments.ArgumentUtil.*;
import static de.z0rdak.yawp.util.ChatLinkBuilder.*;
import static de.z0rdak.yawp.api.MessageSender.sendCmdFeedback;

final class FlagCommands {

    private FlagCommands() {
    }

    static LiteralArgumentBuilder<class_2168> build() {
        return literal(FLAG)
                .then(literal(GLOBAL)
                        .executes(ctx -> CommandUtil.promptRegionFlagList(ctx, getGlobalRegion(), 0))
                        .then(flagSubCmd((ctx) -> getGlobalRegion())))
                .then(literal(DIM)
                        .then(flagDimSubCommands()))
                .then(literal(LOCAL)
                        .then(flagLocalSubCommands()));
    }

    private static RequiredArgumentBuilder<class_2168, class_2960> flagDimSubCommands() {
        return class_2170.method_9244(DIM.toString(), class_2181.method_9288())
                .executes(ctx -> CommandUtil.promptRegionFlagList(ctx, getLevelDataArgument(ctx).getDim(), 0))
                .then(flagSubCmd((ctx) -> getLevelDataArgument(ctx).getDim()));
    }

    private static RequiredArgumentBuilder<class_2168, class_2960> flagLocalSubCommands() {
        return class_2170.method_9244(DIM.toString(), class_2181.method_9288())
                .then(class_2170.method_9244(CommandConstants.LOCAL.toString(), StringArgumentType.word())
                        .suggests((ctx, builder) -> RegionArgumentType.region().listSuggestions(ctx, builder))
                        .executes(ctx -> CommandUtil.promptRegionFlagList(ctx, getLevelDataArgument(ctx).getDim(), 0))
                        .then(flagSubCmd(ArgumentUtil::getRegionArgument))
                );
    }

    private static RequiredArgumentBuilder<class_2168, String> flagSubCmd(Function<CommandContext<class_2168>, IProtectedRegion> regionSupplier) {
        return class_2170.method_9244(FLAG.toString(), StringArgumentType.word())
                .suggests((ctx, builder) -> IFlagArgumentType.flag().listSuggestions(ctx, builder))
                .executes(ctx -> promptFlagInfo(ctx, regionSupplier.apply(ctx), getIFlagArgument(ctx)))
                .then(literal(INFO)
                        .executes(ctx -> promptFlagInfo(ctx, regionSupplier.apply(ctx), getIFlagArgument(ctx)))
                )
                .then(literal(STATE)
                        .executes(ctx -> setFlagState(ctx, regionSupplier.apply(ctx), getIFlagArgument(ctx)))
                        .then(class_2170.method_9244(STATE.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> class_2172.method_9265(FlagState.ValidFlagStates(), builder))
                                .executes(ctx -> setFlagState(ctx, regionSupplier.apply(ctx), getIFlagArgument(ctx), getFlagStateArgument(ctx))))
                )
                .then(literal(OVERRIDE)
                        .executes(ctx -> setOverride(ctx, regionSupplier.apply(ctx), getIFlagArgument(ctx)))
                        .then(class_2170.method_9244(OVERRIDE.toString(), BoolArgumentType.bool())
                                .executes(ctx -> setOverride(ctx, regionSupplier.apply(ctx), getIFlagArgument(ctx), getOverrideArgument(ctx))))
                )
                .then(literal(MSG)
                        .then(literal(MUTE)
                                .executes(ctx -> setFlagMuteState(ctx, regionSupplier.apply(ctx), getIFlagArgument(ctx)))
                                .then(class_2170.method_9244(MUTE.toString(), BoolArgumentType.bool())
                                        .executes(ctx -> setFlagMuteState(ctx, regionSupplier.apply(ctx), getIFlagArgument(ctx), muteArgument(ctx))))
                        )
                        .then(literal(SET)
                                .then(class_2170.method_9244(MSG.toString(), StringArgumentType.string())
                                        .suggests((ctx, builder) -> class_2172.method_9265(flagMsgExamples(), builder))
                                        .executes(ctx -> setRegionFlagMsg(ctx, regionSupplier.apply(ctx), getIFlagArgument(ctx), getFlagMsgArgument(ctx))))
                        )
                        .then(literal(CLEAR)
                                .executes(ctx -> setRegionFlagMsg(ctx, regionSupplier.apply(ctx), getIFlagArgument(ctx), FlagMessage.CONFIG_MSG))
                        )
                );
    }

    private static List<String> flagMsgExamples() {
        final int amountOfExamples = 10;
        List<String> examples = new ArrayList<>(amountOfExamples);
        for (int i = 0; i < amountOfExamples; i++) {
            examples.add(class_2561.method_48321("cli.flag.msg.text.example." + i, "<Your flag message here>").getString());
        }
        return examples;
    }

    private static int promptFlagInfo(CommandContext<class_2168> ctx, IProtectedRegion region, IFlag flag) {
        if (flag == null) return 1;
        MultiLineMessage.send(ctx.getSource(), MultiLineMessage.flagDetail(region, flag));
        return 0;
    }

    private static int setFlagMuteState(CommandContext<class_2168> ctx, IProtectedRegion region, IFlag regionFlag) {
        if (regionFlag == null) return 1;
        if (region.containsFlag(regionFlag.getName())) {
            IFlag flag = region.getFlag(regionFlag.getName());
            return setFlagMuteState(ctx, region, flag, !flag.getFlagMsg().isMuted());
        } else {
            class_5250 hint = class_2561.method_48322("cli.msg.info.region.flag.add-hint", "Add flag by clicking: %s", buildSuggestAddFlagLink(region));
            sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.info.region.flag.not-present", "Region %s does not contain flag '%s'. %",
                    buildRegionInfoLink(region), regionFlag.getName(), hint));
            return 1;
        }
    }

    private static int setFlagMuteState(CommandContext<class_2168> ctx, IProtectedRegion region, IFlag flag, boolean setMuted) {
        if (flag == null) return 1;
        flag.getFlagMsg().mute(setMuted);
        String muteState = flag.getFlagMsg().isMuted() ? "on" : "off";
        class_5250 infoMsg = class_2561.method_48322("cli.flag.msg.mute.success.text", "Set mute state of %s to: '%s'",
                buildFlagInfoLink(region, flag), muteState);
        class_5250 undoLink = buildRegionActionUndoLink(ctx.getInput(), String.valueOf(!setMuted), String.valueOf(setMuted));
        class_5250 msg = Messages.substitutable("%s %s", infoMsg, undoLink);
        sendCmdFeedback(ctx.getSource(), msg);
        RegionManager.get().save();
        return 0;

    }

    private static int setRegionFlagMsg(CommandContext<class_2168> ctx, IProtectedRegion region, IFlag flag, String flagMsgStr) {
        if (flag == null) return 1;
        String oldFlagMsg = flag.getFlagMsg().msg();

        FlagEvent.UpdateFlagMessageEvent editMsgEvent = new FlagEvent.UpdateFlagMessageEvent(ctx.getSource(), region, flag, flagMsgStr);
        Services.EVENT.post(editMsgEvent);

        FlagMessage flagMsg = new FlagMessage(flagMsgStr, flag.getFlagMsg().isMuted());
        flag.setFlagMsg(flagMsg);
        class_5250 infoMsg = class_2561.method_48322("cli.flag.msg.msg.success.text", "Set message of %s to: '%s'",
                buildFlagInfoLink(region, flag), flagMsgStr);
        class_5250 undoLink = buildRegionActionUndoLink(ctx.getInput(), flagMsgStr, oldFlagMsg);
        class_5250 msg = Messages.substitutable("%s %s", infoMsg, undoLink);
        sendCmdFeedback(ctx.getSource(), msg);
        RegionManager.get().save();
        return 0;
    }

    private static int setFlagState(CommandContext<class_2168> ctx, IProtectedRegion region, IFlag regionFlag) {
        if (regionFlag == null) return 1;
        if (region.containsFlag(regionFlag.getName())) {
            IFlag flag = region.getFlag(regionFlag.getName());
            if (flag.getState() == FlagState.ALLOWED || flag.getState() == FlagState.DENIED) {
                return setFlagState(ctx, region, regionFlag, FlagState.invert(flag.getState()));
            }
            if (flag.getState() == FlagState.DISABLED) {
                return setFlagState(ctx, region, regionFlag, FlagState.DENIED);
            }
            return setFlagState(ctx, region, regionFlag, flag.getState());
        } else {
            class_5250 hint = class_2561.method_48322("cli.msg.info.region.flag.add-hint", "Add flag by clicking: %s", buildSuggestAddFlagLink(region));
            sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.info.region.flag.not-present", "Region %s does not contain flag '%s'. %",
                    buildRegionInfoLink(region), regionFlag.getName(), hint));
            return 1;
        }
    }

    private static int setFlagState(CommandContext<class_2168> ctx, IProtectedRegion region, IFlag flag, FlagState flagState) {
        if (flag == null) return 1;
        FlagState oldState = flag.getState();
        flag.setState(flagState);
        class_5250 undoLink = buildRegionActionUndoLink(ctx.getInput(), flagState.name, oldState.name);
        class_5250 infoMsg = class_2561.method_48322("cli.flag.state.success.text", "Set flag state of %s to: '%s'",
                buildFlagInfoLink(region, flag), flag.getState().name);
        class_5250 msg = Messages.substitutable("%s %s", infoMsg, undoLink);
        sendCmdFeedback(ctx.getSource(), msg);
        RegionManager.get().save();
        return 0;

    }

    private static int setOverride(CommandContext<class_2168> ctx, IProtectedRegion region, IFlag regionFlag) {
        if (regionFlag == null) return 1;
        if (region.containsFlag(regionFlag.getName())) {
            IFlag flag = region.getFlag(regionFlag.getName());
            return setOverride(ctx, region, flag, !flag.doesOverride());
        } else {
            class_5250 hint = class_2561.method_48322("cli.msg.info.region.flag.add-hint", "Add flag by clicking: %s", buildSuggestAddFlagLink(region));
            sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.info.region.flag.not-present", "Region %s does not contain flag '%s'. %",
                    buildRegionInfoLink(region), regionFlag.getName(), hint));
            return 1;
        }
    }

    private static int setOverride(CommandContext<class_2168> ctx, IProtectedRegion region, IFlag flag, boolean override) {
        if (flag == null) return 1;
        flag.setOverride(override);
        String overrideState = flag.doesOverride() ? "on" : "off";
        class_5250 infoMsg = class_2561.method_48322("cli.flag.override.success.text", "Set flag override for %s to %s",
                buildFlagInfoLink(region, flag), overrideState);
        class_5250 undoLink = buildRegionActionUndoLink(ctx.getInput(), String.valueOf(!override), String.valueOf(override));
        class_5250 msg = Messages.substitutable("%s %s", infoMsg, undoLink);
        sendCmdFeedback(ctx.getSource(), msg);
        RegionManager.get().save();
        return 0;
    }

}
