package de.z0rdak.yawp.commands;

import com.mojang.authlib.GameProfile;
import com.mojang.brigadier.arguments.BoolArgumentType;
import com.mojang.brigadier.arguments.IntegerArgumentType;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import de.z0rdak.yawp.api.commands.CommandConstants;
import de.z0rdak.yawp.api.core.RegionManager;
import de.z0rdak.yawp.api.events.flag.FlagEvent;
import de.z0rdak.yawp.api.permission.Permissions;
import de.z0rdak.yawp.commands.arguments.flag.IFlagArgumentType;
import de.z0rdak.yawp.commands.arguments.region.RegionArgumentType;
import de.z0rdak.yawp.core.flag.BooleanFlag;
import de.z0rdak.yawp.core.flag.FlagState;
import de.z0rdak.yawp.core.flag.IFlag;
import de.z0rdak.yawp.core.flag.RegionFlag;
import de.z0rdak.yawp.core.group.GroupType;
import de.z0rdak.yawp.core.region.IMarkableRegion;
import de.z0rdak.yawp.core.region.IProtectedRegion;
import de.z0rdak.yawp.core.region.RegionType;
import de.z0rdak.yawp.platform.Services;
import de.z0rdak.yawp.util.ChatLinkBuilder;
import de.z0rdak.yawp.util.MojangApiHelper;
import de.z0rdak.yawp.util.text.Messages;
import de.z0rdak.yawp.util.text.messages.multiline.MultiLineMessage;
import de.z0rdak.yawp.util.text.messages.pagination.*;
import org.apache.commons.lang3.NotImplementedException;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;
import net.minecraft.class_1657;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2172;
import net.minecraft.class_2181;
import net.minecraft.class_2186;
import net.minecraft.class_2243;
import net.minecraft.class_2561;
import net.minecraft.class_270;
import net.minecraft.class_3222;
import net.minecraft.class_5242;
import net.minecraft.class_5250;

import static de.z0rdak.yawp.api.commands.CommandConstants.*;
import static de.z0rdak.yawp.commands.arguments.ArgumentUtil.*;
import static de.z0rdak.yawp.util.ChatComponentBuilder.*;
import static de.z0rdak.yawp.api.MessageSender.sendCmdFeedback;
import static de.z0rdak.yawp.api.MessageSender.sendError;
import static net.minecraft.class_124.field_1061;

public class CommandUtil {

    public static LiteralArgumentBuilder<class_2168> buildClearSubCommand(Function<CommandContext<class_2168>, IProtectedRegion> regionSupplier) {
        return literal(CLEAR)
                .then(literal(FLAGS)
                        .executes(ctx -> CommandUtil.clearFlags(ctx, regionSupplier.apply(ctx)))
                )
                .then(literal(PLAYERS)
                        .executes(ctx -> CommandUtil.clearPlayers(ctx, regionSupplier.apply(ctx)))
                        .then(class_2170.method_9244(GROUP.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> class_2172.method_9265(Permissions.GROUP_LIST, builder))
                                .executes(ctx -> CommandUtil.clearPlayers(ctx, regionSupplier.apply(ctx), getGroupArgument(ctx))))
                )
                .then(literal(TEAMS)
                        .executes(ctx -> CommandUtil.clearTeams(ctx, regionSupplier.apply(ctx)))
                        .then(class_2170.method_9244(GROUP.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> class_2172.method_9265(Permissions.GROUP_LIST, builder))
                                .executes(ctx -> CommandUtil.clearTeams(ctx, regionSupplier.apply(ctx), getGroupArgument(ctx))))
                )
                .then(literal(GROUP)
                        .then(class_2170.method_9244(GROUP.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> class_2172.method_9265(Permissions.GROUP_LIST, builder))
                                .executes(ctx -> CommandUtil.clearGroups(ctx, regionSupplier.apply(ctx), getGroupArgument(ctx))))
                );
    }

    public static LiteralArgumentBuilder<class_2168> buildRemoveSubCommand(Function<CommandContext<class_2168>, IProtectedRegion> regionSupplier) {
        return literal(REMOVE)
                .then(literal(PLAYER)
                        .then(class_2170.method_9244(GROUP.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> class_2172.method_9265(Permissions.GROUP_LIST, builder))
                                .then(class_2170.method_9244(PLAYER.toString(), class_2186.method_9308())
                                        .executes(ctx -> removePlayers(ctx, getPlayersArgument(ctx), regionSupplier.apply(ctx), getGroupArgument(ctx))))
                                .then(literal(BY_UUID)
                                        .then(class_2170.method_9244(PLAYER_UUID.toString(), class_5242.method_27643())
                                                .suggests((ctx, builder) -> class_2172.method_9265(regionSupplier.apply(ctx).getGroup(getGroupArgument(ctx)).getPlayers().keySet().stream().map(UUID::toString).collect(Collectors.toList()), builder))
                                                .executes(ctx -> removePlayerByUUID(ctx, getPlayerUUIDArgument(ctx), regionSupplier.apply(ctx), getGroupArgument(ctx)))))
                                .then(literal(BY_NAME)
                                        .then(class_2170.method_9244(PLAYER_NAMES.toString(), StringArgumentType.greedyString())
                                                .suggests((ctx, builder) -> class_2172.method_9265(regionSupplier.apply(ctx).getGroup(getGroupArgument(ctx)).getPlayers().values(), builder))
                                                .executes(ctx -> removePlayersByName(ctx, getPlayerNamesArgument(ctx), regionSupplier.apply(ctx), getGroupArgument(ctx)))))
                        )
                )
                .then(literal(TEAM)
                        .then(class_2170.method_9244(GROUP.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> class_2172.method_9265(Permissions.GROUP_LIST, builder))
                                .then(class_2170.method_9244(TEAM.toString(), class_2243.method_9482())
                                        .executes(ctx -> removeTeam(ctx, getTeamArgument(ctx), regionSupplier.apply(ctx), getGroupArgument(ctx))))))
                .then(literal(FLAG)
                        .then(class_2170.method_9244(FLAG.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> IFlagArgumentType.flag().listSuggestions(ctx, builder))
                                .executes(ctx -> removeRegionFlag(ctx, regionSupplier.apply(ctx), getFlagArgument(ctx)))
                        )
                )
                .then(literal(FLAGS)
                        .then(class_2170.method_9244(FLAGS.toString(), StringArgumentType.greedyString())
                                .suggests((ctx, builder) -> IFlagArgumentType.flag().listSuggestions(ctx, builder))
                                .executes(ctx -> removeFlags(ctx, regionSupplier.apply(ctx), getFlagArguments(ctx))))
                );
    }

    public static LiteralArgumentBuilder<class_2168> buildAddSubCommand(Function<CommandContext<class_2168>, IProtectedRegion> regionSupplier) {
        return literal(ADD)
                .then(literal(PLAYER)
                        .then(class_2170.method_9244(GROUP.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> class_2172.method_9265(Permissions.GROUP_LIST, builder))
                                .then(class_2170.method_9244(PLAYER.toString(), class_2186.method_9308())
                                        .executes(ctx -> addPlayers(ctx, getPlayersArgument(ctx), regionSupplier.apply(ctx), getGroupArgument(ctx))))
                                .then(literal(BY_UUID)
                                        .then(class_2170.method_9244(PLAYER_UUID.toString(), class_5242.method_27643())
                                                .executes(ctx -> addPlayerByUuid(ctx, getPlayerUUIDArgument(ctx), regionSupplier.apply(ctx), getGroupArgument(ctx)))))
                                .then(literal(BY_NAME)
                                        .then(class_2170.method_9244(PLAYER_NAMES.toString(), StringArgumentType.greedyString())
                                                .executes(ctx -> addPlayersByName(ctx, getPlayerNamesArgument(ctx), regionSupplier.apply(ctx), getGroupArgument(ctx)))))
                        )
                )
                .then(literal(TEAM)
                        .then(class_2170.method_9244(GROUP.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> class_2172.method_9265(Permissions.GROUP_LIST, builder))
                                .then(class_2170.method_9244(TEAM.toString(), class_2243.method_9482())
                                        .executes(ctx -> addTeam(ctx, getTeamArgument(ctx), regionSupplier.apply(ctx), getGroupArgument(ctx))))))
                .then(literal(FLAG)
                        .then(class_2170.method_9244(FLAG.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> IFlagArgumentType.flag().listSuggestions(ctx, builder))
                                .executes(ctx -> addFlag(ctx, regionSupplier.apply(ctx), getFlagArgument(ctx)))
                                .then(class_2170.method_9244(STATE.toString(), StringArgumentType.word())
                                        .suggests((ctx, builder) -> class_2172.method_9265(FlagState.ValidFlagStates(), builder))
                                        .executes(ctx -> addFlag(ctx, regionSupplier.apply(ctx), getFlagArgument(ctx), getFlagStateArgument(ctx), false))
                                        .then(class_2170.method_9244(OVERRIDE.toString(), BoolArgumentType.bool())
                                                .executes(ctx -> addFlag(ctx, regionSupplier.apply(ctx), getFlagArgument(ctx), getFlagStateArgument(ctx), getOverrideArgument(ctx))))
                                )
                        )
                )
                .then(literal(FLAGS)
                        .then(class_2170.method_9244(FLAGS.toString(), StringArgumentType.greedyString())
                                .suggests((ctx, builder) -> IFlagArgumentType.flag().listSuggestions(ctx, builder))
                                .executes(ctx -> addFlags(ctx, regionSupplier.apply(ctx), getFlagArguments(ctx))))
                )
                .then(literal(ALL_FLAGS)
                        .executes(ctx -> addAllFlags(ctx, regionSupplier.apply(ctx)))
                );
    }

    public static LiteralArgumentBuilder<class_2168> buildListSubCommand(Function<CommandContext<class_2168>, IProtectedRegion> regionSupplier) {
        return literal(LIST)
                .then(literal(CHILDREN)
                        .executes(ctx -> promptRegionChildren(ctx, regionSupplier.apply(ctx), 0))
                        .then(class_2170.method_9244(PAGE.toString(), IntegerArgumentType.integer(0))
                                .executes(ctx -> promptRegionChildren(ctx, regionSupplier.apply(ctx), getPageNoArgument(ctx)))))
                .then(literal(FLAG)
                        .executes(ctx -> promptFlagList(ctx, regionSupplier.apply(ctx), 0))
                        .then(class_2170.method_9244(PAGE.toString(), IntegerArgumentType.integer(0))
                                .executes(ctx -> promptFlagList(ctx, regionSupplier.apply(ctx), getPageNoArgument(ctx)))))
                .then(literal(REGION_FLAG)
                        .executes(ctx -> promptRegionFlagList(ctx, regionSupplier.apply(ctx), 0))
                        .then(class_2170.method_9244(PAGE.toString(), IntegerArgumentType.integer(0))
                                .executes(ctx -> promptRegionFlagList(ctx, regionSupplier.apply(ctx), getPageNoArgument(ctx)))))
                .then(literal(GROUP)
                        .then(class_2170.method_9244(GROUP.toString(), StringArgumentType.word())
                                .suggests((ctx, builder) -> class_2172.method_9265(Permissions.GROUP_LIST, builder))
                                .executes(ctx -> CommandUtil.promptGroupLinks(ctx, regionSupplier.apply(ctx), getGroupArgument(ctx)))
                                .then(literal(TEAM)
                                        .executes(ctx -> CommandUtil.promptGroupList(ctx, regionSupplier.apply(ctx), getGroupArgument(ctx), GroupType.TEAM, 0))
                                        .then(class_2170.method_9244(PAGE.toString(), IntegerArgumentType.integer(0))
                                                .executes(ctx -> CommandUtil.promptGroupList(ctx, regionSupplier.apply(ctx), getGroupArgument(ctx), GroupType.TEAM, getPageNoArgument(ctx))))
                                )
                                .then(literal(PLAYER)
                                        .executes(ctx -> CommandUtil.promptGroupList(ctx, regionSupplier.apply(ctx), getGroupArgument(ctx), GroupType.PLAYER, 0))
                                        .then(class_2170.method_9244(PAGE.toString(), IntegerArgumentType.integer(0))
                                                .executes(ctx -> CommandUtil.promptGroupList(ctx, regionSupplier.apply(ctx), getGroupArgument(ctx), GroupType.PLAYER, getPageNoArgument(ctx))))
                                )
                        )
                );
    }

    public static LiteralArgumentBuilder<class_2168> buildCopySubCommand(Function<CommandContext<class_2168>, IProtectedRegion> srcSupplier) {
        return literal(COPY)
                .then(literal(FLAGS)
                        .then(literal(TO_LOCAL)
                                .then(class_2170.method_9244(TARGET_DIM.toString(), class_2181.method_9288())
                                        .then(class_2170.method_9244(TARGET_REGION.toString(), StringArgumentType.word())
                                                .suggests((ctx, builder) -> RegionArgumentType.region().listRegionsInTargetDim(ctx, builder))
                                                .executes(ctx -> copyRegionFlags(ctx, srcSupplier.apply(ctx), getTargetLocalRegionArgument(ctx))))))
                        .then(literal(TO_DIM)
                                .then(class_2170.method_9244(TARGET_DIM.toString(), class_2181.method_9288())
                                        .executes(ctx -> copyRegionFlags(ctx, srcSupplier.apply(ctx), getTargetDimRegionArgument(ctx).getDim()))))
                )
                .then(literal(PLAYERS)
                        .then(literal(TO_LOCAL)
                                .then(class_2170.method_9244(TARGET_DIM.toString(), class_2181.method_9288())
                                        .then(class_2170.method_9244(TARGET_REGION.toString(), StringArgumentType.word())
                                                .suggests((ctx, builder) -> RegionArgumentType.region().listRegionsInTargetDim(ctx, builder))
                                                .then(class_2170.method_9244(GROUP.toString(), StringArgumentType.word())
                                                        .suggests((ctx, builder) -> class_2172.method_9265(Permissions.GROUP_LIST, builder))
                                                        .executes(ctx -> copyRegionPlayers(ctx, srcSupplier.apply(ctx), getTargetLocalRegionArgument(ctx), getGroupArgument(ctx)))))))
                        .then(literal(TO_DIM)
                                .then(class_2170.method_9244(TARGET_DIM.toString(), class_2181.method_9288())
                                        .then(class_2170.method_9244(GROUP.toString(), StringArgumentType.word())
                                                .suggests((ctx, builder) -> class_2172.method_9265(Permissions.GROUP_LIST, builder))
                                                .executes(ctx -> copyRegionPlayers(ctx, srcSupplier.apply(ctx), getTargetDimRegionArgument(ctx).getDim(), getGroupArgument(ctx)))))))
                .then(literal(STATE)
                        .then(literal(TO_LOCAL)
                                .then(class_2170.method_9244(TARGET_DIM.toString(), class_2181.method_9288())
                                        .then(class_2170.method_9244(TARGET_REGION.toString(), StringArgumentType.word())
                                                .suggests((ctx, builder) -> RegionArgumentType.region().listRegionsInTargetDim(ctx, builder))
                                                .executes(ctx -> copyRegionState(ctx, srcSupplier.apply(ctx), getTargetLocalRegionArgument(ctx))))))
                        .then(literal(TO_DIM)
                                .then(class_2170.method_9244(TARGET_DIM.toString(), class_2181.method_9288())
                                        .executes(ctx -> copyRegionState(ctx, srcSupplier.apply(ctx), getTargetDimRegionArgument(ctx).getDim()))))
                );
    }

    public static int promptRegionInfo(CommandContext<class_2168> ctx, IProtectedRegion region) {
        MultiLineMessage.send(ctx.getSource(),  MultiLineMessage.regionInfo(region));
        return 0;
    }

    public static int promptRegionState(CommandContext<class_2168> ctx, IProtectedRegion region) {
        MultiLineMessage.send(ctx.getSource(), MultiLineMessage.regionState(region));
        return 0;
    }

    /**
     * == Group '%s' for '%s'==
     * Players: [n player(s)][+]
     * Teams: [m team(s)][+]
     */
    public static int promptGroupLinks(CommandContext<class_2168> ctx, IProtectedRegion region, String group) {
        if (!Permissions.GROUP_LIST.contains(group)) {
            sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.region.info.group.invalid", "cli.msg.region.info.group.invalid", group).method_27692(field_1061));
            return -1;
        }
        sendCmdFeedback(ctx.getSource(), buildGroupListHeader(region, group));
        sendCmdFeedback(ctx.getSource(), ChatLinkBuilder.buildGroupPlayerListLink(region, group));
        sendCmdFeedback(ctx.getSource(), ChatLinkBuilder.buildGroupTeamListLink(region, group));
        return 0;
    }

    /**
     * List of flags of the provided region and its parents
     */
    public static int promptFlagList(CommandContext<class_2168> ctx, IProtectedRegion region, int pageNo) {
        try {
            int paginationSize = Services.REGION_CONFIG.getPaginationSize();
            ResponsibleFlagPagination flagPagination = new ResponsibleFlagPagination(region, pageNo, paginationSize);
            MultiLineMessage.send(ctx.getSource(), flagPagination);
        } catch (InvalidPageNumberException e) {
            sendError(ctx.getSource(), e.getError());
            return -1;
        }
        return 0;
    }

    /**
     * List of flags only of the provided region
     */
    public static int promptRegionFlagList(CommandContext<class_2168> ctx, IProtectedRegion region, int pageNo) {
        try {
            int paginationSize = Services.REGION_CONFIG.getPaginationSize();
            RegionFlagPagination flagPagination = new RegionFlagPagination(region, pageNo, paginationSize);
            MultiLineMessage.send(ctx.getSource(), flagPagination);
        } catch (InvalidPageNumberException e) {
            sendError(ctx.getSource(), e.getError());
            return -1;
        }
        return 0;
    }

    public static int promptRegionChildren(CommandContext<class_2168> ctx, IProtectedRegion region, int pageNo) {
        try {
            int paginationSize = Services.REGION_CONFIG.getPaginationSize();
            ChildRegionPagination childRegionPagination = new ChildRegionPagination(region, region.getChildren().values().stream().toList(), pageNo, paginationSize);
            MultiLineMessage.send(ctx.getSource(), childRegionPagination);
        } catch (InvalidPageNumberException e) {
            sendError(ctx.getSource(), e.getError());
            return -1;
        }

        return 0;
    }

    public static int promptGroupList(CommandContext<class_2168> ctx, IProtectedRegion region, String group, GroupType groupType, int pageNo) {
        if (!Permissions.GROUP_LIST.contains(group)) {
            sendError(ctx.getSource(), buildInvalidGroupMsg(group));
            return -1;
        }
        List<String> groupNames = getGroupList(region, group, groupType);
        try {
            int paginationSize = Services.REGION_CONFIG.getPaginationSize();
            GroupMemberPagination childRegionPagination = new GroupMemberPagination(region, group, groupType, groupNames, pageNo, paginationSize);
            MultiLineMessage.send(ctx.getSource(), childRegionPagination);
        } catch (InvalidPageNumberException e) {
            sendError(ctx.getSource(), e.getError());
            return -1;
        }
        return 0;
    }

    public static int setActiveState(CommandContext<class_2168> ctx, IProtectedRegion region, boolean activate) {
        region.setIsActive(activate);
        RegionManager.get().save();
        class_5250 undoLink = ChatLinkBuilder.buildRegionActionUndoLink(ctx.getInput(), String.valueOf(!activate), String.valueOf(activate));
        class_5250 msg = class_2561.method_48322("cli.msg.info.region.state.enable.set.value", "Region state of %s is now: %s",
                ChatLinkBuilder.buildRegionInfoLink(region), region.isActive() ? "active" : "inactive");
        sendCmdFeedback(ctx.getSource(), Messages.substitutable("%s %s", msg, undoLink));
        return 0;
    }

    public static int setAlertState(CommandContext<class_2168> ctx, IProtectedRegion region, boolean showAlert) {
        region.setIsMuted(!showAlert);
        RegionManager.get().save();
        class_5250 undoLink = ChatLinkBuilder.buildRegionActionUndoLink(ctx.getInput(), String.valueOf(!showAlert), String.valueOf(showAlert));
        class_5250 msg = class_2561.method_48322("cli.msg.info.state.alert.set.value", "Flag messages of %s are now: %s",
                ChatLinkBuilder.buildRegionInfoLink(region), region.isMuted() ? "muted" : "active");
        sendCmdFeedback(ctx.getSource(), Messages.substitutable("%s %s", msg, undoLink));
        return 0;
    }

    private static class_5250 buildInvalidGroupMsg(String group) {
        return class_2561.method_48322("cli.msg.region.info.group.invalid", "Group '%s' is not defined!", group);
    }

    public static int removeTeam(CommandContext<class_2168> ctx, class_270 team, IProtectedRegion region, String group) {
        if (!Permissions.GROUP_LIST.contains(group)) {
            sendError(ctx.getSource(), buildInvalidGroupMsg(group));
            return -1;
        }
        class_5250 undoLink = ChatLinkBuilder.buildRegionActionUndoLink(ctx.getInput(), REMOVE, ADD);
        class_5250 teamInfo = buildGroupInfo(region, team.method_1197(), GroupType.TEAM);
        if (region.getGroup(group).hasTeam(team.method_1197())) {
            region.removeTeam(team.method_1197(), group);
            RegionManager.get().save();
            class_5250 msg = class_2561.method_48322("cli.msg.info.region.group.team.removed", "Removed team '%s' (group '%s') from %s", teamInfo, group,
                    ChatLinkBuilder.buildRegionInfoLink(region));
            sendCmdFeedback(ctx.getSource(), Messages.substitutable("%s %s", msg, undoLink));
            return 0;
        }
        class_5250 msg = class_2561.method_48322("cli.msg.info.region.group.team.not-present", "Team '%s' (group '%s') is not present in %s", teamInfo, group,
                ChatLinkBuilder.buildRegionInfoLink(region));
        sendCmdFeedback(ctx.getSource(), msg);
        return 1;
    }


    private static int removePlayersByName(CommandContext<class_2168> ctx, Collection<String> playerNames, IProtectedRegion region, String group) {
        if (!Permissions.GROUP_LIST.contains(group)) {
            sendError(ctx.getSource(), buildInvalidGroupMsg(group));
            return -1;
        }
        playerNames.forEach(playerName -> CommandUtil.removePlayer(ctx, playerName, region, group));
        return 0;
    }

    private static void removePlayer(CommandContext<class_2168> ctx, String playerName, IProtectedRegion region, String group) {
        region.getGroup(group).getPlayers().entrySet()
                .stream()
                .filter(e -> e.getValue().equals(playerName))
                .findFirst()
                .ifPresent(e -> removePlayer(ctx, e.getKey(), playerName, region, group));
    }

    private static int removePlayerByUUID(CommandContext<class_2168> ctx, UUID playerUuid, IProtectedRegion region, String group) {
        if (!Permissions.GROUP_LIST.contains(group)) {
            sendError(ctx.getSource(), buildInvalidGroupMsg(group));
            return -1;
        }
        if (region.getGroup(group).hasPlayer(playerUuid)) {
            String playerName = region.getGroup(group).getPlayers().get(playerUuid);
            return removePlayer(ctx, playerUuid, playerName, region, group);
        }
        return 1;
    }


    private static int removePlayers(CommandContext<class_2168> ctx, Collection<class_3222> players, IProtectedRegion region, String group) {
        players.forEach(player -> CommandUtil.removePlayer(ctx, player, region, group));
        return 0;
    }

    private static int removePlayer(CommandContext<class_2168> ctx, class_1657 player, IProtectedRegion region, String group) {
        if (!Permissions.GROUP_LIST.contains(group)) {
            sendError(ctx.getSource(), buildInvalidGroupMsg(group));
            return -1;
        }
        if (region.getGroup(group).hasPlayer(player.method_5667())) {
            String playerName = region.getGroup(group).getPlayers().get(player.method_5667());
            return removePlayer(ctx, player.method_5667(), playerName, region, group);
        }
        return 1;
    }

    public static int removePlayer(CommandContext<class_2168> ctx, UUID playerUuid, String playerName, IProtectedRegion region, String group) {
        class_5250 playerInfo = buildGroupInfo(region, playerName, GroupType.PLAYER);
        class_5250 undoLink = ChatLinkBuilder.buildRegionActionUndoLink(ctx.getInput(), REMOVE, ADD);
        if (region.getGroup(group).hasPlayer(playerUuid)) {
            region.removePlayer(playerUuid, group);
            class_5250 msg = class_2561.method_48322("cli.msg.info.region.group.player.removed", "Removed player '%s' (group '%s') from %s", playerInfo, group,
                    ChatLinkBuilder.buildRegionInfoLink(region));
            sendCmdFeedback(ctx.getSource(), Messages.substitutable("%s %s", msg, undoLink));
            RegionManager.get().save();
            return 0;
        }

        class_5250 msg = class_2561.method_48322("cli.msg.info.region.group.player.not-present", "Player '%s' (group '%s') is not present in %s", playerInfo, group,
                ChatLinkBuilder.buildRegionInfoLink(region));
        sendCmdFeedback(ctx.getSource(), msg);
        return 1;
    }

    private static int addPlayerByUuid(CommandContext<class_2168> ctx, UUID playerUuid, IProtectedRegion region, String group) {
        if (!Permissions.GROUP_LIST.contains(group)) {
            sendError(ctx.getSource(), buildInvalidGroupMsg(group));
            return -1;
        }
        Optional<GameProfile> cachedProfile = MojangApiHelper.lookupGameProfileInCache(ctx, playerUuid);
        if (cachedProfile.isPresent()) {
            GameProfile profile = cachedProfile.get();
            var isComplete = profile.getId() != null && StringUtils.isNotBlank(profile.getName());
            if (isComplete) {
                class_5250 cacheSuccess = class_2561.method_48322("cli.msg.info.player.lookup.cache.success", "Found entry for '%s' in the cache", playerUuid.toString());
                sendCmdFeedback(ctx.getSource(), cacheSuccess);
                return addPlayer(ctx, profile.getId(), profile.getName(), region, group);
            }
            return -1;
        } else {
            class_5250 cacheMiss = class_2561.method_48322("cli.msg.info.player.lookup.pending", "No cache entry existing for '%s'. Looking up %s at Mojang...",
                    playerUuid.toString(), "uuid");
            sendCmdFeedback(ctx.getSource(), cacheMiss);
            CompletableFuture.runAsync(() -> MojangApiHelper.getGameProfileInfo(playerUuid, (gameProfile) -> {
                if (gameProfile != null) {
                    class_5250 lookupSuccess = class_2561.method_48322("cli.msg.info.player.lookup.api.success", "Successfully retrieved game profile info for '%s' from Mojang", playerUuid.toString());
                    sendCmdFeedback(ctx.getSource(), lookupSuccess);
                    addPlayer(ctx, gameProfile.getId(), gameProfile.getName(), region, group);
                } else {
                    class_5250 lookupFailed = class_2561.method_48322("cli.msg.info.player.lookup.api.failed", "Unable to retrieve game profile info for '%s' from Mojang", playerUuid.toString());
                    sendCmdFeedback(ctx.getSource(), lookupFailed);
                }
            }));
            return 0;
        }
    }

    public static int addPlayersByName(CommandContext<class_2168> ctx, List<String> playerNames, IProtectedRegion region, String group) {
        if (!Permissions.GROUP_LIST.contains(group)) {
            sendError(ctx.getSource(), buildInvalidGroupMsg(group));
            return -1;
        }
        playerNames.forEach(name -> CommandUtil.addPlayerByName(ctx, name, region, group));
        return 0;
    }

    private static int addPlayerByName(CommandContext<class_2168> ctx, String playerName, IProtectedRegion region, String group) {
        Optional<GameProfile> cachedProfile = MojangApiHelper.lookupGameProfileInCache(ctx, playerName);
        if (cachedProfile.isPresent()) {
            GameProfile profile = cachedProfile.get();
            var isComplete = profile.getId() != null && StringUtils.isNotBlank(profile.getName());
            if (isComplete) {
                class_5250 cacheSuccess = class_2561.method_48322("cli.msg.info.player.lookup.cache.success", "Found entry for '%s' in the cache", playerName);
                sendCmdFeedback(ctx.getSource(), cacheSuccess);
                return addPlayer(ctx, profile.getId(), profile.getName(), region, group);
            }
            return -1;
        } else {
            class_5250 cacheMiss = class_2561.method_48322("cli.msg.info.player.lookup.pending", "No cache entry existing for '%s'. Looking up %s at Mojang...",
                    playerName, "name");
            sendCmdFeedback(ctx.getSource(), cacheMiss);
            CompletableFuture.runAsync(() -> MojangApiHelper.getGameProfileInfo(playerName, (gameProfile) -> {
                if (gameProfile != null) {
                    class_5250 lookupSuccess = class_2561.method_48322("cli.msg.info.player.lookup.api.success", "Successfully retrieved game profile info for '%s' from Mojang", playerName);
                    sendCmdFeedback(ctx.getSource(), lookupSuccess);
                    addPlayer(ctx, gameProfile.getId(), gameProfile.getName(), region, group);
                } else {
                    class_5250 lookupFailed = class_2561.method_48322("cli.msg.info.player.lookup.api.failed", "Unable to retrieve game profile info for '%s' from Mojang", playerName);
                    sendCmdFeedback(ctx.getSource(), lookupFailed);
                }
            }));
            return 0;
        }
    }

    public static int addPlayers(CommandContext<class_2168> ctx, Collection<class_3222> players, IProtectedRegion region, String group) {
        players.forEach(player -> CommandUtil.addPlayer(ctx, player, region, group));
        return 0;
    }

    public static int addPlayer(CommandContext<class_2168> ctx, class_1657 player, IProtectedRegion region, String group) {
        if (!Permissions.GROUP_LIST.contains(group)) {
            sendError(ctx.getSource(), buildInvalidGroupMsg(group));
            return -1;
        }
        return addPlayer(ctx, player.method_5667(), player.method_5820(), region, group);
    }

    private static int addPlayer(CommandContext<class_2168> ctx, UUID uuid, String name, IProtectedRegion region, String group) {
        class_5250 regionInfoLink = ChatLinkBuilder.buildRegionInfoLink(region);
        class_5250 undoLink = ChatLinkBuilder.buildRegionActionUndoLink(ctx.getInput(), ADD, REMOVE);
        if (!region.hasPlayer(uuid, group)) {
            region.addPlayer(uuid, name, group);
            RegionManager.get().save();
            class_5250 msg = class_2561.method_48322("cli.msg.info.region.group.player.added", "Added player '%s' as '%s' to %s", name, group, regionInfoLink);
            sendCmdFeedback(ctx.getSource(), Messages.substitutable("%s %s", msg, undoLink));
            return 0;
        }
        class_5250 msg = class_2561.method_48322("cli.msg.info.region.group.player.present", "Player '%s' (group '%s') already present in %s", name, group, regionInfoLink);
        sendCmdFeedback(ctx.getSource(), msg);
        return 1;
    }

    public static int addTeam(CommandContext<class_2168> ctx, class_270 team, IProtectedRegion region, String group) {
        if (!Permissions.GROUP_LIST.contains(group)) {
            sendError(ctx.getSource(), buildInvalidGroupMsg(group));
            return -1;
        }
        class_5250 regionInfoLink = ChatLinkBuilder.buildRegionInfoLink(region);
        class_5250 teamHoverInfo = buildTeamHoverComponent(team);
        if (!region.hasTeam(team.method_1197(), group)) {
            region.addTeam(team.method_1197(), group);
            RegionManager.get().save();
            class_5250 undoLink = ChatLinkBuilder.buildRegionActionUndoLink(ctx.getInput(), ADD, REMOVE);
            class_5250 msg = class_2561.method_48322("cli.msg.info.region.group.team.added", "Added team '%s' as '%s' to region %s",
                    teamHoverInfo, group, regionInfoLink);
            sendCmdFeedback(ctx.getSource(), Messages.substitutable("%s %s", msg, undoLink));
            return 0;
        }
        class_5250 msg = class_2561.method_48322("cli.msg.info.region.group.team.present", "Team '%s' (group '%s') already present in %s",
                teamHoverInfo, group, regionInfoLink);
        sendCmdFeedback(ctx.getSource(), msg);
        return 1;
    }

    public static int removeFlags(CommandContext<class_2168> ctx, IProtectedRegion region, Set<RegionFlag> flags) {
        flags.forEach(flag -> CommandUtil.removeRegionFlag(ctx, region, flag));
        return 0;
    }

    public static int removeRegionFlag(CommandContext<class_2168> ctx, IProtectedRegion region, RegionFlag flag) {
        if (region.containsFlag(flag)) {
            IFlag iFlag = region.getFlag(flag.name);
            FlagEvent.RemoveFlagEvent removeFlagEvent = new FlagEvent.RemoveFlagEvent(ctx.getSource(), region, iFlag);
            Services.EVENT.post(removeFlagEvent);
            region.removeFlag(flag.name);
            RegionManager.get().save();
            class_5250 msg = class_2561.method_48322("cli.msg.flag.removed", "Removed flag '%s' from %s", flag.name,
                    ChatLinkBuilder.buildRegionInfoLink(region));
            class_5250 undoLink = ChatLinkBuilder.buildRegionActionUndoLink(ctx.getInput(), REMOVE, ADD);
            sendCmdFeedback(ctx.getSource(), Messages.substitutable("%s %s", msg, undoLink));
            return 0;
        } else {
            class_5250 msg = class_2561.method_48322("cli.msg.flag.not-present", "Flag '%s' is not present in %s", flag.name,
                    ChatLinkBuilder.buildRegionInfoLink(region));
            sendCmdFeedback(ctx.getSource(), msg);
            return 1;
        }
    }

    public static int copyRegionFlags(CommandContext<class_2168> ctx, IProtectedRegion srcRegion, IProtectedRegion targetRegion) {
        if (srcRegion.getFlags().isEmpty()) {
            sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.info.region.flag.empty", "No flags defined in %s", ChatLinkBuilder.buildRegionInfoLink(srcRegion)));
            return 1;
        }
        // get the flags which are in the srcRegion but not in the targetRegion
        Set<IFlag> flagsToCopy = srcRegion.getFlags().flags().stream()
                .filter(flag -> !targetRegion.containsFlag(flag.getName()))
                .collect(Collectors.toSet());
        // flagsToCopy.forEach(region::addFlag);
        srcRegion.getFlags().flags().forEach(targetRegion::addFlag);
        RegionManager.get().save();
        sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.copy.region.flags", "Copied %s flag(s) from region %s to %s", flagsToCopy.size(), ChatLinkBuilder.buildRegionInfoLink(srcRegion), ChatLinkBuilder.buildRegionInfoLink(targetRegion)));
        return 0;
    }

    public static int copyRegionState(CommandContext<class_2168> ctx, IProtectedRegion srcRegion, IProtectedRegion targetRegion) {
        targetRegion.setIsActive(srcRegion.isActive());
        targetRegion.setIsMuted(srcRegion.isMuted());
        if (srcRegion instanceof IMarkableRegion regionSource && targetRegion instanceof IMarkableRegion regionTarget) {
            regionTarget.setPriority(regionSource.getPriority());
        }
        RegionManager.get().save();
        sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.copy.region.state", "Copied state from region %s to %s", ChatLinkBuilder.buildRegionInfoLink(srcRegion), ChatLinkBuilder.buildRegionInfoLink(targetRegion)));
        return 0;
    }

    public static int copyRegionPlayers(CommandContext<class_2168> ctx, IProtectedRegion srcRegion, IProtectedRegion targetRegion, String group) {
        if (!srcRegion.getGroup(group).getPlayers().isEmpty()) {
            // get the players which are in the srcRegion but not in the targetRegion
            Set<Map.Entry<UUID, String>> playerEntriesToCopy = srcRegion.getGroup(group).getPlayers().entrySet().stream()
                    .filter(entry -> !targetRegion.getGroup(group).getPlayers().containsKey(entry.getKey()))
                    .collect(Collectors.toSet());
            //playerEntriesToCopy.forEach(entry -> region.getGroup(group).addPlayer(entry.getKey(), entry.getValue()));

            srcRegion.getGroup(group).getPlayers().forEach((uuid, name) -> targetRegion.getGroup(group).addPlayer(uuid, name));
            RegionManager.get().save();
            sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.copy.region.players", "Copied %s player(s) of group '%s' from %s to %s", playerEntriesToCopy.size(), group, ChatLinkBuilder.buildRegionInfoLink(srcRegion), ChatLinkBuilder.buildRegionInfoLink(targetRegion)));
            return 0;
        }
        sendCmdFeedback(ctx.getSource(), class_2561.method_48322("cli.msg.info.region.group.player.empty", "No players defined as '%s' in %s", group, ChatLinkBuilder.buildRegionInfoLink(srcRegion)));
        return 1;
    }

    public static int copyRegionPlayers(CommandContext<class_2168> ctx, IProtectedRegion region, IProtectedRegion srcRegion) {
        return copyRegionPlayers(ctx, region, srcRegion, CommandConstants.MEMBER.toString())
                + copyRegionPlayers(ctx, region, srcRegion, CommandConstants.OWNER.toString());
    }

    public static int clearFlags(CommandContext<class_2168> ctx, IProtectedRegion region) {
        int amount = region.getFlags().size();
        if (amount == 0) {
            class_5250 feedbackMsg = class_2561.method_48322("cli.msg.info.region.flag.empty", "No flags defined in %s", ChatLinkBuilder.buildRegionInfoLink(region));
            sendCmdFeedback(ctx.getSource(), feedbackMsg);
            return 1;
        }
        Collection<IFlag> flags = region.getFlags().flags(); // TODO: Undo action link
        region.getFlags().clear();
        class_5250 feedbackMsg = class_2561.method_48322("cli.msg.info.region.flag.cleared", "Removed %s flag(s) from %s", amount, ChatLinkBuilder.buildRegionInfoLink(region));
        sendCmdFeedback(ctx.getSource(), feedbackMsg);
        RegionManager.get().save();
        return 0;
    }

    public static int clearPlayers(CommandContext<class_2168> ctx, IProtectedRegion region) {
        return clearPlayers(ctx, region, Permissions.MEMBER) + clearPlayers(ctx, region, Permissions.OWNER);
    }

    public static int clearPlayers(CommandContext<class_2168> ctx, IProtectedRegion region, String groupName) {
        int amount = region.getGroup(groupName).getPlayers().size();
        if (amount == 0) {
            class_5250 feedbackMsg = class_2561.method_48322("cli.msg.info.region.players.empty", "No players (group '%s') present in %s", ChatLinkBuilder.buildRegionInfoLink(region), groupName);
            sendCmdFeedback(ctx.getSource(), feedbackMsg);
            return 1;
        }
        region.getGroup(groupName).clearPlayers();
        class_5250 feedbackMsg = class_2561.method_48322("cli.msg.info.region.players.cleared", "Cleared %s players of group '%s' for %s\",", ChatLinkBuilder.buildRegionInfoLink(region), amount, groupName);
        sendCmdFeedback(ctx.getSource(), feedbackMsg);
        RegionManager.get().save();
        return 0;
    }

    public static int clearTeams(CommandContext<class_2168> ctx, IProtectedRegion region) {
        return clearPlayers(ctx, region, Permissions.MEMBER) + clearPlayers(ctx, region, Permissions.OWNER);
    }

    public static int clearTeams(CommandContext<class_2168> ctx, IProtectedRegion region, String groupName) {
        int amount = region.getGroup(groupName).getTeams().size();
        if (amount == 0) {
            class_5250 feedbackMsg = class_2561.method_48322("cli.msg.info.region.teams.empty", "No teams (group '%s') present in %s", ChatLinkBuilder.buildRegionInfoLink(region), groupName);
            sendCmdFeedback(ctx.getSource(), feedbackMsg);
            return 1;
        }
        region.getGroup(groupName).clearTeams();
        class_5250 feedbackMsg = class_2561.method_48322("cli.msg.info.region.teams.cleared", "Cleared %s teams of group '%s' for %s", ChatLinkBuilder.buildRegionInfoLink(region), amount, groupName);
        sendCmdFeedback(ctx.getSource(), feedbackMsg);
        RegionManager.get().save();
        return 0;
    }


    public static int clearGroups(CommandContext<class_2168> ctx, IProtectedRegion region, String groupName) {
        return CommandUtil.clearTeams(ctx, region, groupName) + CommandUtil.clearPlayers(ctx, region, groupName);
    }

    public static int addAllFlags(CommandContext<class_2168> ctx, IProtectedRegion region) {
        return addFlags(ctx, region, RegionFlag.getFlags());
    }

    public static int addFlag(CommandContext<class_2168> ctx, IProtectedRegion region, RegionFlag flag) {
        return addFlag(ctx, region, flag, FlagState.DENIED, false);
    }

    public static int addFlag(CommandContext<class_2168> ctx, IProtectedRegion region, RegionFlag flag, FlagState state, boolean override) {
        return addRegionFlag(ctx, region, flag, state, override);
    }

    public static int addFlags(CommandContext<class_2168> ctx, IProtectedRegion region, Set<RegionFlag> flags) {
        flags.forEach(flag -> CommandUtil.addRegionFlag(ctx, region, flag));
        return 0;
    }

    public static int addRegionFlag(CommandContext<class_2168> ctx, IProtectedRegion region, RegionFlag flag, FlagState state, boolean override) {
        if (region.getRegionType() == RegionType.LOCAL && flag == RegionFlag.ENTER_DIM) {
            class_5250 msg = class_2561.method_43470("Flag 'enter-dim' is currently not supported for local regions.");
            sendCmdFeedback(ctx.getSource(), msg);
            return 1;
        }
        if (!region.containsFlag(flag)) {
            IFlag iFlag;
            switch (flag.type) {
                case BOOLEAN_FLAG:
                    iFlag = new BooleanFlag(flag, state, override);
                    break;
                case LIST_FLAG:
                case INT_FLAG:
                    throw new NotImplementedException("Not yet implemented = " + flag.name);
                default:
                    throw new IllegalArgumentException("Unexpected value = " + flag.getClass().getName());
            }

            FlagEvent.AddFlagEvent addFlagEvent = new FlagEvent.AddFlagEvent(ctx.getSource(), region, iFlag);
            Services.EVENT.post(addFlagEvent);
            region.addFlag(iFlag);
            RegionManager.get().save();
            class_5250 flagLink = ChatLinkBuilder.buildFlagInfoLink(region, iFlag);
            class_5250 msg = class_2561.method_48322("cli.msg.flag.added", "Added flag '%s' to %s",
                    flagLink, ChatLinkBuilder.buildRegionInfoLink(region));
            class_5250 undoLink = ChatLinkBuilder.buildRegionActionUndoLink(ctx.getInput(), ADD, REMOVE);
            sendCmdFeedback(ctx.getSource(), Messages.substitutable("%s %s", msg, undoLink));
            return 0;
        } else {
            class_5250 msg = class_2561.method_48322("cli.msg.flag.present", "Flag '%s' is already present in %s", flag.name,
                    ChatLinkBuilder.buildRegionInfoLink(region));
            sendCmdFeedback(ctx.getSource(), msg);
            return 1;
        }
    }

    public static int addRegionFlag(CommandContext<class_2168> ctx, IProtectedRegion region, RegionFlag flag) {
        return addRegionFlag(ctx, region, flag, FlagState.DENIED, false);
    }
}
