/*
 * Decompiled with CFR 0.152.
 */
package animalhunger.animalhunger.storage;

import animalhunger.animalhunger.AnimalData;
import animalhunger.animalhunger.AnimalHunger;
import java.io.File;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.ChatColor;

public class DatabaseManager {
    private final AnimalHunger plugin;
    private Connection connection;
    private final StorageType storageType;
    private final String tableName = "animal_data";
    private final Map<Thread, Connection> connectionPool = new ConcurrentHashMap<Thread, Connection>();
    private final String host;
    private final int port;
    private final String database;
    private final String username;
    private final String password;
    private final boolean useSSL;

    public DatabaseManager(AnimalHunger plugin) {
        this.plugin = plugin;
        String typeStr = plugin.getConfig().getString("storage.type", "sqlite").toUpperCase();
        try {
            this.storageType = StorageType.valueOf(typeStr);
        }
        catch (IllegalArgumentException e) {
            throw new RuntimeException("Invalid storage type");
        }
        this.host = plugin.getConfig().getString("storage.mysql.host", "localhost");
        this.port = plugin.getConfig().getInt("storage.mysql.port", 3306);
        this.database = plugin.getConfig().getString("storage.mysql.database", "animalhunger");
        this.username = plugin.getConfig().getString("storage.mysql.username", "root");
        this.password = plugin.getConfig().getString("storage.mysql.password", "password");
        this.useSSL = plugin.getConfig().getBoolean("storage.mysql.useSSL", false);
    }

    public boolean initialize() {
        try {
            if (this.storageType == StorageType.SQLITE) {
                return this.initializeSQLite();
            }
            return this.initializeMySQL();
        }
        catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    private boolean initializeSQLite() throws SQLException {
        String path = this.plugin.getDataFolder().getAbsolutePath() + "/data.db";
        String url = "jdbc:sqlite:" + path;
        this.connection = DriverManager.getConnection(url);
        this.createTables();
        this.checkAndAddMissingColumns();
        return true;
    }

    private boolean initializeMySQL() throws SQLException {
        String url = String.format("jdbc:mariadb://%s:%d/%s?useSSL=%s&autoReconnect=true&useUnicode=true&characterEncoding=UTF-8", this.host, this.port, this.database, this.useSSL);
        this.connection = DriverManager.getConnection(url, this.username, this.password);
        this.createTables();
        this.checkAndAddMissingColumns();
        return true;
    }

    private void createTables() throws SQLException {
        String sql = this.storageType == StorageType.SQLITE ? "CREATE TABLE IF NOT EXISTS animal_data (uuid TEXT PRIMARY KEY NOT NULL,hunger INTEGER NOT NULL,max_hunger INTEGER NOT NULL,level INTEGER NOT NULL,feed_count INTEGER NOT NULL,bond_level INTEGER NOT NULL,resource_count INTEGER NOT NULL,custom_name TEXT,name_color TEXT,animal_id INTEGER,home_center_world TEXT,home_center_x REAL,home_center_y REAL,home_center_z REAL,home_radius INTEGER,last_update TIMESTAMP DEFAULT CURRENT_TIMESTAMP)" : "CREATE TABLE IF NOT EXISTS animal_data (uuid VARCHAR(36) PRIMARY KEY NOT NULL,hunger INT NOT NULL,max_hunger INT NOT NULL,level INT NOT NULL,feed_count INT NOT NULL,bond_level INT NOT NULL,resource_count INT NOT NULL,custom_name VARCHAR(32),name_color VARCHAR(16),animal_id INT,home_center_world VARCHAR(255),home_center_x DOUBLE,home_center_y DOUBLE,home_center_z DOUBLE,home_radius INT,last_update TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,INDEX idx_animal_id (animal_id),INDEX idx_last_update (last_update)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        try (Statement stmt = this.getConnection().createStatement();){
            stmt.execute(sql);
        }
        String metaSql = this.storageType == StorageType.SQLITE ? "CREATE TABLE IF NOT EXISTS meta_data (key TEXT PRIMARY KEY NOT NULL,value TEXT NOT NULL)" : "CREATE TABLE IF NOT EXISTS meta_data (key VARCHAR(32) PRIMARY KEY NOT NULL,value TEXT NOT NULL) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        try (Statement stmt = this.getConnection().createStatement();){
            stmt.execute(metaSql);
        }
    }

    private void checkAndAddMissingColumns() {
        try (Statement stmt = this.getConnection().createStatement();){
            DatabaseMetaData md = this.getConnection().getMetaData();
            ResultSet rs = md.getColumns(null, null, "animal_data", "home_center_world");
            if (!rs.next()) {
                if (this.storageType == StorageType.SQLITE) {
                    stmt.executeUpdate("ALTER TABLE animal_data ADD COLUMN home_center_world TEXT");
                    stmt.executeUpdate("ALTER TABLE animal_data ADD COLUMN home_center_x REAL");
                    stmt.executeUpdate("ALTER TABLE animal_data ADD COLUMN home_center_y REAL");
                    stmt.executeUpdate("ALTER TABLE animal_data ADD COLUMN home_center_z REAL");
                    stmt.executeUpdate("ALTER TABLE animal_data ADD COLUMN home_radius INTEGER");
                } else {
                    stmt.executeUpdate("ALTER TABLE animal_data ADD COLUMN home_center_world VARCHAR(255)");
                    stmt.executeUpdate("ALTER TABLE animal_data ADD COLUMN home_center_x DOUBLE");
                    stmt.executeUpdate("ALTER TABLE animal_data ADD COLUMN home_center_y DOUBLE");
                    stmt.executeUpdate("ALTER TABLE animal_data ADD COLUMN home_center_z DOUBLE");
                    stmt.executeUpdate("ALTER TABLE animal_data ADD COLUMN home_radius INT");
                }
            }
            rs.close();
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
    }

    public CompletableFuture<Boolean> saveAnimalDataAsync(UUID uuid, AnimalData data, int animalId) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                return this.saveAnimalData(uuid, data, animalId);
            }
            catch (SQLException e) {
                return false;
            }
        });
    }

    public boolean saveAnimalData(UUID uuid, AnimalData data, int animalId) throws SQLException {
        String sql;
        if (this.storageType == StorageType.SQLITE) {
            String checkSql = "SELECT 1 FROM animal_data WHERE uuid = ?";
            boolean exists = false;
            try (PreparedStatement ps = this.getConnection().prepareStatement(checkSql);){
                ps.setString(1, uuid.toString());
                try (ResultSet rs = ps.executeQuery();){
                    if (rs.next()) {
                        exists = true;
                    }
                }
            }
            sql = exists ? "UPDATE animal_data SET hunger=?, max_hunger=?, level=?, feed_count=?, bond_level=?, resource_count=?, custom_name=?, name_color=?, animal_id=? WHERE uuid=?" : "INSERT INTO animal_data (hunger, max_hunger, level, feed_count, bond_level, resource_count, custom_name, name_color, animal_id, uuid) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        } else {
            sql = "INSERT INTO animal_data (uuid, hunger, max_hunger, level, feed_count, bond_level, resource_count, custom_name, name_color, animal_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE hunger=VALUES(hunger), max_hunger=VALUES(max_hunger), level=VALUES(level), feed_count=VALUES(feed_count), bond_level=VALUES(bond_level), resource_count=VALUES(resource_count), custom_name=VALUES(custom_name), name_color=VALUES(name_color), animal_id=VALUES(animal_id)";
        }
        try (PreparedStatement ps = this.getConnection().prepareStatement(sql);){
            ps.setInt(1, data.getHunger());
            ps.setInt(2, data.getMaxHunger());
            ps.setInt(3, data.getLevel());
            ps.setInt(4, data.getFeedCount());
            ps.setInt(5, data.getBondLevel());
            ps.setInt(6, data.getResourceCount());
            ps.setString(7, data.getCustomName());
            ps.setString(8, data.getNameColor().name());
            if (animalId >= 0) {
                ps.setInt(9, animalId);
            } else {
                ps.setNull(9, 4);
            }
            ps.setString(10, uuid.toString());
            ps.executeUpdate();
            boolean bl = true;
            return bl;
        }
    }

    public CompletableFuture<AnimalData> loadAnimalDataAsync(UUID uuid) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                return this.loadAnimalData(uuid);
            }
            catch (SQLException e) {
                return null;
            }
        });
    }

    public AnimalData loadAnimalData(UUID uuid) throws SQLException {
        block16: {
            String sql = "SELECT * FROM animal_data WHERE uuid = ?";
            try (PreparedStatement ps = this.getConnection().prepareStatement(sql);){
                ps.setString(1, uuid.toString());
                try (ResultSet rs = ps.executeQuery();){
                    if (!rs.next()) break block16;
                    AnimalData data = new AnimalData(rs.getInt("hunger"), rs.getInt("max_hunger"), rs.getInt("level"), rs.getInt("feed_count"), rs.getInt("bond_level"), rs.getInt("resource_count"));
                    data.setCustomName(rs.getString("custom_name"));
                    String colorName = rs.getString("name_color");
                    if (colorName != null) {
                        try {
                            data.setNameColor(ChatColor.valueOf((String)colorName));
                        }
                        catch (IllegalArgumentException e) {
                            data.setNameColor(ChatColor.YELLOW);
                        }
                    }
                    AnimalData animalData = data;
                    return animalData;
                }
            }
        }
        return null;
    }

    public Map<UUID, AnimalData> loadAllAnimalData() throws SQLException {
        HashMap<UUID, AnimalData> dataMap = new HashMap<UUID, AnimalData>();
        String sql = "SELECT * FROM animal_data";
        try (Statement stmt = this.getConnection().createStatement();
             ResultSet rs = stmt.executeQuery(sql);){
            while (rs.next()) {
                UUID uuid = UUID.fromString(rs.getString("uuid"));
                AnimalData data = new AnimalData(rs.getInt("hunger"), rs.getInt("max_hunger"), rs.getInt("level"), rs.getInt("feed_count"), rs.getInt("bond_level"), rs.getInt("resource_count"));
                data.setCustomName(rs.getString("custom_name"));
                String colorName = rs.getString("name_color");
                if (colorName != null) {
                    try {
                        data.setNameColor(ChatColor.valueOf((String)colorName));
                    }
                    catch (IllegalArgumentException e) {
                        data.setNameColor(ChatColor.YELLOW);
                    }
                }
                dataMap.put(uuid, data);
            }
        }
        return dataMap;
    }

    public Map<Integer, UUID> loadAnimalIdMap() throws SQLException {
        HashMap<Integer, UUID> idMap = new HashMap<Integer, UUID>();
        String sql = "SELECT uuid, animal_id FROM animal_data WHERE animal_id IS NOT NULL";
        try (Statement stmt = this.getConnection().createStatement();
             ResultSet rs = stmt.executeQuery(sql);){
            while (rs.next()) {
                int animalId = rs.getInt("animal_id");
                UUID uuid = UUID.fromString(rs.getString("uuid"));
                idMap.put(animalId, uuid);
            }
        }
        return idMap;
    }

    public CompletableFuture<Boolean> deleteAnimalDataAsync(UUID uuid) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                return this.deleteAnimalData(uuid);
            }
            catch (SQLException e) {
                return false;
            }
        });
    }

    public boolean deleteAnimalData(UUID uuid) throws SQLException {
        String sql = "DELETE FROM animal_data WHERE uuid = ?";
        try (PreparedStatement ps = this.getConnection().prepareStatement(sql);){
            ps.setString(1, uuid.toString());
            ps.executeUpdate();
            boolean bl = true;
            return bl;
        }
    }

    public void saveMetaData(String key, String value) throws SQLException {
        String sql = this.storageType == StorageType.SQLITE ? "INSERT OR REPLACE INTO meta_data (key, value) VALUES (?, ?)" : "INSERT INTO meta_data (key, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value=VALUES(value)";
        try (PreparedStatement ps = this.getConnection().prepareStatement(sql);){
            ps.setString(1, key);
            ps.setString(2, value);
            ps.executeUpdate();
        }
    }

    public String loadMetaData(String key) throws SQLException {
        String sql = "SELECT value FROM meta_data WHERE key = ?";
        try (PreparedStatement ps = this.getConnection().prepareStatement(sql);){
            ps.setString(1, key);
            try (ResultSet rs = ps.executeQuery();){
                if (rs.next()) {
                    String string = rs.getString("value");
                    return string;
                }
            }
        }
        return null;
    }

    public Connection getConnection() throws SQLException {
        if (this.storageType == StorageType.SQLITE) {
            if (this.connection == null || this.connection.isClosed()) {
                this.initializeSQLite();
            }
            return this.connection;
        }
        Thread currentThread = Thread.currentThread();
        Connection conn = this.connectionPool.get(currentThread);
        if (conn == null || conn.isClosed()) {
            conn = DriverManager.getConnection(String.format("jdbc:mariadb://%s:%d/%s?useSSL=%s&autoReconnect=true", this.host, this.port, this.database, this.useSSL), this.username, this.password);
            this.connectionPool.put(currentThread, conn);
        }
        return conn;
    }

    public void cleanupOldRecords(int daysOld) throws SQLException {
        String sql = this.storageType == StorageType.SQLITE ? "DELETE FROM animal_data WHERE last_update < datetime('now', '-' || ? || ' days')" : "DELETE FROM animal_data WHERE last_update < DATE_SUB(NOW(), INTERVAL ? DAY)";
        try (PreparedStatement ps = this.getConnection().prepareStatement(sql);){
            ps.setInt(1, daysOld);
            int deleted = ps.executeUpdate();
            if (deleted > 0) {
                // empty if block
            }
        }
    }

    public void close() {
        try {
            if (this.connection != null && !this.connection.isClosed()) {
                this.connection.close();
            }
            for (Connection conn : this.connectionPool.values()) {
                if (conn == null || conn.isClosed()) continue;
                conn.close();
            }
            this.connectionPool.clear();
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
    }

    public DatabaseStats getStats() throws SQLException {
        DatabaseStats stats = new DatabaseStats();
        String countSql = "SELECT COUNT(*) as total FROM animal_data";
        try (Statement stmt = this.getConnection().createStatement();
             ResultSet rs = stmt.executeQuery(countSql);){
            if (rs.next()) {
                stats.totalRecords = rs.getInt("total");
            }
        }
        if (this.storageType == StorageType.SQLITE) {
            String path = this.plugin.getDataFolder().getAbsolutePath() + "/data.db";
            File dbFile = new File(path);
            stats.databaseSize = dbFile.length();
        } else {
            String sizeSql = "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) as size_mb FROM information_schema.tables WHERE table_schema = ? AND table_name = ?";
            try (PreparedStatement ps = this.getConnection().prepareStatement(sizeSql);){
                ps.setString(1, this.database);
                ps.setString(2, "animal_data");
                try (ResultSet rs = ps.executeQuery();){
                    if (rs.next()) {
                        stats.databaseSize = (long)(rs.getDouble("size_mb") * 1024.0 * 1024.0);
                    }
                }
            }
        }
        stats.storageType = this.storageType;
        return stats;
    }

    public static enum StorageType {
        SQLITE,
        MYSQL;

    }

    public static class DatabaseStats {
        public int totalRecords;
        public long databaseSize;
        public StorageType storageType;

        public String getFormattedSize() {
            if (this.databaseSize < 1024L) {
                return this.databaseSize + " B";
            }
            if (this.databaseSize < 0x100000L) {
                return String.format("%.2f KB", (double)this.databaseSize / 1024.0);
            }
            return String.format("%.2f MB", (double)this.databaseSize / 1024.0 / 1024.0);
        }
    }
}

