package dev.mattidragon.jsonpatcher.metapatch;

import com.google.gson.JsonObject;
import com.mojang.datafixers.util.Either;
import com.mojang.datafixers.util.Pair;
import dev.mattidragon.jsonpatcher.lang.runtime.EvaluationContext;
import dev.mattidragon.jsonpatcher.lang.runtime.lib.builder.DontBind;
import dev.mattidragon.jsonpatcher.lang.runtime.value.Value;
import dev.mattidragon.jsonpatcher.misc.GsonConverter;
import dev.mattidragon.jsonpatcher.misc.ValueOps;
import dev.mattidragon.jsonpatcher.patch.PatchTarget;
import dev.mattidragon.jsonpatcher.patch.PatchingContext;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.util.*;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;

@SuppressWarnings("unused")
public class MetapatchLibrary {
    @DontBind
    private final Map<class_2960, JsonObject> addedFiles = new HashMap<>();
    @DontBind
    private final List<FileFilter> filters = new ArrayList<>();
    @DontBind
    private final class_3300 resourceManager;

    public MetapatchLibrary(class_3300 resourceManager) {
        this.resourceManager = resourceManager;
    }

    @DontBind
    public void apply(MetapatchResourcePack metaPack) {
        metaPack.set(addedFiles, filters);
    }

    @DontBind
    public void clear() {
        addedFiles.clear();
        filters.clear();
    }

    @DontBind
    private boolean isDeleted(class_2960 id) {
        // The last filter added will get priority
        for (var filter : filters.reversed()) {
            if (filter.target().test(id)) {
                return !filter.allow();
            }
        }
        return false;
    }

    @DontBind
    private static Value.ObjectValue valueFromResource(class_3298 resource) throws IOException {
        return GsonConverter.fromGson(MetapatchResourcePack.GSON.fromJson(new InputStreamReader(resource.method_14482()), JsonObject.class));
    }

    public void addFile(EvaluationContext context, Value.StringValue idString, Value.ObjectValue file) {
        var id = class_2960.method_60654(idString.value());

        // Add filter to undo deletion if necessary
        if (isDeleted(id)) {
            filters.add(new FileFilter(
                    new PatchTarget(
                            Optional.of(id.method_12836()),
                            Optional.of(new PatchTarget.Path(Either.left(id.method_12832()))),
                            Optional.empty()),
                    true));
        }
        addedFiles.put(id, GsonConverter.toGson(file));
    }

    public void deleteFile(EvaluationContext context, Value.StringValue idString) {
        var id = class_2960.method_60654(idString.value());

        filters.add(new FileFilter(
                new PatchTarget(
                        Optional.of(id.method_12836()),
                        Optional.of(new PatchTarget.Path(Either.left(id.method_12832()))),
                        Optional.empty()),
                false));
    }

    public void deleteFiles(EvaluationContext context, Value targetValue) {
        var target = PatchTarget.CODEC.decode(ValueOps.INSTANCE, targetValue)
                .getOrThrow(error -> new IllegalStateException("Failed to parse target: " + error))
                .getFirst();
        filters.add(new FileFilter(target, false));
    }

    public Value getFile(EvaluationContext context, Value.StringValue idString) {
        var id = class_2960.method_60654(idString.value());

        try (var __ = PatchingContext.disablePatching()) {
            var resource = resourceManager.method_14486(id);
            if (resource.isPresent()) {
                return valueFromResource(resource.get());
            }
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }

        return Value.NullValue.NULL;
    }

    public Value getFiles(EvaluationContext context, Value.StringValue idString) {
        var id = class_2960.method_60654(idString.value());

        var array = new Value.ArrayValue();
        try (var __ = PatchingContext.disablePatching()) {
            var resources = resourceManager.method_14489(id);
            for (var resource : resources) {
                array.value().add(valueFromResource(resource));
            }
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }

        return array;
    }

    public Value searchFiles(EvaluationContext context, Value targetValue) {
        var target = PatchTarget.CODEC.decode(ValueOps.INSTANCE, targetValue)
                .getOrThrow(error -> new IllegalStateException("Failed to parse target: " + error))
                .getFirst();

        var startingPath = "";
        if (target.path().isPresent()) {
            startingPath = target.path()
                    .get()
                    .path()
                    .map(p -> p, Pair::getFirst);
        }
        // Remove last path segment, as minecraft treats it differently
        var slashIndex = startingPath.lastIndexOf('/');
        if (slashIndex != -1) {
            startingPath = startingPath.substring(0, slashIndex);
        }

        var out = new Value.ObjectValue();
        try (var __ = PatchingContext.disablePatching()) {
            var found = resourceManager.method_14488(startingPath, target);
            for (var entry : found.entrySet()) {
                var id = entry.getKey();
                var resource = entry.getValue();
                out.value().put(id.toString(), valueFromResource(resource));
            }
        } catch (Exception e) {
            throw new UncheckedIOException(new IOException("Failed to search files", e));
        }

        return out;
    }
}
