package dev.mattidragon.jsonpatcher.patch;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.stream.JsonWriter;
import dev.mattidragon.jsonpatcher.lang.runtime.value.Value;
import dev.mattidragon.jsonpatcher.JsonPatcher;
import dev.mattidragon.jsonpatcher.config.Config;
import dev.mattidragon.jsonpatcher.misc.DumpManager;
import dev.mattidragon.jsonpatcher.misc.GsonConverter;
import dev.mattidragon.jsonpatcher.misc.MetaPatchPackAccess;
import org.apache.commons.lang3.mutable.MutableObject;

import java.io.*;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.concurrent.*;
import java.util.function.Consumer;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3264;
import net.minecraft.class_3300;
import net.minecraft.class_3518;
import net.minecraft.class_7367;

public class Patcher {
    public static final ExecutorService PATCH_RUNNER = Executors.newThreadPerTaskExecutor(Thread.ofVirtual().name("JsonPatcher-Patch-Runner").factory());
    private static final Gson GSON = new Gson();
    private final class_3264 resourceType;
    private final PatchStorage patches;

    public Patcher(class_3264 resourceType, PatchStorage patches) {
        this.resourceType = resourceType;
        this.patches = patches;
    }

    public boolean hasPatches(class_2960 id) {
        return patches.hasPatches(id);
    }

    private JsonElement applyPatches(JsonElement json, class_2960 id) {
        var errors = new ArrayList<Exception>();
        var activeJson = new MutableObject<>(class_3518.method_15295(json, "patched file"));
        try {
            for (var patch : patches.getPatches(id)) {
                var root = GsonConverter.fromGson(activeJson.getValue());
                var timeBeforePatch = System.nanoTime();
                var success = runPatch(patch, PATCH_RUNNER, errors::add, root);
                var timeAfterPatch = System.nanoTime();
                JsonPatcher.RELOAD_LOGGER.debug("Patched {} with {} in {}ms", id, patch.id(), (timeAfterPatch - timeBeforePatch) / 1e6);
                if (success) {
                    activeJson.setValue(GsonConverter.toGson(root));
                }
            }
        } catch (RuntimeException e) {
            errors.add(e);
        }
        if (!errors.isEmpty()) {
            errors.forEach(error -> JsonPatcher.RELOAD_LOGGER.error("Error while patching {}", id, error));
            var message = "Encountered %s error(s) while patching %s. See jsonpatcher/jsonpatcher.log for details".formatted(errors.size(), id);
            ErrorLogger.CURRENT.get().accept(class_2561.method_43470(message).method_27692(class_124.field_1061));
            if (Config.MANAGER.get().throwOnFailure()) {
                throw new PatchingException(message);
            } else {
                JsonPatcher.MAIN_LOGGER.error(message);
            }
        }
        return activeJson.getValue();
    }

    /**
     * Runs a patch with proper error handling.
     * @param patch The patch to run.
     * @param executor An executor to run the patch on, required to run on another thread for timeout to work
     * @param errorConsumer A consumer the receives errors from the patch.
     * @param root The root object for the patch context, will be modified
     * @return {@code true} if the patch completed successfully. If {@code false} the {@code errorConsumer} should have received an error.
     */
    public static boolean runPatch(BasePatch patch, Executor executor, Consumer<RuntimeException> errorConsumer, Value.ObjectValue root) {
        try {
            CompletableFuture.runAsync(() -> patch.program().run(root), executor)
                    .get(Config.MANAGER.get().patchTimeoutMillis(), TimeUnit.MILLISECONDS);
            return true;
        } catch (ExecutionException e) {
            if (e.getCause() instanceof RuntimeException cause) {
                errorConsumer.accept(cause);
            } else if (e.getCause() instanceof StackOverflowError cause) {
                errorConsumer.accept(new PatchingException("Stack overflow while applying patch %s".formatted(patch.name()), cause));
            } else {
                errorConsumer.accept(new RuntimeException("Unexpected error while applying patch %s".formatted(patch.name()), e));
            }
        } catch (InterruptedException e) {
            errorConsumer.accept(new PatchingException("Async error while applying patch %s".formatted(patch.name()), e));
        } catch (TimeoutException e) {
            errorConsumer.accept(new PatchingException("Timeout while applying patch %s. Check for infinite loops and increase the timeout in the config.".formatted(patch.name()), e));
        }
        return false;
    }

/*    private static EvaluationContext buildContext(Identifier patchId, EvaluationContext.LibraryLocator libraryLocator, Value.ObjectValue root, Settings settings) {
        var builder = EvaluationContext.builder(ConfigProvider.INSTANCE);
        builder.root(root);
        builder.libraryLocator(libraryLocator);
        builder.debugConsumer(value -> JsonPatcher.RELOAD_LOGGER.info("Debug from {}: {}", patchId, value));
        builder.variable("_isLibrary", settings.isLibrary());
        builder.variable("_target", settings.targetAsValue());
        builder.variable("_isMetapatch", settings.isMetaPatch());
        if (settings.isMetaPatch()) {
            builder.variable("metapatch", new LibraryBuilder(MetapatchLibrary.class, settings.metaPatchLibrary).build());
        }
        return builder.build();
    }*/

    public class_7367<InputStream> patchInputStream(class_2960 id, class_7367<InputStream> stream) {
        if (!hasPatches(id)) return stream;

        try {
            JsonPatcher.RELOAD_LOGGER.debug("Patching {}", id);
            var json = GSON.fromJson(new InputStreamReader(stream.get()), JsonElement.class);
            json = applyPatches(json, id);

            var out = new ByteArrayOutputStream();
            var writer = new OutputStreamWriter(out);
            GSON.toJson(json, new JsonWriter(writer));
            writer.close();

            DumpManager.dumpIfEnabled(id, resourceType, json);
            return () -> new ByteArrayInputStream(out.toByteArray());
        } catch (JsonParseException | IOException e) {
            if (Config.MANAGER.get().throwOnFailure()) {
                throw new RuntimeException("Failed to patch json at %s".formatted(id), e);
            } else {
                JsonPatcher.RELOAD_LOGGER.error("Failed to patch json at {}", id, e);
            }
            return stream;
        }
    }

    public void runMetaPatches(class_3300 manager, Executor executor) {
        if (!(manager instanceof MetaPatchPackAccess packAccess)) {
            JsonPatcher.MAIN_LOGGER.error("Failed to run meta patches: resource manager doesn't expose meta pack");
            return;
        }

        var metaPack = packAccess.jsonpatcher$getMetaPatchPack();
        metaPack.clear();
        patches.metapatchLibrary().clear();

        var metaPatches = new ArrayList<>(patches.getMetaPatches());
        metaPatches.sort(Comparator.comparing(Patch::priority));
        var errors = new ArrayList<RuntimeException>();

        try {
            for (var patch : metaPatches) {
                var timeBeforePatch = System.nanoTime();
                runPatch(patch, executor, errors::add, new Value.ObjectValue());
                var timeAfterPatch = System.nanoTime();
                JsonPatcher.RELOAD_LOGGER.debug("Ran meta patch {} in {}ms", patch.id(), (timeAfterPatch - timeBeforePatch) / 1e6);
            }
        } catch (RuntimeException e) {
            errors.add(e);
        }

        if (!errors.isEmpty()) {
            errors.forEach(error -> JsonPatcher.RELOAD_LOGGER.error("Error while running meta patch", error));
            var message = "Encountered %s error(s) while running meta patches. See jsonpatcher/jsonpatcher.log for details".formatted(errors.size());

            ErrorLogger.CURRENT.get().accept(class_2561.method_43470(message).method_27692(class_124.field_1061));
            if (Config.MANAGER.get().throwOnFailure()) {
                throw new PatchingException(message);
            } else {
                JsonPatcher.MAIN_LOGGER.error(message);
            }
        }
        patches.metapatchLibrary().apply(metaPack);
    }
}
