/*
 * Decompiled with CFR 0.152.
 */
package as.sirhephaistos.simplybetter.core.config;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonParseException;
import com.google.gson.annotations.SerializedName;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Locale;
import java.util.Objects;
import net.fabricmc.loader.api.FabricLoader;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class ConfigManager {
    private static final Logger LOGGER = LogManager.getLogger((String)"SimplyBetter-Core/Config");
    private static final String FILE_NAME = "sbcore-conf.json";
    private static final int MIN_THREADS = 2;
    private static final int MAX_THREADS = 16;
    private final Path configDir;
    private final Path filePath;
    private final Gson gson;
    private Config config;

    private ConfigManager(Path configDir, Gson gson) {
        this.configDir = Objects.requireNonNull(configDir);
        this.filePath = configDir.resolve(FILE_NAME);
        this.gson = Objects.requireNonNull(gson);
    }

    public static ConfigManager createDefault() {
        Path base = FabricLoader.getInstance().getConfigDir().resolve("simplybetter");
        Gson gson = new GsonBuilder().setPrettyPrinting().create();
        return new ConfigManager(base, gson);
    }

    public void loadOrCreate() {
        try {
            Files.createDirectories(this.configDir, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IllegalStateException("Cannot create config directory: " + String.valueOf(this.configDir), e);
        }
        boolean needRewrite = false;
        if (Files.notExists(this.filePath, new LinkOption[0])) {
            this.config = Config.defaults();
            this.writeConfig(this.config);
            LOGGER.info("Created default config at {}", (Object)this.filePath);
            return;
        }
        try (BufferedReader reader = Files.newBufferedReader(this.filePath, StandardCharsets.UTF_8);){
            Config loaded = (Config)this.gson.fromJson((Reader)reader, Config.class);
            if (loaded == null) {
                loaded = Config.defaults();
            }
            NormalizationResult result = this.normalize(loaded);
            this.config = result.normalized;
            needRewrite = result.rewrite;
        }
        catch (JsonParseException | IOException ex) {
            LOGGER.warn("Failed to read/parse config, recreating defaults: {}", (Object)ex.toString());
            this.config = Config.defaults();
            needRewrite = true;
        }
        if (needRewrite) {
            this.writeConfig(this.config);
            LOGGER.info("Rewrote config with normalized values at {}", (Object)this.filePath);
        }
    }

    private void writeConfig(Config cfg) {
        try (BufferedWriter writer = Files.newBufferedWriter(this.filePath, StandardCharsets.UTF_8, new OpenOption[0]);){
            this.gson.toJson((Object)cfg, (Appendable)writer);
        }
        catch (IOException e) {
            throw new IllegalStateException("Cannot write config file: " + String.valueOf(this.filePath), e);
        }
    }

    public int effectiveThreadCount(int detectedLogicalProcessors) {
        int logical = Math.max(1, detectedLogicalProcessors);
        Integer cfg = this.config.threadCount;
        int value = cfg == null ? Math.max(2, logical / 2) : cfg;
        if (value < 2) {
            value = 2;
        }
        if (value > 16) {
            value = 16;
        }
        return value;
    }

    public String journalMode() {
        return this.config.journalMode;
    }

    public String synchronous() {
        return this.config.synchronous;
    }

    public boolean foreignKeys() {
        return this.config.foreignKeys;
    }

    public Path filePath() {
        return this.filePath;
    }

    public void logSummary(int detectedLogicalProcessors) {
        int pool = this.effectiveThreadCount(detectedLogicalProcessors);
        LOGGER.info("Config: threads={}, logicalCPUs={}, journalMode={}, synchronous={}, foreignKeys={}", (Object)pool, (Object)detectedLogicalProcessors, (Object)this.journalMode(), (Object)this.synchronous(), (Object)this.foreignKeys());
    }

    private NormalizationResult normalize(Config in) {
        boolean rewrite = false;
        Config out = new Config();
        if (in.threadCount != null && in.threadCount <= 0) {
            out.threadCount = null;
            rewrite = true;
        } else {
            out.threadCount = in.threadCount;
        }
        String jm = ConfigManager.nonEmptyUpper(in.journalMode);
        if (!JournalMode.isValid(jm)) {
            out.journalMode = Config.DEFAULT_JOURNAL_MODE;
            rewrite = true;
        } else {
            out.journalMode = jm;
        }
        String sync = ConfigManager.nonEmptyUpper(in.synchronous);
        if (!Synchronous.isValid(sync)) {
            out.synchronous = Config.DEFAULT_SYNCHRONOUS;
            rewrite = true;
        } else {
            out.synchronous = sync;
        }
        out.foreignKeys = in.foreignKeys;
        return new NormalizationResult(out, rewrite);
    }

    private static String nonEmptyUpper(String v) {
        if (v == null) {
            return null;
        }
        String t = v.trim();
        if (t.isEmpty()) {
            return null;
        }
        return t.toUpperCase(Locale.ROOT);
    }

    public static final class Config {
        static final String DEFAULT_JOURNAL_MODE = JournalMode.WAL.name();
        static final String DEFAULT_SYNCHRONOUS = Synchronous.NORMAL.name();
        @SerializedName(value="threadCount")
        public Integer threadCount = null;
        @SerializedName(value="journalMode")
        public String journalMode = DEFAULT_JOURNAL_MODE;
        @SerializedName(value="synchronous")
        public String synchronous = DEFAULT_SYNCHRONOUS;
        @SerializedName(value="foreignKeys")
        public boolean foreignKeys = true;

        public static Config defaults() {
            return new Config();
        }
    }

    private record NormalizationResult(Config normalized, boolean rewrite) {
    }

    public static enum JournalMode {
        WAL,
        DELETE,
        TRUNCATE,
        MEMORY,
        PERSIST,
        OFF;


        public static boolean isValid(String s) {
            if (s == null) {
                return false;
            }
            try {
                JournalMode.valueOf(s);
                return true;
            }
            catch (IllegalArgumentException ex) {
                return false;
            }
        }
    }

    public static enum Synchronous {
        OFF,
        NORMAL,
        FULL,
        EXTRA;


        public static boolean isValid(String s) {
            if (s == null) {
                return false;
            }
            try {
                Synchronous.valueOf(s);
                return true;
            }
            catch (IllegalArgumentException ex) {
                return false;
            }
        }
    }
}

