/*
 * Decompiled with CFR 0.152.
 */
package as.sirhephaistos.simplybetter.core.db;

import as.sirhephaistos.simplybetter.core.config.ConfigManager;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import net.fabricmc.loader.api.FabricLoader;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DatabaseManager {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"simplybetter-core-db");
    private static final String DB_FILE_NAME = "simplybetter.db";
    private static final String SCHEMA_RESOURCE = "simplybetter/schema.sql";
    private static final String SENTRY_TABLE = "sb_mails";
    private static final AtomicInteger THREAD_NUM = new AtomicInteger(0);
    private final Path configDir;
    private final Path dbPathAbs;
    private final ConfigManager configManager;
    private volatile boolean initialized = false;
    private ExecutorService executor;

    public DatabaseManager(Path configDir, Path absoluteDbPath, ConfigManager cfgManager) {
        this.configDir = Objects.requireNonNull(configDir, "configDir");
        this.dbPathAbs = Objects.requireNonNull(absoluteDbPath, "absoluteDbPath");
        this.configManager = Objects.requireNonNull(cfgManager, "configManager");
    }

    public static DatabaseManager createDefault() {
        Path baseConfig = FabricLoader.getInstance().getConfigDir();
        Path sbDir = baseConfig.resolve("simplybetter");
        ConfigManager cfg = ConfigManager.createDefault();
        cfg.loadOrCreate();
        Path newDbPath = sbDir.resolve(DB_FILE_NAME);
        Path legacyDbPath = baseConfig.resolve(DB_FILE_NAME);
        if (Files.exists(legacyDbPath, new LinkOption[0]) && !Files.exists(newDbPath, new LinkOption[0])) {
            try {
                Files.createDirectories(sbDir, new FileAttribute[0]);
                Files.move(legacyDbPath, newDbPath, new CopyOption[0]);
                LOGGER.info("Migrated legacy DB from {} to {}", (Object)legacyDbPath, (Object)newDbPath);
            }
            catch (IOException e) {
                throw new IllegalStateException("Failed to migrate legacy DB from " + String.valueOf(legacyDbPath) + " to " + String.valueOf(newDbPath), e);
            }
        }
        return new DatabaseManager(sbDir, newDbPath, cfg);
    }

    @NotNull
    private static String readClasspathResource(String resourcePath) throws IOException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        if (cl == null) {
            cl = DatabaseManager.class.getClassLoader();
        }
        try (InputStream in = cl.getResourceAsStream(resourcePath);){
            String string;
            if (in == null) {
                throw new IOException("Resource not found on classpath: " + resourcePath);
            }
            try (BufferedReader br = new BufferedReader(new InputStreamReader(in, StandardCharsets.UTF_8));){
                String line;
                StringBuilder sb = new StringBuilder();
                while ((line = br.readLine()) != null) {
                    sb.append(line).append('\n');
                }
                string = sb.toString();
            }
            return string;
        }
    }

    private static String stripSqlComments(String sql) {
        if (sql == null || sql.isEmpty()) {
            return "";
        }
        String normalized = sql.replace("\r\n", "\n").replace('\r', '\n');
        normalized = normalized.replaceAll("(?s)/\\*.*?\\*/", "");
        normalized = normalized.replaceAll("(?m)^\\s*--.*$", "");
        normalized = normalized.replaceAll("(?m)\\s+--.*$", "");
        return normalized;
    }

    private static List<String> splitSqlStatements(String sql) {
        String cleaned = DatabaseManager.stripSqlComments(sql);
        String[] parts = cleaned.split(";");
        ArrayList<String> out = new ArrayList<String>(parts.length);
        for (String part : parts) {
            String stmt = part.trim();
            if (stmt.isEmpty()) continue;
            out.add(stmt);
        }
        return out;
    }

    public synchronized void init() {
        if (this.initialized) {
            throw new IllegalStateException("DatabaseManager has already been initialized.");
        }
        try {
            Files.createDirectories(this.configDir, new FileAttribute[0]);
            Path parent = this.dbPathAbs.getParent();
            if (parent != null) {
                Files.createDirectories(parent, new FileAttribute[0]);
            }
            String jdbcUrl = this.jdbcUrl();
            LOGGER.info("Initializing SimplyBetter DB at: {}", (Object)this.dbPathAbs);
            Class.forName("org.sqlite.JDBC");
            try (Connection conn = DriverManager.getConnection(jdbcUrl);){
                if (!this.schemaExists(conn)) {
                    LOGGER.info("Schema not found. Applying schema from classpath: {}", (Object)SCHEMA_RESOURCE);
                    this.applySchema(conn);
                } else {
                    LOGGER.info("Schema already present. Skipping schema application.");
                }
            }
            String effectiveJournal = this.applyPragmasFromConfig();
            this.startExecutorFromConfig();
            this.initialized = true;
            int logical = Runtime.getRuntime().availableProcessors();
            LOGGER.info("Database initialized: path={}, journalMode={}, synchronous={}, foreignKeys={}", new Object[]{this.dbPathAbs, effectiveJournal, this.configManager.synchronous(), this.configManager.foreignKeys()});
            this.configManager.logSummary(logical);
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to initialize SimplyBetter database at " + String.valueOf(this.dbPathAbs), e);
        }
    }

    public Connection getConnection() throws SQLException {
        if (!this.initialized) {
            throw new SQLException("Database not initialized. Call init() first.");
        }
        Connection conn = DriverManager.getConnection(this.jdbcUrl());
        try (Statement s = conn.createStatement();){
            s.execute("PRAGMA foreign_keys = ON;");
            s.execute("PRAGMA busy_timeout = 5000;");
        }
        return conn;
    }

    public ExecutorService executor() {
        ExecutorService ex = this.executor;
        if (!this.initialized || ex == null || ex.isShutdown()) {
            throw new IllegalStateException("DatabaseManager not initialized or executor is shut down.");
        }
        return ex;
    }

    public void shutdown(Duration timeout) {
        ExecutorService ex = this.executor;
        if (ex == null) {
            return;
        }
        ex.shutdown();
        try {
            if (!ex.awaitTermination(timeout.toMillis(), TimeUnit.MILLISECONDS)) {
                ex.shutdownNow();
                ex.awaitTermination(timeout.toMillis(), TimeUnit.MILLISECONDS);
            }
        }
        catch (InterruptedException ie) {
            ex.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    public synchronized void reset(Duration shutdownTimeout) {
        this.shutdown(shutdownTimeout);
        try {
            Files.deleteIfExists(this.dbPathAbs);
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to delete DB file at " + String.valueOf(this.dbPathAbs), e);
        }
        this.initialized = false;
        this.init();
    }

    private String jdbcUrl() {
        return "jdbc:sqlite:" + String.valueOf(this.dbPathAbs.toAbsolutePath());
    }

    private boolean schemaExists(Connection conn) throws SQLException {
        try (PreparedStatement ps = conn.prepareStatement("SELECT 1 FROM sqlite_master WHERE type='table' AND name=? LIMIT 1");){
            boolean bl;
            block12: {
                ps.setString(1, SENTRY_TABLE);
                ResultSet rs = ps.executeQuery();
                try {
                    bl = rs.next();
                    if (rs == null) break block12;
                }
                catch (Throwable throwable) {
                    if (rs != null) {
                        try {
                            rs.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                rs.close();
            }
            return bl;
        }
    }

    private void applySchema(Connection conn) throws IOException, SQLException {
        String ddl = DatabaseManager.readClasspathResource(SCHEMA_RESOURCE);
        List<String> statements = DatabaseManager.splitSqlStatements(ddl);
        conn.setAutoCommit(false);
        try (Statement st = conn.createStatement();){
            for (String sql : statements) {
                String trimmed = sql.trim();
                if (trimmed.isEmpty()) continue;
                st.execute(trimmed);
            }
            conn.commit();
        }
        catch (SQLException ex) {
            conn.rollback();
            throw ex;
        }
        finally {
            conn.setAutoCommit(true);
        }
    }

    private String applyPragmasFromConfig() {
        String requestedJournal = this.configManager.journalMode();
        String sync = this.configManager.synchronous();
        boolean fk = this.configManager.foreignKeys();
        String effectiveJournal = requestedJournal;
        try (Connection c = DriverManager.getConnection(this.jdbcUrl());
             Statement s = c.createStatement();){
            try (ResultSet rs = s.executeQuery("PRAGMA journal_mode = " + requestedJournal + ";");){
                if (rs.next()) {
                    effectiveJournal = rs.getString(1);
                }
            }
            s.execute("PRAGMA synchronous = " + sync + ";");
            s.execute("PRAGMA foreign_keys = " + (fk ? "ON" : "OFF") + ";");
            s.execute("PRAGMA busy_timeout = 5000;");
        }
        catch (SQLException e) {
            throw new IllegalStateException("Failed to apply SQLite PRAGMAs from config.", e);
        }
        if (!requestedJournal.equalsIgnoreCase(effectiveJournal)) {
            LOGGER.warn("Requested journal_mode={}, but SQLite applied={}", (Object)requestedJournal, (Object)effectiveJournal);
        }
        return effectiveJournal;
    }

    private void startExecutorFromConfig() {
        int logical = Runtime.getRuntime().availableProcessors();
        int size = this.configManager.effectiveThreadCount(logical);
        this.executor = Executors.newFixedThreadPool(size, r -> {
            Thread t = new Thread(r, "SimplyBetter-DB-" + THREAD_NUM.incrementAndGet());
            t.setDaemon(true);
            t.setUncaughtExceptionHandler((th, ex) -> LOGGER.error("Uncaught exception in DB executor thread", ex));
            return t;
        });
        LOGGER.info("Started DB executor with {} threads (logicalCPUs={})", (Object)size, (Object)logical);
    }
}

