#version 120

// Shader by LoLip_p

#include "/settings.glsl"

in vec2 TexCoords;

uniform vec3 skyColor;
uniform vec3 sunPosition;
uniform float rainStrength;
uniform int worldTime;
uniform vec3 shadowLightPosition;
uniform vec3 playerPosition;
uniform float frameTimeCounter;

uniform sampler2D colortex0;
uniform sampler2D colortex2;
uniform sampler2D depthtex0;

uniform mat4 gbufferProjectionInverse;
uniform mat4 gbufferModelViewInverse;
uniform mat4 shadowModelView;
uniform mat4 shadowProjection;

/*
const int colortex0Format = RGB8;
const int colortex2Format = RGB8;
*/

float state;

float smoothTransition(float time) {
    if (time >= 0.0 && time <= 1000.0) return time / 1000.0;
    else if (time < 12000.0) return 1.0;
    else if (time <= 13000.0) return 1.0 - (time - 12000.0) / 1000.0;
    else return 0.0;
}

#if ENABLE_SHADOW == 1
uniform sampler2D shadowtex0;

vec3 DistortPosition(vec3 p){
    p.xy *= inversesqrt(p.x*p.x + p.y*p.y + 0.01) * 0.909;
    p.z *= 0.5;
    return p;
}

float swapDayNight(float time) {
	if (time >= 12300.0f && time <= 12800.0f) {
		return (time - 12300.0f) / 500.0f; // Плавный переход от 0 до 1
	} else if (time > 12800.0f && time <= 13200.0f) {
		return 1.0f - (time - 12800.0f) / 400.0f; // Плавный переход от 1 до 0
	} else if (time >= 22700.0f && time <= 23200.0f) {
		return (time - 22700.0f) / 500.0f; // Плавный переход от 0 до 1
	} else if (time > 23200.0f && time <= 23700.0f) {
		return 1.0f - (time - 23200.0f) / 500.0f; // Плавный переход от 1 до 0
	} else {
		return 0.0f; // Вне указанных диапазонов
	}
}

float Visibility(in sampler2D ShadowMap, in vec3 SampleCoords, in vec3 WorldPosition) {
	float distance = length(WorldPosition - playerPosition);
    float bias = mix(0.0001, 0.003, min(1.0, distance / shadowDistance));
	
    return step(SampleCoords.z - bias, texture2D(ShadowMap, SampleCoords.xy).r);
}

vec3 GetShadow(float depth) {
    vec3 ClipSpace = vec3(TexCoords, depth) * 2.0f - 1.0f;
	
    vec4 ViewW = gbufferProjectionInverse * vec4(ClipSpace, 1.0f);
    vec3 View = ViewW.xyz / ViewW.w;
	
    vec4 World = gbufferModelViewInverse * vec4(View, 1.0f);
    vec4 ShadowSpace = shadowProjection * shadowModelView * World;
    ShadowSpace.xyz = DistortPosition(ShadowSpace.xyz);
    vec3 SampleCoords = ShadowSpace.xyz * 0.5f + 0.5f;
	
	vec3 CurrentSampleCoordinate = vec3(SampleCoords.xy, SampleCoords.z);
	vec3 ShadowAccum = mix(vec3(0.0f), vec3(1.0f), Visibility(shadowtex0, CurrentSampleCoordinate, World.xyz));
	
	float distanceShadowFade = length(View) / shadowDistance;
	float shadowFade = smoothstep(0.0, 1.0, 1.0 - distanceShadowFade);

	float t = 1.0 - shadowFade;
    float t2 = t * t;
    float t4 = t2 * t2;
    float t8 = t4 * t4;
    float fastPow = t8 * t2; // ~ t^10

    ShadowAccum = clamp(mix(vec3(1.0), ShadowAccum, 1.0 - fastPow), 0.0, 1.0);
    return ShadowAccum;
}
#endif

float AdjustLightmapTorch(in float torchLight) {
    return 2 * torchLight * torchLight * torchLight;
}

float AdjustLightmapSky(in float sky){
    float sky_2 = sky * sky;
    return sky_2 * sky_2;
}

vec2 AdjustLightmap(in vec2 Lightmap){
    vec2 NewLightMap;
    NewLightMap.x = AdjustLightmapTorch(Lightmap.x);
    NewLightMap.y = AdjustLightmapSky(Lightmap.y);
    return NewLightMap;
}

vec3 GetLightmapColor(in vec2 Lightmap){
    Lightmap = AdjustLightmap(Lightmap);
    vec3 SkyColor = vec3(0.09f, 0.17f, 0.3f);
	
	#if ENABLE_SHADOW == 0
	SkyColor = vec3(1) * 0.1f;
	#endif
	
    vec3 TorchLighting = Lightmap.x * vec3(1.0f, 1.0f, 1.0f);
    vec3 SkyLighting = Lightmap.y * SkyColor * state * (1 - rainStrength);
	
	//TorchLighting -= (Lightmap.y * LIGHT_ABSORPTION * state * (1 - rainStrength));
	
	float occ = (Lightmap.y * LIGHT_ABSORPTION * state * (1 - rainStrength));
	float torchMask = smoothstep(-0.2, 0.8, Lightmap.x);
	TorchLighting -= occ * torchMask;
	
	vec3 LightmapLighting = clamp(TorchLighting * vec3(1.0f, 0.85f, 0.7f), 0.0f, 1.0f) + SkyLighting;
	
    return LightmapLighting;
}

#if FOG == 1
uniform int isEyeInWater;
uniform float near, far;
uniform float blindness;
uniform float darknessFactor;
uniform vec3 fogColor;

vec3 projectAndDivide(mat4 projectionMatrix, vec3 position){
	vec4 homPos = projectionMatrix * vec4(position, 1.0);
	return homPos.xyz / homPos.w;
}
#endif

void main(){
	vec3 Albedo = texture2D(colortex0, TexCoords).rgb;
	
    float Depth = texture2D(depthtex0, TexCoords).r;
    if(Depth == 1.0) { gl_FragColor.rgb = Albedo; return; }

	Albedo *= Albedo * sqrt(Albedo);
	
	vec3 lightBrightnessV = vec3(0.1);
    vec3 ShadowColor = vec3(1);
	vec3 NdotL = vec3(1);
	
	vec3 colortex2 = texture2D(colortex2, TexCoords).rgb;
	vec2 Lightmap = colortex2.rg;
	float adjLM = AdjustLightmapSky(Lightmap.y);
	state = smoothTransition(worldTime);
	vec3 LightmapColor = GetLightmapColor(Lightmap);
	
	#if SHADING == 1
	float lightBrightness = colortex2.z;
	lightBrightness *= state;
	lightBrightness *= 1.0f - rainStrength;
	lightBrightness += (rainStrength - 0.5f) * (1.0f - state) * rainStrength; //Rain
	lightBrightnessV = vec3(lightBrightness) * state;
	lightBrightnessV += vec3(0.0, 0.0, 0.5) * (1.0f - state); //Night
	#endif
	
	lightBrightnessV *= adjLM;
	lightBrightnessV += mix(0.2f, 0.35f, state);
	
    #if ENABLE_SHADOW == 1
	if (float(Depth < 0.56) < 0.5) {
		vec3 originalShadow = GetShadow(Depth);
		float shadow_transperency_coef = mix(0.25, SHADOW_TRANSPARENCY, state);
		ShadowColor = mix(originalShadow * vec3(shadow_transperency_coef / 0.75), originalShadow, state);
		ShadowColor = mix(ShadowColor, vec3(1), swapDayNight(worldTime));
		vec3 colorRainDayOrNight = mix(vec3(0.2), vec3(1), state);
		ShadowColor = mix(ShadowColor, colorRainDayOrNight, rainStrength);
	
		ShadowColor *= adjLM;
		ShadowColor += shadow_transperency_coef;
	}
    #endif
	
    vec3 Diffuse = Albedo * (LightmapColor + lightBrightnessV * ShadowColor);
	Diffuse = clamp(Diffuse, 0, 1);
	
	#if FOG == 1
	vec3 NDCPos = vec3(TexCoords.xy, Depth) * 2.0 - 1.0;
	vec3 viewPos = projectAndDivide(gbufferProjectionInverse, NDCPos);
	float farW = far;
	float distW = FOG_DISTANCE;
	if(isEyeInWater == 1) {
		farW = 64;
		distW = 5;
	}
	float distance = length(viewPos) / (farW * mix(mix(1, near * 0.5, blindness), near, darknessFactor));
	float fogFactor = exp(-mix(distW, 1, blindness) * (1.0 - distance));
	Diffuse = mix(Diffuse, fogColor * fogColor * sqrt(fogColor), clamp(fogFactor, 0.0, 1.0));
	#endif
	
	Diffuse = pow(Diffuse, vec3(0.4));
    float average = (Diffuse.r + Diffuse.g + Diffuse.b) / 3.0;
    Diffuse.rgb = mix(vec3(average), Diffuse.rgb, SATURATION);
    Diffuse.rgb += BRIGHTNESS;
    Diffuse.rgb = ((Diffuse.rgb - 0.5) * CONTRAST) + 0.5;
	gl_FragColor.rgb = Diffuse;
}
